package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.bannerstorage.client.Utils;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.banner.aggrstatus.StatusAggregationBanner;
import ru.yandex.direct.core.entity.banner.model.BannerButtonStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerDisplayHrefStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerFlags;
import ru.yandex.direct.core.entity.banner.model.BannerLogoStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerMulticardSetStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusSitelinksModerate;
import ru.yandex.direct.core.entity.banner.model.BannerTurboLandingStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerVcardStatusModerate;
import ru.yandex.direct.core.entity.banner.model.NewStatusImageModerate;
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate;
import ru.yandex.direct.core.entity.banner.type.system.BannerWithSystemFieldsMappings;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.OrganizationsStatusPublish;
import ru.yandex.direct.dbschema.ppc.enums.PerfCreativesCreativeType;
import ru.yandex.direct.dbschema.ppc.tables.Images;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerIdFilter;
import static ru.yandex.direct.dbschema.ppc.tables.AdditionsItemCallouts.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerButtons.BANNER_BUTTONS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerDisplayHrefs.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImages.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerLogos.BANNER_LOGOS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerMulticardSets.BANNER_MULTICARD_SETS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerPermalinks.BANNER_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerTurbolandings.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.BannersAdditions.BANNERS_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.tables.BannersInternal.BANNERS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.tables.BannersPerformance.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Organizations.ORGANIZATIONS;
import static ru.yandex.direct.dbschema.ppc.tables.PerfCreatives.PERF_CREATIVES;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromFields;

@Repository
@ParametersAreNonnullByDefault
public class AggregatedStatusesBannerRepository {
    private static final Field<Boolean> BANNER_HAS_VCARD =
            DSL.when(BANNERS.VCARD_ID.isNotNull(), true).otherwise(false).as("hasVCard");

    private static final Field<Boolean> BANNER_HAS_SITELINKS =
            DSL.when(BANNERS.SITELINKS_SET_ID.isNotNull(), true).otherwise(false).as("hasSitelinks");

    private static final Field<Boolean> BANNER_HAS_TURBOLANDING =
            DSL.when(BANNER_TURBOLANDINGS.IS_DISABLED.isFalse(), true).otherwise(false).as("hasTurbolanding");

    private static final Field<Boolean> BANNER_HAS_HREF =
            DSL.when(BANNERS.HREF.length().gt(0), true).otherwise(false).as("hasHref");

    private static final Field<Boolean> BANNER_HAS_PUBLISHED_ORGANIZATION =
            DSL.when(BANNER_PERMALINKS.PERMALINK.isNotNull()
                    .and(BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK.eq(0L))
                    .and(ORGANIZATIONS.STATUS_PUBLISH.eq(OrganizationsStatusPublish.published)
                            .or(BANNER_PERMALINKS.CHAIN_ID.isNotNull())), true)
                    .otherwise(false).as("hasPublishedOrganization");

    private static final Field<BannersPerformanceStatusmoderate> VIDEO_ADDITIONS_STATUS_MODERATE =
            DSL.when(BANNERS.BANNER_TYPE.eq(BannersBannerType.text), BANNERS_PERFORMANCE.STATUS_MODERATE)
                    .as("videoAdditionsStatusModerate");


    /**
     * Специальный статус для креативов, которые целиком представляют собой объявление(cpc_video, image_ad, cpm_banner)
     * Статус на связку, у самого креатива при этом тоже есть собственный
     * статус модерации CREATIVE_STATUS_MODERATE (из perf_creatives.statusModerate)
     */
    private static final Field<BannersPerformanceStatusmoderate> BANNER_BASED_ON_CREATIVE_STATUS_MODERATE =
            DSL.when(BANNERS.BANNER_TYPE.ne(BannersBannerType.text), BANNERS_PERFORMANCE.STATUS_MODERATE)
                    .as("bannerBasedOnCreativeStatusModerate");

    private static final Field<Boolean> ADDITIONS_CALLOUTS_DECLINED = DSL.when(
            DSL.count(DSL.when(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.isNotNull()
                    .and(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.eq(AdditionsItemCalloutsStatusmoderate.No)), 1))
                    .gt(0), true)
            .otherwise(false)
            .as("additionsCalloutsDeclined");

    private static final List<AdditionsItemCalloutsStatusmoderate> calloutsModerationStatuses = List.of(
            AdditionsItemCalloutsStatusmoderate.Ready,
            AdditionsItemCalloutsStatusmoderate.Sending,
            AdditionsItemCalloutsStatusmoderate.Sent);

    private static final Field<Boolean> ADDITIONS_CALLOUTS_ON_MODERATION = DSL.when(
            DSL.count(DSL.when(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.isNotNull()
                    .and(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.in(calloutsModerationStatuses)), 1))
                    .gt(0), true)
            .otherwise(false)
            .as("additionsCalloutsOnModeration");


    private final DslContextProvider ppcDslContextProvider;
    private final Collection<Field<?>> fields;
    private final JooqReaderWithSupplier<StatusAggregationBanner> reader;

    @Autowired
    public AggregatedStatusesBannerRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;

        reader = JooqReaderWithSupplierBuilder.builder(StatusAggregationBanner::new)
                .readProperty(StatusAggregationBanner.ID,
                        fromField(BANNERS.BID))
                .readProperty(StatusAggregationBanner.ADGROUP_ID,
                        fromField(BANNERS.PID))
                .readProperty(StatusAggregationBanner.BS_BANNER_ID,
                        fromField(BANNERS.BANNER_ID))
                .readProperty(StatusAggregationBanner.BANNER_TYPE,
                        fromField(BANNERS.BANNER_TYPE))
                .readProperty(StatusAggregationBanner.STATUS_MODERATE,
                        fromField(BANNERS.STATUS_MODERATE).by(BannerStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.STATUS_POST_MODERATE,
                        fromField(BANNERS.STATUS_POST_MODERATE).by(BannerStatusPostModerate::fromSource))
                .readProperty(StatusAggregationBanner.STATUS_BS_SYNCED,
                        fromField(BANNERS.STATUS_BS_SYNCED).by(BannerWithSystemFieldsMappings::statusBsSyncedFromDb))
                .readProperty(StatusAggregationBanner.STATUS_ARCHIVED,
                        fromField(BANNERS.STATUS_ARCH).by(BannerWithSystemFieldsMappings::statusArchivedFromDb))
                .readProperty(StatusAggregationBanner.STATUS_SHOW,
                        fromField(BANNERS.STATUS_SHOW).by(BannerWithSystemFieldsMappings::statusShowFromDb))
                .readProperty(StatusAggregationBanner.STATUS_ACTIVE,
                        fromField(BANNERS.STATUS_ACTIVE).by(BannerWithSystemFieldsMappings::statusActiveFromDb))
                .readProperty(StatusAggregationBanner.PHONE_FLAG,
                        fromField(BANNERS.PHONEFLAG).by(BannerVcardStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.FLAGS, fromField(BANNERS.FLAGS).by(BannerFlags::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_SITELINKS_STATUS_MODERATE,
                        fromField(BANNERS.STATUS_SITELINKS_MODERATE).by(BannerStatusSitelinksModerate::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_DISPLAY_HREF_STATUS_MODERATE,
                        fromField(BANNER_DISPLAY_HREFS.STATUS_MODERATE)
                                .by(BannerDisplayHrefStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_IMAGE_STATUS_MODERATE,
                        fromField(BANNER_IMAGES.STATUS_MODERATE).by(StatusBannerImageModerate::fromSource))
                .readProperty(StatusAggregationBanner.CREATIVE_STATUS_MODERATE,
                        fromField(PERF_CREATIVES.STATUS_MODERATE).by(StatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.CREATIVE_STATUS_OBSOLETE,
                        fromFields(PERF_CREATIVES.CREATIVE_TYPE, PERF_CREATIVES.LAYOUT_ID)
                                .by((type, layoutId) -> type != PerfCreativesCreativeType.performance ? null :
                                        Utils.isPerformanceLayoutObsolete(layoutId)))
                .readProperty(StatusAggregationBanner.BANNER_IMAGE_AD_STATUS_MODERATE,
                        fromField(Images.IMAGES.STATUS_MODERATE).by(NewStatusImageModerate::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_TURBOLANDING_STATUS_MODERATE,
                        fromField(BANNER_TURBOLANDINGS.STATUS_MODERATE)
                                .by(BannerTurboLandingStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_BASED_ON_CREATIVE_STATUS_MODERATE,
                        fromField(BANNER_BASED_ON_CREATIVE_STATUS_MODERATE)
                                .by(BannerCreativeStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.VIDEO_ADDITIONS_STATUS_MODERATE,
                        fromField(VIDEO_ADDITIONS_STATUS_MODERATE).by(BannerCreativeStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_HAS_HREF,
                        fromField(BANNER_HAS_HREF))
                .readProperty(StatusAggregationBanner.ADDITIONS_CALLOUTS_DECLINED,
                        fromField(ADDITIONS_CALLOUTS_DECLINED))
                .readProperty(StatusAggregationBanner.ADDITIONS_CALLOUTS_ON_MODERATION,
                        fromField(ADDITIONS_CALLOUTS_ON_MODERATION))
                .readProperty(StatusAggregationBanner.BANNER_LOGO_STATUS_MODERATE,
                        fromField(BANNER_LOGOS.STATUS_MODERATE).by(BannerLogoStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_BUTTON_STATUS_MODERATE,
                        fromField(BANNER_BUTTONS.STATUS_MODERATE).by(BannerButtonStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.BANNER_MULTICARD_SET_STATUS_MODERATE,
                        fromField(BANNER_MULTICARD_SETS.STATUS_MODERATE).by(BannerMulticardSetStatusModerate::fromSource))
                .readProperty(StatusAggregationBanner.INTERNAL_BANNER_STOPPED_BY_URL_MONITORING,
                        fromField(BANNERS_INTERNAL.IS_STOPPED_BY_URL_MONITORING).by(RepositoryUtils::booleanFromLong))
                .readProperty(StatusAggregationBanner.CAMPAIGN_TYPE, fromField(CAMPAIGNS.TYPE))
                .readProperty(StatusAggregationBanner.HAS_SITELINKS, fromField(BANNER_HAS_SITELINKS))
                .readProperty(StatusAggregationBanner.HAS_V_CARD, fromField(BANNER_HAS_VCARD))
                .readProperty(StatusAggregationBanner.HAS_TURBOLANDING, fromField(BANNER_HAS_TURBOLANDING))
                .readProperty(StatusAggregationBanner.HAS_PUBLISHED_ORGANIZATION, fromField(BANNER_HAS_PUBLISHED_ORGANIZATION))
                .readProperty(StatusAggregationBanner.TEMPLATE_ID, fromField(BANNERS_INTERNAL.TEMPLATE_ID))
                .build();

        fields = this.reader.getFieldsToRead();
    }

    public List<StatusAggregationBanner> getBanners(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyList();
        }
        SelectJoinStep<Record> step = joinAllBannersRelatedTables(
                ppcDslContextProvider.ppc(shard).select(fields).from(BANNERS));
        bannerIdFilter(bannerIds).apply(step.getQuery());

        return step
                .groupBy(
                        BANNERS.BID,
                        BANNERS_PERFORMANCE.BANNER_CREATIVE_ID,
                        BANNER_PERMALINKS.PERMALINK,
                        BANNER_PERMALINKS.CHAIN_ID,
                        ORGANIZATIONS.PERMALINK_ID,
                        ORGANIZATIONS.CLIENT_ID
                ).fetch(m -> reader.fromDb(m, new StatusAggregationBanner()));
    }

    private SelectJoinStep<Record> joinAllBannersRelatedTables(SelectJoinStep<Record> step) {
        return step
                .leftJoin(BANNERS_PERFORMANCE).on(BANNERS.BID.eq(BANNERS_PERFORMANCE.BID)
                        .and(BANNERS.PID.eq(BANNERS_PERFORMANCE.PID))
                        .and(BANNERS.CID.eq(BANNERS_PERFORMANCE.CID)))
                .leftJoin(PERF_CREATIVES).on(BANNERS_PERFORMANCE.CREATIVE_ID.eq(PERF_CREATIVES.CREATIVE_ID))
                .leftJoin(BANNER_TURBOLANDINGS).on(BANNERS.BID.eq(BANNER_TURBOLANDINGS.BID))
                .leftJoin(BANNER_DISPLAY_HREFS).on(BANNERS.BID.eq(BANNER_DISPLAY_HREFS.BID))
                .leftJoin(BANNER_IMAGES).on(BANNERS.BID.eq(BANNER_IMAGES.BID)
                        .and(BANNER_IMAGES.STATUS_SHOW.eq(BannerImagesStatusshow.Yes)))
                .leftJoin(Images.IMAGES).on(BANNERS.BID.eq(Images.IMAGES.BID)
                        .and(BANNERS.PID.eq(Images.IMAGES.PID))
                        .and(BANNERS.CID.eq(Images.IMAGES.CID)))
                .leftJoin(BANNERS_ADDITIONS).on(BANNERS.BID.eq(BANNERS_ADDITIONS.BID))
                .leftJoin(ADDITIONS_ITEM_CALLOUTS)
                .on(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.eq(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                .leftJoin(BANNER_LOGOS).on(BANNERS.BID.eq(BANNER_LOGOS.BID))
                .leftJoin(BANNER_BUTTONS).on(BANNERS.BID.eq(BANNER_BUTTONS.BID))
                .leftJoin(BANNER_MULTICARD_SETS).on(BANNERS.BID.eq(BANNER_MULTICARD_SETS.BID))
                .leftJoin(BANNERS_INTERNAL).on(BANNERS.BID.eq(BANNERS_INTERNAL.BID))
                .leftJoin(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .leftJoin(BANNER_PERMALINKS).on(BANNERS.BID.eq(BANNER_PERMALINKS.BID))
                .leftJoin(ORGANIZATIONS).on(BANNER_PERMALINKS.PERMALINK.eq(ORGANIZATIONS.PERMALINK_ID)
                        .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(BannerPermalinksPermalinkAssignType.manual)));
    }

    public List<Long> getBannerIdsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return ppcDslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.BID)
                .from(BANNERS)
                .where(BANNERS.PID.in(adGroupIds))
                .fetch(BANNERS.BID).stream().collect(Collectors.toUnmodifiableList());
    }

    public List<Long> getAdGroupIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        return ppcDslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.PID)
                .from(BANNERS)
                .where(BANNERS.BID.in(bannerIds))
                .fetch(BANNERS.PID);
    }
}
