package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaignMoney;
import ru.yandex.direct.core.entity.campaign.aggrstatus.CampaignDelayedOperation;
import ru.yandex.direct.core.entity.campaign.aggrstatus.CampaignSource;
import ru.yandex.direct.core.entity.campaign.aggrstatus.CampaignStatusBsSynced;
import ru.yandex.direct.core.entity.campaign.aggrstatus.DayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter;
import ru.yandex.direct.core.entity.campaign.converter.PriceFlightConverter;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusPostmoderate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusempty;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromLongFieldToInteger;
import static ru.yandex.direct.common.jooqmapperex.read.ReaderBuildersEx.fromYesNoEnumFieldToBoolean;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.UNDER_WALLET;
import static ru.yandex.direct.dbschema.ppc.tables.AggrStatusesCampaigns.AGGR_STATUSES_CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampOperationsQueue.CAMP_OPERATIONS_QUEUE;
import static ru.yandex.direct.dbschema.ppc.tables.CampOptions.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsCpmPrice.CAMPAIGNS_CPM_PRICE;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignsInternal.CAMPAIGNS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.tables.Subcampaigns.SUBCAMPAIGNS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@Repository
@ParametersAreNonnullByDefault
public class AggregatedStatusesCampaignRepository {
    private static final Set<CampaignsType> CAMPAIGN_TYPES = StreamEx.of(UNDER_WALLET)
            .append(CampaignType.INTERNAL_AUTOBUDGET)
            .append(CampaignType.INTERNAL_DISTRIB)
            .append(CampaignType.INTERNAL_FREE)

            .append(CampaignType.CONTENT_PROMOTION)

            .map(CampaignType::toSource)
            .toSet();

    private final DslContextProvider contextProvider;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final JooqReaderWithSupplier<AggregatedStatusCampaign> campaignJooqReader;
    private final Collection<Field<?>> fieldsToRead;
    private final JooqReaderWithSupplier<AggregatedStatusCampaignMoney> campaignMoneyJooqReader;
    private final Collection<Field<?>> moneyFieldsToRead;
    private final JooqReaderWithSupplier<AggregatedStatusCampaign> walletCampaignJooqReader;
    private final Collection<Field<?>> walletFieldsToRead;


    @Autowired
    public AggregatedStatusesCampaignRepository(DslContextProvider contextProvider,
                                                AggregatedStatusesRepository aggregatedStatusesRepository) {
        this.contextProvider = contextProvider;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;

        walletCampaignJooqReader = JooqReaderWithSupplierBuilder.builder(AggregatedStatusCampaign::new)
                .readProperty(AggregatedStatusCampaign.ID, fromField(CAMPAIGNS.CID))
                .readProperty(AggregatedStatusCampaign.CLIENT_ID, fromField(CAMPAIGNS.CLIENT_ID))
                .readProperty(AggregatedStatusCampaign.ARCHIVED, fromYesNoEnumFieldToBoolean(CAMPAIGNS.ARCHIVED))
                .readProperty(AggregatedStatusCampaign.STATUS_MODERATE, fromField(CAMPAIGNS.STATUS_MODERATE)
                        .by(CampaignStatusModerate::fromSource))
                .readProperty(AggregatedStatusCampaign.STATUS_POST_MODERATE,
                        fromField(CAMP_OPTIONS.STATUS_POST_MODERATE)
                                .by(CampaignStatusPostmoderate::fromSource))
                .readProperty(AggregatedStatusCampaign.CURRENCY_CODE, fromField(CAMPAIGNS.CURRENCY)
                        .by(c -> CurrencyCode.valueOf(c.name().toUpperCase())))
                .readProperty(AggregatedStatusCampaign.SUM, fromField(CAMPAIGNS.SUM))
                .readProperty(AggregatedStatusCampaign.SUM_TO_PAY, fromField(CAMPAIGNS.SUM_TO_PAY))
                .readProperty(AggregatedStatusCampaign.DAY_BUDGET, fromField(CAMPAIGNS.DAY_BUDGET))
                .readProperty(AggregatedStatusCampaign.DAY_BUDGET_STOP_TIME,
                        fromField(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME)).build();
        walletFieldsToRead = walletCampaignJooqReader.getFieldsToRead();

        campaignJooqReader = JooqReaderWithSupplierBuilder.builder(AggregatedStatusCampaign::new)
                .readProperty(AggregatedStatusCampaign.ID, fromField(CAMPAIGNS.CID))
                .readProperty(AggregatedStatusCampaign.ORDER_ID, fromField(CAMPAIGNS.ORDER_ID))
                .readProperty(AggregatedStatusCampaign.USER_ID, fromField(CAMPAIGNS.UID))
                .readProperty(AggregatedStatusCampaign.AGENCY_UID, fromField(CAMPAIGNS.AGENCY_UID))
                .readProperty(AggregatedStatusCampaign.MANAGER_UID, fromField(CAMPAIGNS.MANAGER_UID))
                .readProperty(AggregatedStatusCampaign.WALLET_ID, fromField(CAMPAIGNS.WALLET_CID))
                .readProperty(AggregatedStatusCampaign.CLIENT_ID, fromField(CAMPAIGNS.CLIENT_ID))
                .readProperty(AggregatedStatusCampaign.MASTER_CID, fromField(SUBCAMPAIGNS.MASTER_CID))
                .readProperty(AggregatedStatusCampaign.ARCHIVED, fromYesNoEnumFieldToBoolean(CAMPAIGNS.ARCHIVED))
                .readProperty(AggregatedStatusCampaign.EMPTY, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_EMPTY))
                .readProperty(AggregatedStatusCampaign.TYPE, fromField(CAMPAIGNS.TYPE).by(CampaignType::fromSource))

                .readProperty(AggregatedStatusCampaign.HAS_ENABLE_CPC_HOLD,
                        fromField(CAMPAIGNS.OPTS).by(CampaignConverter::hasEnableCpcHoldFromDb))

                .readProperty(AggregatedStatusCampaign.TIME_TARGET, fromField(CAMPAIGNS.TIME_TARGET)
                        .by(CampaignMappings::timeTargetFromDb))
                .readProperty(AggregatedStatusCampaign.TIMEZONE_ID, fromField(CAMPAIGNS.TIMEZONE_ID))

                .readProperty(AggregatedStatusCampaign.START_DATE, fromField(CAMPAIGNS.START_TIME))
                .readProperty(AggregatedStatusCampaign.FINISH_DATE, fromField(CAMPAIGNS.FINISH_TIME))

                .readProperty(AggregatedStatusCampaign.SUM, fromField(CAMPAIGNS.SUM))
                .readProperty(AggregatedStatusCampaign.SUM_SPENT, fromField(CAMPAIGNS.SUM_SPENT))
                .readProperty(AggregatedStatusCampaign.SUM_LAST, fromField(CAMPAIGNS.SUM_LAST))
                .readProperty(AggregatedStatusCampaign.SUM_TO_PAY, fromField(CAMPAIGNS.SUM_TO_PAY))

                .readProperty(AggregatedStatusCampaign.RESTRICTION_VALUE,
                        fromField(CAMPAIGNS_INTERNAL.RESTRICTION_VALUE))
                .readProperty(AggregatedStatusCampaign.SUM_SPENT_UNITS, fromField(CAMPAIGNS.SUM_SPENT_UNITS))

                .readProperty(AggregatedStatusCampaign.CURRENCY_CODE, fromField(CAMPAIGNS.CURRENCY)
                        .by(c -> CurrencyCode.valueOf(c.name().toUpperCase())))
                .readProperty(AggregatedStatusCampaign.CURRENCY_CONVERTED,
                        fromYesNoEnumFieldToBoolean(CAMPAIGNS.CURRENCY_CONVERTED))
                .readProperty(AggregatedStatusCampaign.SHOWS, fromField(CAMPAIGNS.SHOWS))
                .readProperty(AggregatedStatusCampaign.NO_PAY, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_NO_PAY))
                .readProperty(AggregatedStatusCampaign.SHOWING, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_SHOW))
                .readProperty(AggregatedStatusCampaign.ACTIVE, fromYesNoEnumFieldToBoolean(CAMPAIGNS.STATUS_ACTIVE))
                .readProperty(AggregatedStatusCampaign.STATUS_BS_SYNCED, fromField(CAMPAIGNS.STATUS_BS_SYNCED)
                        .by(CampaignStatusBsSynced::fromSource))
                .readProperty(AggregatedStatusCampaign.STATUS_MODERATE, fromField(CAMPAIGNS.STATUS_MODERATE)
                        .by(CampaignStatusModerate::fromSource))
                .readProperty(AggregatedStatusCampaign.STATUS_POST_MODERATE,
                        fromField(CAMP_OPTIONS.STATUS_POST_MODERATE)
                                .by(CampaignStatusPostmoderate::fromSource))
                .readProperty(AggregatedStatusCampaign.DAY_BUDGET, fromField(CAMPAIGNS.DAY_BUDGET))
                .readProperty(AggregatedStatusCampaign.DAY_BUDGET_STOP_TIME,
                        fromField(CAMP_OPTIONS.DAY_BUDGET_STOP_TIME))
                .readProperty(AggregatedStatusCampaign.DAY_BUDGET_DAILY_CHANGE_COUNT,
                        fromField(CAMP_OPTIONS.DAY_BUDGET_DAILY_CHANGE_COUNT).by(Long::intValue))
                .readProperty(AggregatedStatusCampaign.DAY_BUDGET_SHOW_MODE, fromField(CAMPAIGNS.DAY_BUDGET_SHOW_MODE)
                        .by(DayBudgetShowMode::fromSource))
                .readProperty(AggregatedStatusCampaign.PLATFORM,
                        fromField(CAMPAIGNS.PLATFORM).by(CampaignsPlatform::fromSource))
                .readProperty(AggregatedStatusCampaign.CONTEXT_LIMIT, fromLongFieldToInteger(CAMPAIGNS.CONTEXT_LIMIT))

                .readProperty(AggregatedStatusCampaign.DELAYED_OPERATION, fromField(CAMP_OPERATIONS_QUEUE.OPERATION)
                        .by(CampaignDelayedOperation::fromSource))
                .readProperty(AggregatedStatusCampaign.HAS_SITE_MONITORING,
                        fromYesNoEnumFieldToBoolean(CAMP_OPTIONS.STATUS_METRICA_CONTROL))
                .readProperty(AggregatedStatusCampaign.STOP_TIME, fromField(CAMP_OPTIONS.STOP_TIME))

                .readProperty(AggregatedStatusCampaign.FLIGHT_STATUS_CORRECT,
                        fromField(CAMPAIGNS_CPM_PRICE.STATUS_CORRECT)
                                .by(PriceFlightConverter::statusCorrectFromDbFormat))
                .readProperty(AggregatedStatusCampaign.FLIGHT_STATUS_APPROVE,
                        fromField(CAMPAIGNS_CPM_PRICE.STATUS_APPROVE)
                                .by(PriceFlightConverter::statusApproveFromDbFormat))
                .readProperty(AggregatedStatusCampaign.IS_DRAFT_APPROVE_ALLOWED,
                        fromField(CAMPAIGNS_CPM_PRICE.IS_DRAFT_APPROVE_ALLOWED)
                                .by(RepositoryUtils::booleanFromLong))
                .readProperty(AggregatedStatusCampaign.STRATEGY,
                        fromField(CAMPAIGNS.STRATEGY_DATA).by(this::getStrategyFromSource))
                .readProperty(AggregatedStatusCampaign.MEANINGFUL_GOALS,
                        fromField(CAMP_OPTIONS.MEANINGFUL_GOALS).by(CampaignConverter::meaningfulGoalsFromDb))
                .readProperty(AggregatedStatusCampaign.SOURCE, fromField(CAMPAIGNS.SOURCE)
                        .by(CampaignSource::fromSource))
                .build();

        Set<Field<?>> fieldsToRead = new HashSet<>(campaignJooqReader.getFieldsToRead());
        fieldsToRead.add(AGGR_STATUSES_CAMPAIGNS.AGGR_DATA);
        fieldsToRead.add(AGGR_STATUSES_CAMPAIGNS.IS_OBSOLETE);
        this.fieldsToRead = fieldsToRead;

        campaignMoneyJooqReader = JooqReaderWithSupplierBuilder.builder(AggregatedStatusCampaignMoney::new)
                .readProperty(AggregatedStatusCampaignMoney.ID, fromField(CAMPAIGNS.CID))
                .readProperty(AggregatedStatusCampaignMoney.SUM, fromField(CAMPAIGNS.SUM))
                .readProperty(AggregatedStatusCampaignMoney.SUM_SPENT, fromField(CAMPAIGNS.SUM_SPENT))
                .readProperty(AggregatedStatusCampaignMoney.CURRENCY_CODE, fromField(CAMPAIGNS.CURRENCY)
                        .by(c -> CurrencyCode.valueOf(c.name().toUpperCase())))
                .build();

        this.moneyFieldsToRead = StreamEx.of(campaignMoneyJooqReader.getFieldsToRead())
                .append(CAMPAIGNS.WALLET_CID)
                .distinct().toImmutableList();
    }

    public List<AggregatedStatusCampaign> getCampaigns(int shard, Collection<Long> campaignIds) {
        return List.copyOf(getCampaignById(shard, campaignIds).values());
    }

    public Map<Long, AggregatedStatusCampaign> getCampaignById(int shard, Collection<Long> campaignIds) {
        if (CollectionUtils.isEmpty(campaignIds)) {
            return Map.of();
        }
        SelectJoinStep<Record> campaignsSelect = campaignsSelectStep(shard);
        return campaignsSelect
                .where(CAMPAIGNS.TYPE.in(CAMPAIGN_TYPES)
                        .and(CAMPAIGNS.CID.in(campaignIds))
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .fetchMap(CAMPAIGNS.CID, this::fromDb);
    }

    public Map<Long, List<Long>> getSubCampaignIdsByMasterId(int shard, Collection<Long> campaignIds) {
        return contextProvider.ppc(shard)
                .select(SUBCAMPAIGNS.MASTER_CID, SUBCAMPAIGNS.CID)
                .from(SUBCAMPAIGNS)
                .where(SUBCAMPAIGNS.MASTER_CID.in(campaignIds))
                .fetchGroups(SUBCAMPAIGNS.MASTER_CID, SUBCAMPAIGNS.CID);
    }

    public Map<Long, List<AggregatedStatusCampaignMoney>> getCampaignsMoneyByWalletIds(int shard,
                                                                                       Collection<Long> walletIds) {
        return contextProvider.ppc(shard)
                .select(moneyFieldsToRead)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.TYPE.in(CAMPAIGN_TYPES)
                        .and(CAMPAIGNS.WALLET_CID.in(walletIds))
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .fetchGroups(CAMPAIGNS.WALLET_CID, campaignMoneyJooqReader::fromDb);
    }

    public Set<Long> getCampaignIdsByWalletIds(int shard, Collection<Long> walletIds) {
        return contextProvider.ppc(shard)
                .select(CAMPAIGNS.CID)
                .from(CAMPAIGNS)
                .where(CAMPAIGNS.TYPE.in(CAMPAIGN_TYPES)
                        .and(CAMPAIGNS.WALLET_CID.in(walletIds))
                        .and(CAMPAIGNS.STATUS_EMPTY.eq(CampaignsStatusempty.No)))
                .fetchSet(CAMPAIGNS.CID);
    }

    private SelectJoinStep<Record> campaignsSelectStep(int shard) {
        return contextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(CAMPAIGNS
                        .leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(SUBCAMPAIGNS).on(SUBCAMPAIGNS.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMP_OPERATIONS_QUEUE).on(CAMP_OPERATIONS_QUEUE.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMPAIGNS_CPM_PRICE).on(CAMPAIGNS_CPM_PRICE.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(CAMPAIGNS_INTERNAL).on(CAMPAIGNS_INTERNAL.CID.eq(CAMPAIGNS.CID))
                        .leftJoin(AGGR_STATUSES_CAMPAIGNS).on(AGGR_STATUSES_CAMPAIGNS.CID.eq(CAMPAIGNS.CID))
                );
    }

    public List<AggregatedStatusCampaign> getWallets(int shard, Collection<Long> walletIds) {
        return contextProvider.ppc(shard)
                .select(walletFieldsToRead)
                .from(CAMPAIGNS.leftJoin(CAMP_OPTIONS).on(CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID)))
                .where(CAMPAIGNS.TYPE.eq(CampaignType.toSource(CampaignType.WALLET)).and(CAMPAIGNS.CID.in(walletIds)))
                .fetch(walletCampaignJooqReader::fromDb);
    }

    private AggregatedStatusCampaign fromDb(Record record) {
        AggregatedStatusCampaign campaign = campaignJooqReader.fromDb(record);
        String aggrStatusJson = record.get(AGGR_STATUSES_CAMPAIGNS.AGGR_DATA);
        Boolean isObsolete = RepositoryUtils.booleanFromLong(record.get(AGGR_STATUSES_CAMPAIGNS.IS_OBSOLETE));
        AggregatedStatusCampaignData aggrStatus = aggrStatusJson != null
                ? aggregatedStatusesRepository.fromJsonSafe(campaign.getId(), aggrStatusJson,
                AggregatedStatusCampaignData.class).withIsObsolete(isObsolete)
                : null;
        campaign.withShows(nvl(campaign.getShows(), 0L)).withAggregatedStatus(aggrStatus);
        return campaign;
    }

    private DbStrategy getStrategyFromSource(String source) {
        StrategyData strategyData = JsonUtils.fromJsonIgnoringErrors(source, StrategyData.class);
        return (DbStrategy) new DbStrategy().withStrategyData(strategyData);
    }
}
