package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.aggregatedstatuses.ChangesHolder;
import ru.yandex.direct.core.entity.keyword.aggrstatus.StatusAggregationKeyword;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.aggregatedstatuses.ChangesHolder.keywordChangesEvent;
import static ru.yandex.direct.dbschema.ppc.tables.Bids.BIDS;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;

@Repository
@ParametersAreNonnullByDefault
public class AggregatedStatusesKeywordRepository {

    private final DslContextProvider ppcDslContextProvider;
    private final Collection<Field<?>> fields;
    private final JooqReaderWithSupplier<StatusAggregationKeyword> reader;

    @Autowired
    public AggregatedStatusesKeywordRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;

        reader = JooqReaderWithSupplierBuilder.builder(StatusAggregationKeyword::new)
                .readProperty(StatusAggregationKeyword.ID, fromField(BIDS.ID))
                .readProperty(StatusAggregationKeyword.AD_GROUP_ID, fromField(BIDS.PID))
                .readProperty(StatusAggregationKeyword.CAMPAIGN_ID, fromField(BIDS.CID))
                .readProperty(StatusAggregationKeyword.STATUS_MODERATE, fromField(BIDS.STATUS_MODERATE).by(
                        ru.yandex.direct.core.entity.keyword.model.StatusModerate::fromSource))
                .readProperty(StatusAggregationKeyword.IS_SUSPENDED, fromField(BIDS.IS_SUSPENDED).by(
                        RepositoryUtils::booleanFromLong))
                .build();

        fields = this.reader.getFieldsToRead();
    }

    public List<StatusAggregationKeyword> getKeywords(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyList();
        }

        return ppcDslContextProvider.ppc(shard).select(fields).from(BIDS).where(BIDS.ID.in(ids))
                .fetch(r -> reader.fromDb(r, new StatusAggregationKeyword()));
    }

    public List<StatusAggregationKeyword> getKeywordsByAdGroupIds(int shard, Set<Long> adgroupIds) {
        if (adgroupIds.isEmpty()) {
            return emptyList();
        }
        return ppcDslContextProvider.ppc(shard).select(fields).from(BIDS).where(BIDS.PID.in(adgroupIds))
                .fetch(r -> reader.fromDb(r, new StatusAggregationKeyword()));
    }

    public List<ChangesHolder.KeywordChangesEvent> getKeywordChangeEventsByAdGroupIds(int shard, Set<Long> adgroupIds) {
        if (adgroupIds.isEmpty()) {
            return emptyList();
        }
        return ppcDslContextProvider.ppc(shard).select(fields).from(BIDS).where(BIDS.PID.in(adgroupIds))
                .fetch(this::toKeywordChangesEvent);
    }

    /**
     * Метод возвращает список id, которые есть в bids
     * Нужен, чтобы понять какие из удаленных из bids_arc записей были перенесены в bids при разархивации
     */
    public Set<Long> getKeywordsForArchivedDeletion(int shard, Collection<Long> ids) {
        return ppcDslContextProvider.ppc(shard).select(BIDS.ID).from(BIDS).where(BIDS.ID.in(ids)).fetchSet(BIDS.ID);
    }

    private ChangesHolder.KeywordChangesEvent toKeywordChangesEvent(Record record) {
        return keywordChangesEvent(record.get(BIDS.ID), record.get(BIDS.PID), record.get(BIDS.CID), false);
    }
}
