package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.google.common.annotations.VisibleForTesting;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record3;
import org.jooq.Record4;
import org.jooq.Record5;
import org.jooq.Result;
import org.jooq.SelectJoinStep;
import org.jooq.Table;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.AdWithStatus;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.KeywordWithStatus;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RejectReasonsAndCountersByIds;
import ru.yandex.direct.core.aggregatedstatuses.repository.model.RetargetingWithStatus;
import ru.yandex.direct.core.entity.aggregatedstatuses.AggregatedStatusBaseData;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupCounters;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.AggregatedStatusCampaignData;
import ru.yandex.direct.core.entity.aggregatedstatuses.campaign.CampaignCounters;
import ru.yandex.direct.core.entity.aggregatedstatuses.keyword.AggregatedStatusKeywordData;
import ru.yandex.direct.core.entity.aggregatedstatuses.retargeting.AggregatedStatusRetargetingData;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.dbschema.ppc.enums.AggrStatusesKeywordsReason;
import ru.yandex.direct.dbschema.ppc.enums.AggrStatusesKeywordsStatus;
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType;
import ru.yandex.direct.dbschema.ppc.tables.AggrStatusesAdgroups;
import ru.yandex.direct.dbschema.ppc.tables.AggrStatusesBanners;
import ru.yandex.direct.dbschema.ppc.tables.AggrStatusesCampaigns;
import ru.yandex.direct.dbschema.ppc.tables.AggrStatusesKeywords;
import ru.yandex.direct.dbschema.ppc.tables.AggrStatusesRetargetings;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbschema.ppc.tables.Bids;
import ru.yandex.direct.dbschema.ppc.tables.BidsArc;
import ru.yandex.direct.dbschema.ppc.tables.BidsBase;
import ru.yandex.direct.dbschema.ppc.tables.BidsRetargeting;
import ru.yandex.direct.dbschema.ppc.tables.Phrases;
import ru.yandex.direct.dbschema.ppc.tables.records.AggrStatusesKeywordsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.common.util.RepositoryUtils.booleanFromLong;
import static ru.yandex.direct.dbschema.ppc.tables.AggrStatusesAdgroups.AGGR_STATUSES_ADGROUPS;
import static ru.yandex.direct.dbschema.ppc.tables.AggrStatusesBanners.AGGR_STATUSES_BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.AggrStatusesCampaigns.AGGR_STATUSES_CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS;
import static ru.yandex.direct.dbschema.ppc.tables.AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS;
import static ru.yandex.direct.dbschema.ppc.tables.BidsBase.BIDS_BASE;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;

@Repository
@ParametersAreNonnullByDefault
public class AggregatedStatusesRepository {
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(AggregatedStatusesRepository.class);

    // Для всех таблиц одинаковые, поэтому используем константу
    private static final Field<String> AGGR_DATA = AGGR_STATUSES_CAMPAIGNS.AGGR_DATA;
    private static final Field<Long> IS_OBSOLETE = AGGR_STATUSES_CAMPAIGNS.IS_OBSOLETE;

    // Поля json-а которые обновляем при изменении самого объекта
    private static final String JSON_COUNTERS_SUBFIELD = AggregatedStatusBaseData.COUNTERS;
    private static final String JSON_REJECT_REASONS_SUBFIELD = AggregatedStatusBaseData.REJECT_REASONS;

    private final ObjectMapper objectMapper;

    private final DslContextProvider dslContextProvider;

    @Autowired
    public AggregatedStatusesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        objectMapper = new ObjectMapper()
                // иначе будет сложно обновлять код
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        objectMapper.registerModule(new Jdk8Module());
    }

    /*
    m dt:ppc:1 "
    INSERT INTO aggr_statuses_keywords
        (id, aggr_data, updated)
        VALUES(11, '{\"counters\":{\"archived\":\"11\"}}', NOW())
        ON DUPLICATE KEY UPDATE
            WHEN updated<updateBefore THEN VALUES(aggr_data) ELSE aggr_data,
            WHEN updated<updateBefore THEN NOW() ELSE updated"
     */
    private <T extends Record> void updateTableStatusJson(int shard, @Nullable LocalDateTime updateBefore,
                                                          Table<T> table, Field<Long> idField,
                                                          Map<Long, String> statusJsons) {
        if (statusJsons.isEmpty()) {
            return;
        }
        Field<String> aggrData = getFieldForTable(table, AGGR_DATA);
        Field<LocalDateTime> updated = getFieldForTable(table, AGGR_STATUSES_CAMPAIGNS.UPDATED);
        Field<Long> isObsolete = getFieldForTable(table, AGGR_STATUSES_CAMPAIGNS.IS_OBSOLETE);

        var query = dslContextProvider.ppc(shard).dsl().insertQuery(table);
        for (var entry : statusJsons.entrySet()) {
            query.newRecord();
            query.addValue(idField, entry.getKey());
            query.addValue(aggrData, entry.getValue());
        }

        query.onDuplicateKeyUpdate(true);

        // Если updateBefore == null, то ограничение по времени обновления учитывать не нужно
        // при этом если статус не обновился, то в этой ветке время не обновляем
        // Порядок ВАЖЕН! Иначе поле сравнивается само с собой в when для updated
        if (updateBefore == null) {
            query.addValueForUpdate(updated,
                    DSL.when(aggrData.ne(sqlValues(aggrData))
                            .or(isObsolete.ne(RepositoryUtils.FALSE)), DSL.currentLocalDateTime()).otherwise(updated));
            query.addValueForUpdate(isObsolete, RepositoryUtils.FALSE);
            query.addValueForUpdate(aggrData, sqlValues(aggrData));
        } else {
            // Добавление сделано через одиночные addValueForUpdate, так как важен порядок применения изменений:
            // сперва данные (aggrData), потом время (updated). Иначе сперва обновлялось время, а данные уже не
            // обновлялись, потому что новое значение времени не удовлетворяло условию
            // время обновляем, независимо от того поменялся статус или нет
            query.addValueForUpdate(aggrData,
                    DSL.when(updated.lt(updateBefore), sqlValues(aggrData)).otherwise(aggrData));
            query.addValueForUpdate(isObsolete,
                    DSL.when(updated.lt(updateBefore), RepositoryUtils.FALSE).otherwise(isObsolete));
            query.addValueForUpdate(updated,
                    DSL.when(updated.lt(updateBefore), DSL.currentLocalDateTime()).otherwise(updated));
        }

        query.execute();
    }

    public void updateCampaigns(int shard, @Nullable LocalDateTime updateBefore,
                                Map<Long, AggregatedStatusCampaignData> statusById) {
        AggrStatusesCampaigns t = AGGR_STATUSES_CAMPAIGNS;
        updateTableStatusJson(shard, updateBefore, t, t.CID, objToJsonMap(statusById));
    }

    public void updateAdGroups(int shard, @Nullable LocalDateTime updateBefore,
                               Map<Long, AggregatedStatusAdGroupData> statusById) {
        AggrStatusesAdgroups t = AggrStatusesAdgroups.AGGR_STATUSES_ADGROUPS;
        updateTableStatusJson(shard, updateBefore, t, t.PID, objToJsonMap(statusById));
    }

    public void updateAds(int shard, @Nullable LocalDateTime updateBefore,
                          Map<Long, AggregatedStatusAdData> statusById) {
        AggrStatusesBanners t = AggrStatusesBanners.AGGR_STATUSES_BANNERS;
        updateTableStatusJson(shard, updateBefore, t, t.BID, objToJsonMap(statusById));
    }

    public void updateKeywords(int shard, @Nullable LocalDateTime updateBefore,
                               Map<Long, AggregatedStatusKeywordData> statusById) {
        AggrStatusesKeywords table = AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS;
        TableField<AggrStatusesKeywordsRecord, Long> idField = table.ID;

        var statusField = getFieldForTable(table, table.STATUS);
        var reasonField = getFieldForTable(table, table.REASON);
        var isObsolete = getFieldForTable(table, table.IS_OBSOLETE);
        var statusSqlValues = sqlValues(statusField, AggrStatusesKeywordsStatus.class);
        var reasonSqlValues = sqlValues(reasonField,
                AggrStatusesKeywordsReason.class);
        Field<LocalDateTime> updated = getFieldForTable(table, AGGR_STATUSES_CAMPAIGNS.UPDATED);

        var query = dslContextProvider.ppc(shard).dsl().insertQuery(table);
        for (var entry : statusById.entrySet()) {
            AggregatedStatusKeywordData statusData = entry.getValue();
            query.newRecord();
            query.addValue(idField, entry.getKey());
            if (statusData.getStatus().isPresent()) {
                query.addValue(statusField, keywordStatusToDb(statusData.getStatusUnsafe()));
            }
            if (!statusData.getReasons().isEmpty()) {
                query.addValue(reasonField, keywordReasonToDb(statusData.getReasons().get(0)));
            } else if (statusData.getStatus().isPresent()) {
                query.addValue(reasonField, keywordReasonToDbByStatus(statusData.getStatusUnsafe()));
            }   // если и ризона нет и статуса нет, при этом происходит INSERT (по UPDATE) то упадем и это правильно,
            // т.к. такой ситуации случаться не должно
        }

        query.onDuplicateKeyUpdate(true);

        // Если updateBefore == null, то ограничение по времени обновления учитывать не нужно
        // при этом если статус не обновился, то в этой ветке время не обновляем
        // Порядок ВАЖЕН! Иначе поле сравнивается само с собой в when для updated
        if (updateBefore == null) {
            query.addValueForUpdate(updated,
                    DSL.when(statusField.ne(statusSqlValues)
                            .or(reasonField.ne(reasonSqlValues))
                            .or(isObsolete.ne(RepositoryUtils.FALSE)), DSL.currentLocalDateTime()).otherwise(updated));
            query.addValueForUpdate(isObsolete, RepositoryUtils.FALSE);
            query.addValueForUpdate(statusField, statusSqlValues);
            query.addValueForUpdate(reasonField, reasonSqlValues);
        } else {
            // Добавление сделано через одиночные addValueForUpdate, так как важен порядок применения изменений:
            // сперва данные (status, reason), потом время (updated). Иначе сперва обновлялось время, а данные уже не
            // обновлялись, потому что новое значение времени не удовлетворяло условию
            // время обновляем, независимо от того поменялся статус или нет
            query.addValueForUpdate(statusField,
                    DSL.when(updated.lt(updateBefore), statusSqlValues).otherwise(statusField));
            query.addValueForUpdate(reasonField,
                    DSL.when(updated.lt(updateBefore), reasonSqlValues).otherwise(reasonField));
            query.addValueForUpdate(isObsolete,
                    DSL.when(updated.lt(updateBefore), RepositoryUtils.FALSE).otherwise(isObsolete));

            query.addValueForUpdate(updated,
                    DSL.when(updated.lt(updateBefore), DSL.currentLocalDateTime()).otherwise(updated));
        }

        query.execute();
    }

    public void markAdStatusesAsObsolete(int shard, @Nullable LocalDateTime updateBefore,
                                         Collection<Long> bannerIds) {
        DSLContext context = dslContextProvider.ppc(shard).dsl();
        markAdStatusesAsObsolete(context, updateBefore, bannerIds);
    }

    public void markAdStatusesAsObsolete(DSLContext context, @Nullable LocalDateTime updateBefore,
                                         Collection<Long> bannerIds) {
        updateIsObsolete(context, updateBefore, AGGR_STATUSES_BANNERS, AGGR_STATUSES_BANNERS.BID, bannerIds);
    }

    public void markKeywordStatusesAsObsolete(DSLContext context, @Nullable LocalDateTime updateBefore,
                                              Collection<Long> keywordIds) {
        updateIsObsolete(context, updateBefore, AGGR_STATUSES_KEYWORDS, AGGR_STATUSES_KEYWORDS.ID, keywordIds);
    }

    public void markRetargetingStatusesAsObsolete(int shard, @Nullable LocalDateTime updateBefore,
                                                  Collection<Long> retargetingIds) {
        DSLContext context = dslContextProvider.ppc(shard).dsl();
        updateIsObsolete(context, updateBefore, AGGR_STATUSES_RETARGETINGS, AGGR_STATUSES_RETARGETINGS.RET_ID,
                retargetingIds);
    }

    public void markCampaignStatusesAsObsolete(DSLContext context, @Nullable LocalDateTime updateBefore,
                                               Collection<Long> campaignIds) {
        updateIsObsolete(context, updateBefore, AGGR_STATUSES_CAMPAIGNS, AGGR_STATUSES_CAMPAIGNS.CID, campaignIds);
    }

    private <T extends Record> void updateIsObsolete(DSLContext context, @Nullable LocalDateTime updateBefore,
                                                     Table<T> table, Field<Long> idField,
                                                     Collection<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }
        Field<Long> isObsoleteField = getFieldForTable(table, AGGR_STATUSES_CAMPAIGNS.IS_OBSOLETE);
        Field<LocalDateTime> updatedField = getFieldForTable(table, AGGR_STATUSES_CAMPAIGNS.UPDATED);

        Condition condition = idField.in(ids);
        // Если updateBefore == null, то ограничение по времени обновления учитывать не нужно
        if (updateBefore != null) {
            condition = condition.and(updatedField.lt(updateBefore));
        }
        context
                .update(table)
                .set(isObsoleteField, RepositoryUtils.TRUE)
                .set(updatedField, DSL.currentLocalDateTime())
                .where(condition)
                .execute();
    }

    public static GdSelfStatusEnum keywordStatusFromDb(AggrStatusesKeywordsStatus status) {
        switch (status) {
            case DRAFT:
                return GdSelfStatusEnum.DRAFT;
            case ARCHIVED:
                return GdSelfStatusEnum.ARCHIVED;
            case RUN_OK:
                return GdSelfStatusEnum.RUN_OK;
            case RUN_WARN:
                return GdSelfStatusEnum.RUN_WARN;
            case STOP_OK:
                return GdSelfStatusEnum.STOP_OK;
            case STOP_CRIT:
                return GdSelfStatusEnum.STOP_CRIT;
            case STOP_WARN:
                return GdSelfStatusEnum.STOP_WARN;
            case PAUSE_OK:
                return GdSelfStatusEnum.PAUSE_OK;
            case PAUSE_CRIT:
                return GdSelfStatusEnum.PAUSE_CRIT;
            case PAUSE_WARN:
                return GdSelfStatusEnum.PAUSE_WARN;
            default:
                throw new IllegalStateException("Can't map status " + status + " from db");
        }
    }

    public static AggrStatusesKeywordsStatus keywordStatusToDb(GdSelfStatusEnum status) {
        switch (status) {
            case DRAFT:
                return AggrStatusesKeywordsStatus.DRAFT;
            case ARCHIVED:
                return AggrStatusesKeywordsStatus.ARCHIVED;
            case RUN_OK:
                return AggrStatusesKeywordsStatus.RUN_OK;
            case RUN_WARN:
                return AggrStatusesKeywordsStatus.RUN_WARN;
            case STOP_OK:
                return AggrStatusesKeywordsStatus.STOP_OK;
            case STOP_CRIT:
                return AggrStatusesKeywordsStatus.STOP_CRIT;
            case STOP_WARN:
                return AggrStatusesKeywordsStatus.STOP_WARN;
            case PAUSE_OK:
                return AggrStatusesKeywordsStatus.PAUSE_OK;
            case PAUSE_CRIT:
                return AggrStatusesKeywordsStatus.PAUSE_CRIT;
            case PAUSE_WARN:
                return AggrStatusesKeywordsStatus.PAUSE_WARN;
            default:
                throw new IllegalStateException("Can't map status " + status + " to db");
        }
    }

    public static Optional<GdSelfStatusReason> keywordReasonFromDb(AggrStatusesKeywordsReason reason) {
        switch (reason) {
            case ARCHIVED:
            case DRAFT:
            case ACTIVE:
                return Optional.empty(); // перестали отдавать дефолтные причины
            case REJECTED_ON_MODERATION:
                return Optional.of(GdSelfStatusReason.REJECTED_ON_MODERATION);
            case SUSPENDED_BY_USER:
                return Optional.of(GdSelfStatusReason.KEYWORD_SUSPENDED_BY_USER);
            case RELEVANCE_MATCH_SUSPENDED_BY_USER:
                return Optional.of(GdSelfStatusReason.RELEVANCE_MATCH_SUSPENDED_BY_USER);
            default:
                throw new IllegalStateException("Can't map reason " + reason + " from db");
        }
    }

    public static AggrStatusesKeywordsReason keywordReasonToDb(GdSelfStatusReason reason) {
        switch (reason) {
            case ARCHIVED:
                return AggrStatusesKeywordsReason.ARCHIVED;
            case DRAFT:
                return AggrStatusesKeywordsReason.DRAFT;
            case REJECTED_ON_MODERATION:
                return AggrStatusesKeywordsReason.REJECTED_ON_MODERATION;
            case KEYWORD_SUSPENDED_BY_USER:
                return AggrStatusesKeywordsReason.SUSPENDED_BY_USER;
            case ACTIVE:
                return AggrStatusesKeywordsReason.ACTIVE;
            case RELEVANCE_MATCH_SUSPENDED_BY_USER:
                return AggrStatusesKeywordsReason.RELEVANCE_MATCH_SUSPENDED_BY_USER;
            default:
                throw new IllegalStateException("Can't map reason " + reason + " to db");
        }
    }

    // На случай если причины нет (т.к. отказались от причин повторяющих статус) вычислим ее перед записью в БД.
    // Идеально убрать эти причины из ключевиков и просто писать null, но сейчас уже править не целесообразно
    public static AggrStatusesKeywordsReason keywordReasonToDbByStatus(GdSelfStatusEnum status) {
        switch (status) {
            case ARCHIVED:
                return AggrStatusesKeywordsReason.ARCHIVED;
            case DRAFT:
                return AggrStatusesKeywordsReason.DRAFT;
            case RUN_OK:
                return AggrStatusesKeywordsReason.ACTIVE;
            default:
                throw new IllegalStateException("Can't map reason " + status + " to db");
        }
    }

    public void updateRetargetings(int shard, @Nullable LocalDateTime updateBefore,
                                   Map<Long, AggregatedStatusRetargetingData> statusById) {
        AggrStatusesRetargetings t = AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS;
        updateTableStatusJson(shard, updateBefore, t, t.RET_ID, objToJsonMap(statusById));
    }

    private <V> Map<Long, String> objToJsonMap(Map<Long, V> objMap) {
        return EntryStream.of(objMap).mapValues(this::toJson).toMap();
    }

    public void deleteCampaignStatuses(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard).dsl()
                .delete(AGGR_STATUSES_CAMPAIGNS)
                .where(AGGR_STATUSES_CAMPAIGNS.CID.in(ids))
                .execute();
    }

    public void deleteAdGroupStatuses(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard).dsl()
                .delete(AggrStatusesAdgroups.AGGR_STATUSES_ADGROUPS)
                .where(AggrStatusesAdgroups.AGGR_STATUSES_ADGROUPS.PID.in(ids))
                .execute();
    }

    public void deleteAdStatuses(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard).dsl()
                .delete(AggrStatusesBanners.AGGR_STATUSES_BANNERS)
                .where(AggrStatusesBanners.AGGR_STATUSES_BANNERS.BID.in(ids))
                .execute();
    }

    public void deleteKeywordStatuses(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard).dsl()
                .delete(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS)
                .where(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.ID.in(ids))
                .execute();
    }

    public void deleteRetargetingStatuses(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard).dsl()
                .delete(AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS)
                .where(AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS.RET_ID.in(ids))
                .execute();
    }

    public Map<Long, AggregatedStatusKeywordData> getKeywordStatusesByIds(int shard, Collection<Long> ids) {
        AggrStatusesKeywords keywordsTable = AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS;
        return dslContextProvider.ppc(shard).dsl()
                .select(keywordsTable.ID, keywordsTable.STATUS, keywordsTable.REASON)
                .from(keywordsTable)
                .where(keywordsTable.ID.in(ids))
                .fetchMap(keywordsTable.ID, this::aggregatedStatusKeywordData);
    }

    private AggregatedStatusKeywordData aggregatedStatusKeywordData(
            Record5<Long, Long, AggrStatusesKeywordsStatus, AggrStatusesKeywordsReason, Long> record) {
        return aggregatedStatusKeywordData(
                record.get(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.STATUS),
                record.get(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.REASON)
        ).withIsObsolete(booleanFromLong(record.get(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.IS_OBSOLETE)));
    }

    private AggregatedStatusKeywordData aggregatedStatusKeywordData(
            Record3<Long, AggrStatusesKeywordsStatus, AggrStatusesKeywordsReason> record) {
        return aggregatedStatusKeywordData(record.get(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.STATUS),
                record.get(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.REASON));
    }

    private AggregatedStatusKeywordData aggregatedStatusKeywordData(AggrStatusesKeywordsStatus status,
                                                                    AggrStatusesKeywordsReason reason) {
        final var selfStatus = keywordStatusFromDb(status);
        final var selfReason = keywordReasonFromDb(reason);

        return selfReason
                .map(gdSelfStatusReason -> new AggregatedStatusKeywordData(selfStatus, gdSelfStatusReason))
                .orElseGet(() -> new AggregatedStatusKeywordData(selfStatus));
    }

    public Map<Long, AggregatedStatusAdData> getAdStatusesByIds(int shard, Collection<Long> bids) {
        AggrStatusesBanners bannersTable = AggrStatusesBanners.AGGR_STATUSES_BANNERS;
        Map<Long, Pair<String, Long>> jsonsMap = dslContextProvider.ppc(shard).dsl()
                .select(bannersTable.BID, bannersTable.AGGR_DATA, bannersTable.IS_OBSOLETE)
                .from(bannersTable)
                .where(bannersTable.BID.in(bids))
                .fetchMap(bannersTable.BID,
                        r -> Pair.of(r.get(bannersTable.AGGR_DATA), r.get(bannersTable.IS_OBSOLETE)));
        return jsonPairMapToObjectMap(jsonsMap, AggregatedStatusAdData.class);
    }

    public Map<Long, AggregatedStatusAdGroupData> getAdGroupStatusesByIds(int shard, Collection<Long> pids) {
        AggrStatusesAdgroups t = AggrStatusesAdgroups.AGGR_STATUSES_ADGROUPS;
        Map<Long, Pair<String, Long>> jsonsMap = dslContextProvider.ppc(shard).dsl()
                .select(t.PID, t.AGGR_DATA, t.IS_OBSOLETE)
                .from(t)
                .where(t.PID.in(pids))
                .fetchMap(t.PID, r -> Pair.of(r.get(t.AGGR_DATA), r.get(t.IS_OBSOLETE)));
        return jsonPairMapToObjectMap(jsonsMap, AggregatedStatusAdGroupData.class);
    }

    public List<AdWithStatus> getAdStatusesDataForView(int shard, Collection<Long> bids) {
        var st = AggrStatusesBanners.AGGR_STATUSES_BANNERS;
        var b = Banners.BANNERS;
        var c = CAMPAIGNS;
        return dslContextProvider.ppc(shard).dsl()
                .select(b.BID, b.PID, c.PLATFORM, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(b).join(st).on(b.BID.eq(st.BID))
                .join(c).on(b.CID.eq(c.CID))
                .where(b.BID.in(bids))
                .fetch(this::adStatusViewDataFromDb)
                .stream().filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
    }

    public List<AdWithStatus> getAdStatusesDataByPidsForPopup(int shard, ClientId clientId, Collection<Long> adgroupIds) {
        var st = AggrStatusesBanners.AGGR_STATUSES_BANNERS;
        var b = Banners.BANNERS;
        var c = CAMPAIGNS;
        return dslContextProvider.ppc(shard).dsl()
                .select(b.BID, b.PID, c.PLATFORM, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(b)
                .join(st).on(b.BID.eq(st.BID))
                .join(c).on(c.CID.eq(b.CID))
                .where(b.PID.in(adgroupIds).and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetch(this::adStatusViewDataFromDb)
                .stream().filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
    }

    private AdWithStatus adStatusViewDataFromDb(Record5<Long, Long,
            ru.yandex.direct.dbschema.ppc.enums.CampaignsPlatform, String, Long> record) {
        String jsonStatus = record.get(AggrStatusesBanners.AGGR_STATUSES_BANNERS.AGGR_DATA);
        if (jsonStatus == null) {
            return null;
        }
        Long bid = record.get(Banners.BANNERS.BID);
        Long pid = record.get(Banners.BANNERS.PID);
        CampaignsPlatform platform = CampaignsPlatform.fromSource(record.get(CAMPAIGNS.PLATFORM));
        Boolean isObsolete = booleanFromLong(record.get(AggrStatusesBanners.AGGR_STATUSES_BANNERS.IS_OBSOLETE));
        AggregatedStatusAdData statusData = fromJsonSafe(bid, jsonStatus, AggregatedStatusAdData.class);
        if (statusData == null) {
            return null;
        }
        statusData.setIsObsolete(isObsolete);
        statusData.setPlatform(platform);

        return new AdWithStatus()
                .withAdId(bid)
                .withAdGroupId(pid)
                .withStatusData(statusData);
    }

    public List<KeywordWithStatus> getKeywordStatusesDataForView(int shard, Set<Long> adgroupIds,
                                                                 Collection<Long> ids) {
        return unionKeywordsSelect(shard, adgroupIds, ids)
                .where(Bids.BIDS.ID.in(ids))
                .fetch(this::keywordStatusViewDataFromDb)
                .stream().filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
    }

    /**
     * Статусы фраз для отображения попапа на группе. Только по таблице bids, не смотрим на архивные и relevance_match,
     * т.к. их не нужно отображать в попапе
     *
     * @param shard
     * @param clientId  id клиента в рамках которого ищем группы
     * @param adgroupIds id групп фразы из которых ищем
     * @return статусы фраз
     */
    public List<KeywordWithStatus> getKeywordStatusesDataByPidsForPopup(int shard, ClientId clientId, Collection<Long> adgroupIds) {
        var st = AGGR_STATUSES_KEYWORDS;
        var kw = Bids.BIDS;
        /*  Здесь мы не используем unionKeywordsSelect, потому как SQL чтобы выбрать и архивные фразы тоже довольно
            сложный, его пришлось бы приспособить для выбора по ClientId c мерджем по кампании, а реально в
            попапе нам архивные фразы не нужны. Выше по коду есть фильтрация архивных сттаусов, и в этом месте
            правильнее было бы их выбирать, но без них код проще и производетельнее */
        return dslContextProvider.ppc(shard).dsl()
                .select(kw.PID, st.ID, st.STATUS, st.REASON, st.IS_OBSOLETE)
                .from(kw).join(st).on(kw.ID.eq(st.ID))
                .join(CAMPAIGNS).on(kw.CID.eq(CAMPAIGNS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()).and(kw.PID.in(adgroupIds)))
                .fetch(this::keywordStatusViewDataFromDb)
                .stream().filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
    }

    private KeywordWithStatus keywordStatusViewDataFromDb(
            Record5<Long, Long, AggrStatusesKeywordsStatus, AggrStatusesKeywordsReason, Long> record) {
        Long id = record.get(Bids.BIDS.ID);
        Long pid = record.get(Bids.BIDS.PID);

        var status = record.get(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.STATUS);
        var reason = record.get(AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS.REASON);
        Long isObsolete = record.get(AGGR_STATUSES_KEYWORDS.IS_OBSOLETE);

        AggregatedStatusKeywordData statusData = aggregatedStatusKeywordData(status, reason);
        statusData.setIsObsolete(booleanFromLong(isObsolete));

        return new KeywordWithStatus()
                .withKeywordId(id)
                .withAdGroupId(pid)
                .withStatusData(statusData);
    }

    public List<RetargetingWithStatus> getRetargetingStatusesDataForView(int shard, Collection<Long> ids) {
        var st = AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS;
        var ret = BidsRetargeting.BIDS_RETARGETING;
        return dslContextProvider.ppc(shard).dsl()
                .select(ret.RET_ID, ret.PID, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(ret).join(st).on(ret.RET_ID.eq(st.RET_ID))
                .where(ret.RET_ID.in(ids))
                .fetch(this::retargetingStatusViewDataFromDb)
                .stream().filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
    }

    public List<RetargetingWithStatus> getRetargetingStatusesDataByPidsForPopup(int shard, ClientId clientId,
                                                                               Collection<Long> adgroupIds) {
        var st = AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS;
        var ret = BidsRetargeting.BIDS_RETARGETING;
        var g = Phrases.PHRASES;
        return dslContextProvider.ppc(shard).dsl()
                .select(ret.RET_ID, ret.PID, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(ret).join(st).on(ret.RET_ID.eq(st.RET_ID))
                .join(g).on(ret.PID.eq(g.PID))
                .join(CAMPAIGNS).on(g.CID.eq(CAMPAIGNS.CID))
                .where(ret.PID.in(adgroupIds).and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong())))
                .fetch(this::retargetingStatusViewDataFromDb)
                .stream().filter(Objects::nonNull).collect(Collectors.toUnmodifiableList());
    }

    private RetargetingWithStatus retargetingStatusViewDataFromDb(Record4<Long, Long, String, Long> record) {
        String jsonStatus = record.get(AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS.AGGR_DATA);
        if (jsonStatus == null) {
            return null;
        }
        Long retargetingId = record.get(BidsRetargeting.BIDS_RETARGETING.RET_ID);
        Long pid = record.get(BidsRetargeting.BIDS_RETARGETING.PID);
        Boolean isObsolete =
                booleanFromLong(record.get(AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS.IS_OBSOLETE));
        var statusData = fromJsonSafe(retargetingId, jsonStatus, AggregatedStatusRetargetingData.class);
        if (statusData == null) {
            return null;
        }
        statusData.setIsObsolete(isObsolete);

        return new RetargetingWithStatus()
                .withRetargetingId(retargetingId)
                .withAdGroupId(pid)
                .withStatusData(statusData);
    }

    public Map<Long, AggregatedStatusCampaignData> getCampaignStatusesByIds(int shard, Collection<Long> cids) {
        AggrStatusesCampaigns t = AGGR_STATUSES_CAMPAIGNS;
        Map<Long, Pair<String, Long>> jsonsMap = dslContextProvider.ppc(shard).dsl()
                .select(t.CID, t.AGGR_DATA, t.IS_OBSOLETE)
                .from(t)
                .where(t.CID.in(cids))
                .fetchMap(t.CID, r -> Pair.of(r.get(t.AGGR_DATA), r.get(t.IS_OBSOLETE)));
        return jsonPairMapToObjectMap(jsonsMap, AggregatedStatusCampaignData.class);
    }

    // Возвращаются в одном методе, чтобы получать их одним запросом в БД вместо двух, т.к. они всегда нужны вместе
    public RejectReasonsAndCountersByIds<AdGroupCounters> getAdGroupRejectReasonsAndCountersByIds(int shard,
                                                                                                  Collection<Long> pids) {
        AggrStatusesAdgroups t = AggrStatusesAdgroups.AGGR_STATUSES_ADGROUPS;
        Field<String> rr = rejectReasonsFieldForTableSelect(t);
        Field<String> cnts = countersFieldForTableSelect(t);
        Tuple2<Map<Long, String>, Map<Long, String>> jsonsMaps = recordsIntoMaps(
                dslContextProvider.ppc(shard).dsl().select(t.PID, rr, cnts)
                        .from(t)
                        .where(t.PID.in(pids))
                        .fetch()
        );
        var reasons = jsonStringMapToObjectMap(jsonsMaps._1, new TypeReference<Map<ModerationDiagType, Set<Long>>>() { });
        var counters = jsonStringMapToObjectMap(jsonsMaps._2, AdGroupCounters.class);
        return new RejectReasonsAndCountersByIds<>(reasons, counters);
    }

    public RejectReasonsAndCountersByIds<CampaignCounters> getCampaignRejectReasonsAndCountersByIds(int shard,
                                                                                                    Collection<Long> cids) {
        AggrStatusesCampaigns t = AGGR_STATUSES_CAMPAIGNS;
        Field<String> rr = rejectReasonsFieldForTableSelect(t);
        Field<String> cnts = countersFieldForTableSelect(t);
        Tuple2<Map<Long, String>, Map<Long, String>> jsonsMaps = recordsIntoMaps(
                dslContextProvider.ppc(shard).dsl()
                        .select(t.CID, rr, cnts)
                        .from(t)
                        .where(t.CID.in(cids))
                        .fetch()
        );
        var reasons = jsonStringMapToObjectMap(jsonsMaps._1, new TypeReference<Map<ModerationDiagType, Set<Long>>>() { });
        var counters = jsonStringMapToObjectMap(jsonsMaps._2, CampaignCounters.class);
        return new RejectReasonsAndCountersByIds<>(reasons, counters);
    }

    /**
     * Метод возвращает список статусов групп на кампанию для обновления счетчиков, поэтому id - фраз
     * не возвращаются
     */
    public Map<Long, List<AggregatedStatusAdGroupData>> getAdGroupStatusesOnCampaignsForCount(int shard,
                                                                                              @Nullable Collection<Long> cids,
                                                                                              boolean parallel) {
        if (cids == null || cids.isEmpty()) {
            return Map.of();
        }

        Phrases t = Phrases.PHRASES;
        AggrStatusesAdgroups st = AggrStatusesAdgroups.AGGR_STATUSES_ADGROUPS;
        Result<Record4<Long, Long, String, Long>> result = dslContextProvider.ppc(shard).dsl()
                .select(t.CID, t.PID, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(t).join(st).on(t.PID.eq(st.PID))
                .where(t.CID.in(cids))
                .fetch();
        Stream<Record4<Long, Long, String, Long>> stream;
        if (parallel) {
            stream = result.parallelStream();
        } else {
            stream = result.stream();
        }
        return stream.collect(forCountGroupByCollector(t.CID, t.PID, AggregatedStatusAdGroupData.class));
    }

    /**
     * Метод возвращает список статусов объявлений на группу для обновления счетчиков, поэтому id - фраз
     * не возвращаются
     */
    public Map<Long, List<AggregatedStatusAdData>> getAdsStatusesOnAdGroupsForCount(int shard,
                                                                                    @Nullable Collection<Long> pids) {
        if (pids == null || pids.isEmpty()) {
            return Map.of();
        }

        Banners t = Banners.BANNERS;
        AggrStatusesBanners st = AggrStatusesBanners.AGGR_STATUSES_BANNERS;
        return dslContextProvider.ppc(shard).dsl()
                .select(t.PID, t.BID, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(t).join(st).on(t.BID.eq(st.BID))
                .where(t.PID.in(pids))
                .fetch().stream().collect(forCountGroupByCollector(t.PID, t.BID, AggregatedStatusAdData.class));
    }

    public Map<Long, Map<Long, AggregatedStatusAdData>> getAdsStatusesOnAdGroupsForCountWithId(int shard,
                                                                                               @Nullable Collection<Long> pids) {
        if (pids == null || pids.isEmpty()) {
            return Map.of();
        }

        Banners t = Banners.BANNERS;
        AggrStatusesBanners st = AggrStatusesBanners.AGGR_STATUSES_BANNERS;
        return dslContextProvider.ppc(shard).dsl()
                .select(t.PID, t.BID, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(t).join(st).on(t.BID.eq(st.BID))
                .where(t.PID.in(pids))
                .fetch().stream().collect(forCountWithIdGroupByCollector(t.PID, t.BID, AggregatedStatusAdData.class));
    }

    /**
     * Метод возвращает список статусов ключевых слов на группу для обновления счетчиков, поэтому id - фраз
     * не возвращаются
     */
    public Map<Long, List<AggregatedStatusKeywordData>> getKeywordsStatusesOnAdGroupsForCount(int shard,
                                                                                              @Nullable Collection<Long> pids) {
        if (pids == null || pids.isEmpty()) {
            return Map.of();
        }

        Bids kw = Bids.BIDS;
        AggrStatusesKeywords st = AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS;

        SelectJoinStep<Record5<Long, Long, AggrStatusesKeywordsStatus, AggrStatusesKeywordsReason, Long>> allKeywordsSelect =
                unionKeywordsSelect(shard, pids, null);

        return allKeywordsSelect.fetch().stream().collect(
                Collectors.groupingBy(
                        r -> r.get(kw.PID),
                        Collectors.mapping(this::aggregatedStatusKeywordData,
                                Collectors.filtering(Objects::nonNull, Collectors.toList()))));
    }

    /**
     * Метод возвращает список статусов ретаргетингов на группу для обновления счетчиков
     */
    public Map<Long, List<AggregatedStatusRetargetingData>> getRetargetingsStatusesOnAdGroupsForCount(int shard,
                                                                                                      @Nullable Collection<Long> pids) {
        if (pids == null || pids.isEmpty()) {
            return Map.of();
        }

        BidsRetargeting t = BidsRetargeting.BIDS_RETARGETING;
        AggrStatusesRetargetings st = AggrStatusesRetargetings.AGGR_STATUSES_RETARGETINGS;
        return dslContextProvider.ppc(shard)
                .select(t.PID, t.RET_ID, st.AGGR_DATA, st.IS_OBSOLETE)
                .from(t).join(st).on(t.RET_ID.eq(st.RET_ID))
                .where(t.PID.in(pids))
                .fetch().stream()
                .collect(forCountGroupByCollector(t.PID, t.RET_ID, AggregatedStatusRetargetingData.class));
    }

    private SelectJoinStep<Record5<Long, Long, AggrStatusesKeywordsStatus, AggrStatusesKeywordsReason, Long>>
    unionKeywordsSelect(int shard, @NotNull Collection<Long> pids, @Nullable Collection<Long> ids) {
        Bids kw = Bids.BIDS;
        BidsArc kwArc = BidsArc.BIDS_ARC;
        BidsBase relevanceMatch = BidsBase.BIDS_BASE;
        AggrStatusesKeywords st = AggrStatusesKeywords.AGGR_STATUSES_KEYWORDS;
        Phrases g = Phrases.PHRASES;

        var bidsSelect = dslContextProvider.ppc(shard).dsl().select(kw.PID, kw.ID).from(kw);
        var bidsSelectWithWhere = ids != null
                ? bidsSelect.where(kw.PID.in(pids).and(kw.ID.in(ids)))
                : bidsSelect.where(kw.PID.in(pids));

        var bidsArcSelect = dslContextProvider.ppc(shard).dsl().select(kwArc.PID, kwArc.ID)
                .from(g).join(kwArc).on(g.CID.eq(kwArc.CID).and(g.PID.eq(kwArc.PID)));
        var bidsArcSelectWithWhere = ids != null
                ? bidsArcSelect.where(g.PID.in(pids).and(kwArc.ID.in(ids)))
                : bidsArcSelect.where(g.PID.in(pids));

        var bidsBaseSelect = dslContextProvider.ppc(shard).dsl()
                .select(relevanceMatch.PID, relevanceMatch.BID_ID.as(kw.ID))
                .from(relevanceMatch)
                .where(relevanceMatch.PID.in(pids));
        if (ids != null) {
            bidsBaseSelect.and(relevanceMatch.BID_ID.in(ids));
        }
        var bidsBaseSelectWithWhere = bidsBaseSelect.and(BIDS_BASE.BID_TYPE.ne(BidsBaseBidType.keyword));

        return dslContextProvider.ppc(shard).dsl()
                .select(kw.PID, st.ID, st.STATUS, st.REASON, st.IS_OBSOLETE)
                .from(bidsSelectWithWhere.union(bidsArcSelectWithWhere)
                        .union(bidsBaseSelectWithWhere).asTable(kw.getName()))
                .join(st).using(st.ID);
    }

    // сделано просто чтобы один и тот же коллектор не повторять четыре раза
    private <V> Collector<Record4<Long, Long, String, Long>, ?, Map<Long, List<V>>> forCountGroupByCollector(
            Field<Long> keyField, Field<Long> statusIdField, Class<V> cls) {
        return Collectors.groupingBy(
                r -> r.get(keyField),
                Collectors.mapping(r ->
                                fromJsonSafeAndObsolete(r.get(statusIdField), r.get(AGGR_DATA),
                                        r.get(IS_OBSOLETE), cls),
                        Collectors.filtering(Objects::nonNull, Collectors.toList())));
    }

    private <V> Collector<Record4<Long, Long, String, Long>, ?, Map<Long, Map<Long, V>>> forCountWithIdGroupByCollector(
            Field<Long> keyField, Field<Long> statusIdField, Class<V> cls) {
        return Collectors.groupingBy(
                r -> r.get(keyField),
                Collectors.toMap(
                        r -> r.get(statusIdField),
                        r -> fromJsonSafeAndObsolete(r.get(statusIdField), r.get(AGGR_DATA), r.get(IS_OBSOLETE), cls)));
    }

    private <K, V1, V2> Tuple2<Map<K, V1>, Map<K, V2>> recordsIntoMaps(Iterable<Record3<K, V1, V2>> records) {
        var firstMap = new HashMap<K, V1>();
        var secondMap = new HashMap<K, V2>();
        for (var row : records) {
            firstMap.put(row.value1(), row.value2());
            secondMap.put(row.value1(), row.value3());
        }
        return Tuple2.tuple(firstMap, secondMap);
    }

    private static Field<String> countersFieldForTableSelect(Table<?> table) {
        Field<String> field = getFieldForTable(table, AGGR_DATA);
        return DSL.field("JSON_EXTRACT({0}, {1})",
                field.getDataType(), field, DSL.val("$." + JSON_COUNTERS_SUBFIELD)).as(JSON_COUNTERS_SUBFIELD);
    }

    private static Field<String> rejectReasonsFieldForTableSelect(Table<?> table) {
        var field = getFieldForTable(table, AGGR_DATA);
        return DSL.field("JSON_EXTRACT({0}, {1})",
                field.getDataType(), field, DSL.val("$." + JSON_REJECT_REASONS_SUBFIELD)).as(JSON_REJECT_REASONS_SUBFIELD);
    }

    // когда точно такое же поле, но в другой таблице
    private static <T> Field<T> getFieldForTable(Table<?> table, Field<T> field) {
        return table.field(field.getName(), field.getDataType());
    }

    private static Field<String> sqlValues(Field<String> field) {
        return DSL.function("VALUES", String.class, DSL.field(field));
    }

    private static <T> Field<T> sqlValues(Field<T> field, Class<T> fieldClass) {
        return DSL.function("VALUES", fieldClass, DSL.field(field));
    }

    private <K> Map<Long, K> jsonPairMapToObjectMap(Map<Long, Pair<String, Long>> jsonMap, Class<K> cls) {
        return EntryStream.of(jsonMap)
                .mapToValue((id, pair) -> pair.getLeft() == null
                        ? null
                        : fromJsonSafeAndObsolete(id, pair.getLeft(), pair.getRight(), cls)
                )
                .nonNullValues()
                .toMap();
    }

    public <K> K fromJsonSafeAndObsolete(Long id, String json, @Nullable Long isObsolete, Class<K> cls) {
        K object = fromJsonSafe(id, json, cls);
        if (isObsolete != null && object instanceof AggregatedStatusBaseData) {
            ((AggregatedStatusBaseData) object).setIsObsolete(booleanFromLong(isObsolete));
        }
        return object;
    }

    private <K> Map<Long, K> jsonStringMapToObjectMap(Map<Long, String> jsonMap, Class<K> cls) {
        return EntryStream.of(jsonMap)
                .mapToValue((id, json) -> json == null ? null : fromJsonSafe(id, json, cls))
                .nonNullValues()
                .toMap();
    }

    private <K> Map<Long, K> jsonStringMapToObjectMap(Map<Long, String> jsonMap, TypeReference<K> typeRef) {
        return EntryStream.of(jsonMap)
                .mapToValue((id, json) -> json == null ? null : fromJsonSafe(id, json, typeRef))
                .nonNullValues()
                .toMap();
    }

    // id исключительно для правильного логирования
    public <K> K fromJsonSafe(Long id, String json, Class<K> cls) {
        logger.debug("Deserializing json for {} id: {}", cls, id);
        try {
            return objectMapper.readValue(json, cls);
        } catch (IOException e) {
            logger.error("Can't deserialize json for {} id: {}, json: {}\nException: {}", cls, id, json, e);
            return null;
        }
    }

    public <K> K fromJsonSafe(Long id, String json, TypeReference<K> typeRef) {
        logger.debug("Deserializing json for {} id: {}", typeRef.getType(), id);
        try {
            return objectMapper.readValue(json, typeRef);
        } catch (IOException e) {
            logger.error("Can't deserialize json for {} id: {}, json: {}\nException: {}", typeRef.getType(), id, json, e);
            return null;
        }
    }

    private <V> String toJson(V data) {
        try {
            return objectMapper.writeValueAsString(data);
        } catch (JsonProcessingException e) {
            logger.error("Error when generating json for AggregatedStatusData: {}", data);
            throw new IllegalArgumentException("Error when generating json for AggregatedStatusData", e);
        }
    }

    /**
     * Используется только для тестирования.
     * Обновляет значение UPDATED в таблице агрегированных статусов кампаний.
     */
    @VisibleForTesting
    public void setCampaignStatusUpdateTime(int shard, Long campaignId, LocalDateTime update) {
        dslContextProvider.ppc(shard)
                .update(AGGR_STATUSES_CAMPAIGNS)
                .set(AGGR_STATUSES_CAMPAIGNS.UPDATED, update)
                .where(AGGR_STATUSES_CAMPAIGNS.CID.equal(campaignId))
                .execute();
    }

    /**
     * Используется только для тестирования.
     * Обновляет значение UPDATED в таблице агрегированных статусов кампаний.
     */
    @VisibleForTesting
    public void setKeywordStatusUpdateTime(int shard, Long campaignId, LocalDateTime update) {
        dslContextProvider.ppc(shard)
                .update(AGGR_STATUSES_KEYWORDS)
                .set(AGGR_STATUSES_KEYWORDS.UPDATED, update)
                .where(AGGR_STATUSES_KEYWORDS.ID.equal(campaignId))
                .execute();
    }

    @VisibleForTesting
    public void setAdGroupStatusUpdateTime(int shard, Long adGroupId, LocalDateTime update) {
        dslContextProvider.ppc(shard)
                .update(AGGR_STATUSES_ADGROUPS)
                .set(AGGR_STATUSES_ADGROUPS.UPDATED, update)
                .where(AGGR_STATUSES_ADGROUPS.PID.equal(adGroupId))
                .execute();
    }

    @VisibleForTesting
    public void setAdStatusUpdateTime(int shard, Long bannerId, LocalDateTime update) {
        dslContextProvider.ppc(shard)
                .update(AGGR_STATUSES_BANNERS)
                .set(AGGR_STATUSES_BANNERS.UPDATED, update)
                .where(AGGR_STATUSES_BANNERS.BID.equal(bannerId))
                .execute();
    }

    @VisibleForTesting
    public void setRetargetingStatusUpdateTime(int shard, Long retargetingId, LocalDateTime update) {
        dslContextProvider.ppc(shard)
                .update(AGGR_STATUSES_RETARGETINGS)
                .set(AGGR_STATUSES_RETARGETINGS.UPDATED, update)
                .where(AGGR_STATUSES_RETARGETINGS.RET_ID.equal(retargetingId))
                .execute();
    }

    /**
     * Используется только для тестирования.
     * Возвращает map id => is_obsolete
     */
    @VisibleForTesting
    public Map<Long, Boolean> getAdStatusesIsObsolete(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(AGGR_STATUSES_BANNERS.BID, AGGR_STATUSES_BANNERS.IS_OBSOLETE)
                .from(AGGR_STATUSES_BANNERS)
                .where(AGGR_STATUSES_BANNERS.BID.in(ids))
                .fetchMap(r -> r.getValue(AGGR_STATUSES_BANNERS.BID),
                        r -> booleanFromLong(r.getValue(AGGR_STATUSES_BANNERS.IS_OBSOLETE)));
    }

    @VisibleForTesting
    public Map<Long, Boolean> getAdGroupStatusesIsObsolete(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(AGGR_STATUSES_ADGROUPS.PID, AGGR_STATUSES_ADGROUPS.IS_OBSOLETE)
                .from(AGGR_STATUSES_ADGROUPS)
                .where(AGGR_STATUSES_ADGROUPS.PID.in(ids))
                .fetchMap(r -> r.getValue(AGGR_STATUSES_ADGROUPS.PID),
                        r -> booleanFromLong(r.getValue(AGGR_STATUSES_ADGROUPS.IS_OBSOLETE)));
    }

    @VisibleForTesting
    public Map<Long, Boolean> getCampaignStatusesIsObsolete(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(AGGR_STATUSES_CAMPAIGNS.CID, AGGR_STATUSES_CAMPAIGNS.IS_OBSOLETE)
                .from(AGGR_STATUSES_CAMPAIGNS)
                .where(AGGR_STATUSES_CAMPAIGNS.CID.in(ids))
                .fetchMap(r -> r.getValue(AGGR_STATUSES_CAMPAIGNS.CID),
                        r -> booleanFromLong(r.getValue(AGGR_STATUSES_CAMPAIGNS.IS_OBSOLETE)));
    }

    @VisibleForTesting
    public Map<Long, Boolean> getKeywordStatusesIsObsolete(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(AGGR_STATUSES_KEYWORDS.ID, AGGR_STATUSES_KEYWORDS.IS_OBSOLETE)
                .from(AGGR_STATUSES_KEYWORDS)
                .where(AGGR_STATUSES_KEYWORDS.ID.in(ids))
                .fetchMap(r -> r.getValue(AGGR_STATUSES_KEYWORDS.ID),
                        r -> booleanFromLong(r.getValue(AGGR_STATUSES_KEYWORDS.IS_OBSOLETE)));
    }

    @VisibleForTesting
    public Map<Long, Boolean> getRetargetingStatusesIsObsolete(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(AGGR_STATUSES_RETARGETINGS.RET_ID, AGGR_STATUSES_RETARGETINGS.IS_OBSOLETE)
                .from(AGGR_STATUSES_RETARGETINGS)
                .where(AGGR_STATUSES_RETARGETINGS.RET_ID.in(ids))
                .fetchMap(r -> r.getValue(AGGR_STATUSES_RETARGETINGS.RET_ID),
                        r -> booleanFromLong(r.getValue(AGGR_STATUSES_RETARGETINGS.IS_OBSOLETE)));
    }
}
