package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusQueueEntityStatus;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusQueueEntityType;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusResyncQueueEntity;
import ru.yandex.direct.dbschema.ppc.enums.AggrStatusesResyncQueueStatus;
import ru.yandex.direct.dbschema.ppc.tables.records.AggrStatusesResyncQueueRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.tables.AggrStatusesResyncQueue.AGGR_STATUSES_RESYNC_QUEUE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class AggregatedStatusesResyncQueueRepository {
    private final DslContextProvider ppcDslContextProvider;
    private final JooqMapperWithSupplier<AggregatedStatusResyncQueueEntity> mapper;

    @Autowired
    public AggregatedStatusesResyncQueueRepository(DslContextProvider ppcDslContextProvider) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.mapper = createMapper();
    }

    private static JooqMapperWithSupplier<AggregatedStatusResyncQueueEntity> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(AggregatedStatusResyncQueueEntity::new)
                .map(property(AggregatedStatusResyncQueueEntity.ID, AGGR_STATUSES_RESYNC_QUEUE.ID))
                .map(convertibleProperty(AggregatedStatusResyncQueueEntity.TYPE, AGGR_STATUSES_RESYNC_QUEUE.TYPE,
                        AggregatedStatusQueueEntityType::fromSource, AggregatedStatusQueueEntityType::toSource))
                .map(property(AggregatedStatusResyncQueueEntity.ENTITY_ID, AGGR_STATUSES_RESYNC_QUEUE.ENTITY_ID))
                .map(convertibleProperty(AggregatedStatusResyncQueueEntity.STATUS, AGGR_STATUSES_RESYNC_QUEUE.STATUS,
                        AggregatedStatusQueueEntityStatus::fromSource, AggregatedStatusQueueEntityStatus::toSource))
                .build();
    }

    /**
     * Вычитывает записи из таблицы AGGR_STATUSES_RESYNC_QUEUE записи в порядке QUEUE_TIME
     *
     * @param shard шард
     * @param limit лимит, в таблице может накопиться много записей
     */
    public List<AggregatedStatusResyncQueueEntity> getAggregatedStatusResyncQueueEntities(int shard, int limit) {
        return ppcDslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead())
                .from(AGGR_STATUSES_RESYNC_QUEUE)
                .orderBy(AGGR_STATUSES_RESYNC_QUEUE.QUEUE_TIME)
                .limit(limit)
                .fetch(mapper::fromDb);
    }

    /**
     * Обновление флага IS_AWAITING_PROCESSING
     */
    public int setAwaitingProcessingFlag(int shard, Collection<Long> ids, AggregatedStatusQueueEntityStatus status) {
        return ppcDslContextProvider.ppc(shard)
                .update(AGGR_STATUSES_RESYNC_QUEUE)
                .set(AGGR_STATUSES_RESYNC_QUEUE.STATUS, AggregatedStatusQueueEntityStatus.toSource(status))
                .where(AGGR_STATUSES_RESYNC_QUEUE.ID.in(ids))
                .execute();
    }

    /**
     * Запись в очередь пересчета статусов, при наличии объекта в очереди
     * обновляем QUEUE_TIME только если объект в статусе processing
     */
    public int addAggregatedStatusResyncQueueEntities(int shard, List<AggregatedStatusResyncQueueEntity> entities) {
        DSLContext ppc = ppcDslContextProvider.ppc(shard);
        InsertHelper<AggrStatusesResyncQueueRecord> insertHelper = new InsertHelper<>(ppc, AGGR_STATUSES_RESYNC_QUEUE);
        insertHelper.addAll(mapper, entities);
        return insertHelper
                .onDuplicateKeyUpdate()
                .set(AGGR_STATUSES_RESYNC_QUEUE.QUEUE_TIME, DSL.decode()
                        .when(AGGR_STATUSES_RESYNC_QUEUE.STATUS.eq(AggrStatusesResyncQueueStatus.processing),
                                DSL.currentLocalDateTime())
                        .otherwise(AGGR_STATUSES_RESYNC_QUEUE.QUEUE_TIME))
                .set(AGGR_STATUSES_RESYNC_QUEUE.STATUS, MySQLDSL.values(AGGR_STATUSES_RESYNC_QUEUE.STATUS))
                .executeIfRecordsAdded();
    }

    /**
     * Удаляем объекты из очереди, если у них не проставился статус IS_AWAITING_PROCESSING == 1, иначе считаем,
     * что статус объекта нужно будет пересчитать
     */
    public int deleteProcessedAggregatedStatusResyncQueueEntities(int shard, Collection<Long> ids) {
        return ppcDslContextProvider.ppc(shard)
                .deleteFrom(AGGR_STATUSES_RESYNC_QUEUE)
                .where(AGGR_STATUSES_RESYNC_QUEUE.ID.in(ids)
                        .and(AGGR_STATUSES_RESYNC_QUEUE.STATUS.eq(AggrStatusesResyncQueueStatus.processing)))
                .execute();
    }
}
