package ru.yandex.direct.core.aggregatedstatuses.repository;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.BannerDataForStatus;
import ru.yandex.direct.core.entity.banner.repository.ModerateBannerPagesRepository;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiag;
import ru.yandex.direct.core.entity.moderationdiag.model.ModerationDiagType;
import ru.yandex.direct.core.entity.moderationreason.model.ModerationReasonObjectType;
import ru.yandex.direct.core.entity.moderationreason.service.ModerationReasonService;

import static java.util.Collections.emptyList;

@Service
@ParametersAreNonnullByDefault
public final class BannerDataForStatusService {
    private final AggregatedStatusesBannerRepository bannerRepository;
    private final ModerateBannerPagesRepository moderateBannerPagesRepository;
    private final ModerationReasonService moderationReasonService;

    @Autowired
    public BannerDataForStatusService(AggregatedStatusesBannerRepository bannerRepository,
                                      ModerateBannerPagesRepository moderateBannerPagesRepository,
                                      ModerationReasonService moderationReasonService) {
        this.bannerRepository = bannerRepository;
        this.moderateBannerPagesRepository = moderateBannerPagesRepository;
        this.moderationReasonService = moderationReasonService;
    }

    private Map<ModerationDiagType, Set<Long>> diagIdsByDiagType(
            Map<ModerationReasonObjectType, List<ModerationDiag>> diagsByModerationObjectType
    ) {
        final var result = new HashMap<ModerationDiagType, Set<Long>>();

        diagsByModerationObjectType.values().stream()
                .flatMap(Collection::stream)
                .forEach(diag -> result.computeIfAbsent(diag.getType(), d -> new HashSet<>()).add(diag.getId()));

        return result;
    }

    public List<BannerDataForStatus> bannerDataForStatusByBids(int shard, List<Long> bids) {
        if (bids.isEmpty()) {
            return emptyList();
        }

        final var banners = bannerRepository.getBanners(shard, bids);
        final var pagesModeration = moderateBannerPagesRepository.getModerateBannerPagesByBannerIds(
                shard, new HashSet<>(bids)
        );

        final var reasons = moderationReasonService.getReasons(shard, bids)
                .entrySet()
                .stream()
                .collect(Collectors.toMap(Map.Entry::getKey, entry -> this.diagIdsByDiagType(entry.getValue())));

        return banners.stream().map(banner -> new BannerDataForStatus(
                banner.withPagesModeration(pagesModeration.getOrDefault(banner.getId(), List.of())),
                reasons.getOrDefault(banner.getId(), Map.of()))
        ).collect(Collectors.toList());
    }
}
