package ru.yandex.direct.core.aggregatedstatuses.service;

import java.time.LocalDate;
import java.time.Period;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.campaign.aggrstatus.AggregatedStatusCampaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategyBase;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.service.CampaignStrategyUtils;
import ru.yandex.direct.core.entity.campaignstatistic.repository.CampaignStatisticService;
import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.model.GoalConversion;
import ru.yandex.direct.grid.core.entity.model.campaign.AggregatorGoal;

import static ru.yandex.direct.core.entity.campaign.service.CampaignStrategyConstants.NUM_OF_DAYS_FOR_CONVERSION_LOOK_UP;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class AggregatedStatusesHelper {
    private final CampaignStatisticService campaignStatisticService;

    @Autowired
    public AggregatedStatusesHelper(CampaignStatisticService campaignStatisticService) {
        this.campaignStatisticService = campaignStatisticService;
    }

    /**
     * Получает конверсии по кампаниям пессимизированных клиентов.
     * При выключенной проперти ENABLE_CALCULATION_LACK_OF_CONVERSION_AGGREGATED_STATUS,
     * пессимизированных клиентов считаем -- нет
     */
    public Map<Long, GoalConversion> getPessimizedGoalConversionsCount(
            Collection<AggregatedStatusCampaign> campaigns,
            Map<ClientId, Boolean> pessimizedFlagByClientId) {
        Set<AggregatedStatusCampaign> campaignsOfPessimizedClients = StreamEx.of(campaigns)
                .mapToEntry(AggregatedStatusCampaign::getClientId)
                .mapValues(ClientId::fromLong)
                .filterValues(pessimizedFlagByClientId::containsKey)
                .filterValues(pessimizedFlagByClientId::get)
                .keys()
                .toSet();

        Map<Long, Long> goalIdByCampaignId = StreamEx.of(campaignsOfPessimizedClients)
                .mapToEntry(
                        AggregatedStatusCampaign::getId,
                        campaign -> campaign.getStrategy().getStrategyData().getGoalId())
                .nonNullValues()
                .toMap();
        Map<Long, List<AggregatorGoal>> aggregatorGoalsByCampaignId =
                getAggregatorGoalsByCampaignId(campaignsOfPessimizedClients);
        LocalDate now = LocalDate.now();
        LocalDate dateFrom = now.minus(Period.ofDays(NUM_OF_DAYS_FOR_CONVERSION_LOOK_UP));
        LocalDateRange localDateRange = new LocalDateRange().withFromInclusive(dateFrom).withToInclusive(now);
        return campaignStatisticService.getCampaignGoalConversionsCount(
                localDateRange, goalIdByCampaignId, aggregatorGoalsByCampaignId);
    }

    private Map<Long, List<AggregatorGoal>> getAggregatorGoalsByCampaignId(
            Set<AggregatedStatusCampaign> campaignsOfPessimizedClients) {
        return StreamEx.of(campaignsOfPessimizedClients)
                .mapToEntry(AggregatedStatusCampaign::getStrategy)
                .mapValues(DbStrategyBase::getStrategyData)
                .mapValues(StrategyData::getGoalId)
                .nonNullValues()
                .filterValues(CampaignStrategyUtils::isAggregatorGoalId)
                .mapToValue((campaign, goalId) -> List.of(
                        new AggregatorGoal()
                                .withId(MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID)
                                .withSubGoalIds(mapList(campaign.getMeaningfulGoals(), MeaningfulGoal::getGoalId))))
                .mapKeys(AggregatedStatusCampaign::getId)
                .toMap();
    }
}
