package ru.yandex.direct.core.aggregatedstatuses.service;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesResyncQueueRepository;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedEntityIdWithType;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusQueueEntityStatus;
import ru.yandex.direct.core.entity.aggregatedstatus.model.AggregatedStatusResyncQueueEntity;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;


@Service
@ParametersAreNonnullByDefault
public class AggregatedStatusesResyncQueueService {
    public static final int LIMIT = 50_000;
    public static final int LIMIT_TO_ADD = 5000;
    private final AggregatedStatusesResyncQueueRepository aggregatedStatusesResyncQueueRepository;

    @Autowired
    public AggregatedStatusesResyncQueueService(AggregatedStatusesResyncQueueRepository aggregatedStatusesResyncQueueRepository) {
        this.aggregatedStatusesResyncQueueRepository = aggregatedStatusesResyncQueueRepository;
    }

    public List<AggregatedStatusResyncQueueEntity> getResyncQueueEntitiesAndMarkAsProcessing(int shard) {
        var resyncQueueEntities =
                getLimitedAggregatedStatusResyncQueueEntities(shard);

        List<Long> resyncIds = mapList(resyncQueueEntities, AggregatedStatusResyncQueueEntity::getId);
        setAwaitingProcessingFlag(shard, resyncIds, AggregatedStatusQueueEntityStatus.PROCESSING);
        return resyncQueueEntities;
    }

    /**
     * Частично выгружаем объекты, которые нужно пересчитать
     * ограничиваем набор выгружаемых объектов, чтобы за ожидаемое время пересчитывать статусы из очереди
     **/
    public List<AggregatedStatusResyncQueueEntity> getLimitedAggregatedStatusResyncQueueEntities(int shard) {
        return aggregatedStatusesResyncQueueRepository.getAggregatedStatusResyncQueueEntities(shard, LIMIT);
    }

    /**
     * Обновление флага IS_AWAITING_PROCESSING
     */
    public int setAwaitingProcessingFlag(int shard, List<Long> ids, AggregatedStatusQueueEntityStatus status) {
        return aggregatedStatusesResyncQueueRepository.setAwaitingProcessingFlag(shard, ids, status);
    }

    /**
     * получаем список уже находящихся объектов в базе
     * те кто сейчас уже обрабатывается:
     * добавляем(на случай если запись успеет удалиться) или обновляем статус и queue_time
     * те кто уже есть в очереди -- не меняем, тк они уже в очереди
     * те кого нет добавляем
     */
    public void addEntitiesToAggregatedStatusesResyncQueue(
            int shard,
            Collection<AggregatedEntityIdWithType> entityIdWithTypeCollection) {
        if (entityIdWithTypeCollection.isEmpty()) {
            return;
        }
        List<AggregatedStatusResyncQueueEntity> entitiesForModification = StreamEx.of(entityIdWithTypeCollection)
                .map(entityIdWithType -> new AggregatedStatusResyncQueueEntity()
                        .withEntityId(entityIdWithType.getEntityId())
                        .withStatus(AggregatedStatusQueueEntityStatus.WAITING)
                        .withType(entityIdWithType.getType()))
                .distinct()
                .toList();
        StreamEx.ofSubLists(entitiesForModification, LIMIT_TO_ADD).forEach(entitiesToAdd ->
                aggregatedStatusesResyncQueueRepository.addAggregatedStatusResyncQueueEntities(shard, entitiesToAdd));
    }

    /**
     * Удаляет из очереди записи.
     * <p>
     * если у объекта изменился статус обработки в ходе обновления агрегиированного статуса
     * удалять объект из очереди, может быть неправильно, тк появились новые изменения для таких id
     **/
    public void deleteProcessedAggregatedStatusResyncQueueEntities(int shard, List<Long> ids) {
        if (ids.isEmpty()) {
            return;
        }
        aggregatedStatusesResyncQueueRepository.deleteProcessedAggregatedStatusResyncQueueEntities(shard, ids);
    }
}
