package ru.yandex.direct.core.bsexport.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsAutobudget;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.libs.timetarget.TimeTarget;

import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.dbschema.ppc.Tables.AB_SEGMENT_MULTIPLIER_VALUES;
import static ru.yandex.direct.dbschema.ppc.Tables.HIERARCHICAL_MULTIPLIERS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_MULTIPLIER_VALUES;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class BsExportMultipliersRepository {
    private final DslContextProvider dslContextProvider;
    private final FeatureService featureService;
    private final CampaignRepository campaignRepository;

    public BsExportMultipliersRepository(DslContextProvider dslContextProvider, CampaignRepository campaignRepository,
                                         FeatureService featureService) {
        this.dslContextProvider = dslContextProvider;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
    }

    /**
     * Получить hierarchical_multiplier_id корректировок из retargeting_multiplier_values,
     * которые используют указанные условия ретаргетинга
     */
    public List<Long> getRetargetingHierarchicalMultiplierIdsByRetargetingConditionIds(int shard, Collection<Long> retCondIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)
                .from(RETARGETING_MULTIPLIER_VALUES)
                .where(RETARGETING_MULTIPLIER_VALUES.RET_COND_ID.in(retCondIds))
                .fetch(RETARGETING_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
    }

    /**
     * Получить hierarchical_multiplier_id корректировок
     * из retargeting_multiplier_values и ab_segment_multiplier_values,
     * которые используют указанные условия ретаргетинга
     */
    public List<Long> getMultiplierIdsByRetargetingConditionIds(int shard, Collection<Long> retCondIds) {
        return StreamEx.of(getRetargetingHierarchicalMultiplierIdsByRetargetingConditionIds(shard, retCondIds))
                .append(getAbSegmentHierarchicalMultiplierIdsByAbSegmentRetargetingConditionIds(shard, retCondIds))
                .toList();
    }

    /**
     * Получить ID корректировок из hierarchical_multipliers, которые используют указанные условия ретаргетинга
     */
    private List<Long> getAbSegmentHierarchicalMultiplierIdsByAbSegmentRetargetingConditionIds(int shard, Collection<Long> retCondIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID)
                .from(AB_SEGMENT_MULTIPLIER_VALUES)
                .where(AB_SEGMENT_MULTIPLIER_VALUES.AB_SEGMENT_RET_COND_ID.in(retCondIds))
                .fetch(AB_SEGMENT_MULTIPLIER_VALUES.HIERARCHICAL_MULTIPLIER_ID);
    }

    /**
     * Получить timeTarget на кампаниях.
     * Возвращается результат только если кампания найдена и на ней задан timeTarget и выключен автобюджет
     */
    public Map<Long, TimeTarget> getCampaignsTimeTargetWithoutAutobudget(int shard, Collection<Long> campaignIds) {
        final var campaignsData = campaignRepository.getCampaignsAutobudgetData(shard, campaignIds);

        return campaignsData
                .stream()
                .filter(item ->
                        item.timeTarget != null
                        && item.autobudget == CampaignsAutobudget.No
                )
                .collect(toMap(
                        item -> item.campaignId,
                        item -> item.timeTarget
                ));
    }

    /**
     * Получить ID групп по ID корректировок
     */
    public List<Long> getAdGroupIdsByHierarchicalMultiplierIds(int shard, Collection<Long> hierarchicalMultiplierIds) {
        return dslContextProvider.ppc(shard)
                .select(HIERARCHICAL_MULTIPLIERS.PID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(hierarchicalMultiplierIds))
                .and(HIERARCHICAL_MULTIPLIERS.PID.isNotNull())
                .fetch(HIERARCHICAL_MULTIPLIERS.PID);
    }

    /**
     * Получить ID кампаний по ID корректировок
     */
    public List<Long> getCampaignIdsByHierarchicalMultiplierIds(int shard, Collection<Long> hierarchicalMultiplierIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(HIERARCHICAL_MULTIPLIERS.CID)
                .from(HIERARCHICAL_MULTIPLIERS)
                .where(HIERARCHICAL_MULTIPLIERS.HIERARCHICAL_MULTIPLIER_ID.in(hierarchicalMultiplierIds))
                .and(HIERARCHICAL_MULTIPLIERS.PID.isNull())
                .fetch(HIERARCHICAL_MULTIPLIERS.CID);
    }

    public Map<Long, Boolean> getSmartTVEnableByCampaignIds(int shard, Collection<Long> campaignIds) {
        Map<Long, Long> cid2clientid= campaignRepository.getClientIdsForCids(shard, campaignIds);
        Map<ClientId, Boolean> clientId2enable = featureService.isEnabledForClientIdsOnlyFromDb(
                listToSet(mapList(cid2clientid.values(), ClientId::fromLong)), FeatureName.SMARTTV_BID_MODIFIER_ENABLED.getName());

        return EntryStream.of(cid2clientid)
                .collect(toMap(
                        Map.Entry::getKey,
                        item -> clientId2enable.get(ClientId.fromLong(item.getValue()))));

    }
}
