package ru.yandex.direct.core.bsexport.repository.bids

import org.jooq.Condition
import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.bsexport.model.BsExportAbstractBid
import ru.yandex.direct.core.bsexport.model.BsExportBidDynamic
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.entity.bids.repository.BidMappings
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbschema.ppc.tables.BidsDynamic.BIDS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.tables.DynamicConditions.DYNAMIC_CONDITIONS
import ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder
import ru.yandex.direct.jooqmapper.read.ReaderBuilders

@Repository
class BsExportBidsDynamicRepository(
    private val dslContextProvider: DslContextProvider
) : BsExportBidsRepositoryInterface<BsExportBidDynamic> {

    private val paramsReader = ReaderBuilders.fromField(DYNAMIC_CONDITIONS.CONDITION_NAME)
        .by { conditionName: String? ->
            mapOf(
                126 to conditionName,
            )
        }

    private val reader = JooqReaderWithSupplierBuilder.builder(::BsExportBidDynamic)
        .readProperty(BsExportBidDynamic.ID, ReaderBuilders.fromField(BIDS_DYNAMIC.DYN_ID))
        .readProperty(BsExportBidDynamic.AD_GROUP_ID, ReaderBuilders.fromField(BIDS_DYNAMIC.PID))
        .readProperty(BsExportBidDynamic.CAMPAIGN_ID, ReaderBuilders.fromField(PHRASES.CID))
        .readProperty(BsExportBidDynamic.ORDER_ID, ReaderBuilders.fromField(CAMPAIGNS.ORDER_ID))
        .readProperty(BsExportBidDynamic.CLIENT_ID, ReaderBuilders.fromField(CAMPAIGNS.CLIENT_ID))
        .readProperty(BsExportBidDynamic.CURRENCY, ReaderBuilders.fromField(CAMPAIGNS.CURRENCY)
            .by { Currencies.getCurrency(it.literal) })
        .readProperty(
            BsExportBidKeyword.CAMPAIGN_TYPE, ReaderBuilders.fromField(CAMPAIGNS.TYPE)
            .by { CampaignType.fromSource(it) })
        .readProperty(BsExportBidDynamic.DYN_COND_ID, ReaderBuilders.fromField(BIDS_DYNAMIC.DYN_COND_ID))
        .readProperty(BsExportBidDynamic.PRICE, ReaderBuilders.fromField(BIDS_DYNAMIC.PRICE))
        .readProperty(BsExportBidDynamic.PRICE_CONTEXT, ReaderBuilders.fromField(BIDS_DYNAMIC.PRICE_CONTEXT))
        .readProperty(BsExportBidDynamic.IS_SUSPENDED,
            ReaderBuilders.fromField(BIDS_DYNAMIC.OPTS)
                .by { BidMappings.isSuspendedFromDbOpts(it) })
        .readProperty(BsExportBidDynamic.PARAMS, paramsReader)
        .build()

    override fun getBidsByCids(
        shard: Int, cids: List<Long>,
        limit: Int, from: BsExportBidDynamic?
    ): List<BsExportBidDynamic> =
        dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(CAMPAIGNS)
            .straightJoin(PHRASES).on(PHRASES.CID.eq(CAMPAIGNS.CID))
            .straightJoin(BIDS_DYNAMIC).on(BIDS_DYNAMIC.PID.eq(PHRASES.PID))
            .leftJoin(DYNAMIC_CONDITIONS).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
            .where(
                CAMPAIGNS.CID.`in`(cids)
                    .and(
                        if (from == null) DSL.trueCondition()
                        else DSL.row(CAMPAIGNS.CID, PHRASES.PID, BIDS_DYNAMIC.DYN_ID)
                            .gt(from.campaignId, from.adGroupId, from.id)
                    )
            )
            .orderBy(CAMPAIGNS.CID, PHRASES.PID, BIDS_DYNAMIC.DYN_ID)
            .limit(limit)
            .fetch { reader.fromDb(it) }

    override fun getBids(shard: Int, ids: Collection<Long>): List<BsExportBidDynamic> =
        getBidsByCondition(shard, BIDS_DYNAMIC.DYN_ID.`in`(ids))

    fun getBidsByDynamicConditions(shard: Int, dynCondIds: Collection<Long>): List<BsExportAbstractBid> =
        getBidsByCondition(shard, BIDS_DYNAMIC.DYN_COND_ID.`in`(dynCondIds))

    private fun getBidsByCondition(shard: Int, condition: Condition): List<BsExportBidDynamic> =
        dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(BIDS_DYNAMIC)
            .join(PHRASES).on(PHRASES.PID.eq(BIDS_DYNAMIC.PID))
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
            .leftJoin(DYNAMIC_CONDITIONS).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
            .where(condition)
            .fetch { reader.fromDb(it) }

    override fun getCids(shard: Int, lastCampaignId: Long, bidsLimit: Int): List<Long> =
        dslContextProvider.ppc(shard)
            .selectDistinct(DSL.field(CAMPAIGNS.CID.name, Long::class.java))
            .from(
                DSL.select(CAMPAIGNS.CID)
                    .from(CAMPAIGNS)
                    .join(PHRASES).on(PHRASES.CID.eq(CAMPAIGNS.CID))
                    .join(BIDS_DYNAMIC).on(BIDS_DYNAMIC.PID.eq(PHRASES.PID))
                    .where(CAMPAIGNS.CID.gt(lastCampaignId))
                    .orderBy(CAMPAIGNS.CID)
                    .limit(bidsLimit)
            )
            .fetch { it.value1() }

    fun getIdsByDynamicConditionIds(shard: Int, dynamicConditionIds: Collection<Long>): Map<Long, List<Long>> =
        dslContextProvider.ppc(shard)
            .select(BIDS_DYNAMIC.DYN_ID)
            .from(BIDS_DYNAMIC)
            .where(BIDS_DYNAMIC.DYN_COND_ID.`in`(dynamicConditionIds))
            .fetchGroups(BIDS_DYNAMIC.DYN_COND_ID, BIDS_DYNAMIC.DYN_ID)

}
