package ru.yandex.direct.core.bsexport.repository.bids

import org.jooq.Condition
import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.bsexport.model.BidsStatusModerate
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbschema.ppc.tables.Bids.BIDS
import ru.yandex.direct.dbschema.ppc.tables.BidsHrefParams.BIDS_HREF_PARAMS
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder
import ru.yandex.direct.jooqmapper.read.ReaderBuilders

@Repository
class BsExportBidsKeywordRepository(
    private val dslContextProvider: DslContextProvider
) : BsExportBidsWithParamsRepositoryInterface<BsExportBidKeyword> {

    private val paramsReader = ReaderBuilders.fromFields(BIDS_HREF_PARAMS.PARAM1, BIDS_HREF_PARAMS.PARAM2)
        .by { param1: String?, param2: String? ->
            mapOf(
                1 to param1,
                2 to param2,
            )
        }

    private val reader = JooqReaderWithSupplierBuilder.builder(::BsExportBidKeyword)
        .readProperty(BsExportBidKeyword.ID, ReaderBuilders.fromField(BIDS.ID))
        .readProperty(BsExportBidKeyword.AD_GROUP_ID, ReaderBuilders.fromField(BIDS.PID))
        .readProperty(BsExportBidKeyword.CAMPAIGN_ID, ReaderBuilders.fromField(BIDS.CID))
        .readProperty(BsExportBidKeyword.ORDER_ID, ReaderBuilders.fromField(CAMPAIGNS.ORDER_ID))
        .readProperty(BsExportBidKeyword.CLIENT_ID, ReaderBuilders.fromField(CAMPAIGNS.CLIENT_ID))
        .readProperty(BsExportBidKeyword.CURRENCY, ReaderBuilders.fromField(CAMPAIGNS.CURRENCY)
            .by { Currencies.getCurrency(it.literal) })
        .readProperty(BsExportBidKeyword.CAMPAIGN_TYPE, ReaderBuilders.fromField(CAMPAIGNS.TYPE)
            .by { CampaignType.fromSource(it) })
        .readProperty(BsExportBidKeyword.PHRASE_BS_ID, ReaderBuilders.fromField(BIDS.PHRASE_ID)
            .by { it.toBigInteger() })
        .readProperty(BsExportBidKeyword.PRICE, ReaderBuilders.fromField(BIDS.PRICE))
        .readProperty(BsExportBidKeyword.PRICE_CONTEXT, ReaderBuilders.fromField(BIDS.PRICE_CONTEXT))
        .readProperty(BsExportBidKeyword.PHRASE, ReaderBuilders.fromField(BIDS.PHRASE))
        .readProperty(BsExportBidKeyword.STATUS_MODERATE,
            ReaderBuilders.fromField(BIDS.STATUS_MODERATE)
                .by { BidsStatusModerate.fromSource(it) })
        .readProperty(BsExportBidKeyword.SHOWS_FORECAST,
            ReaderBuilders.fromField(BIDS.SHOWS_FORECAST).by { l: Long? -> RepositoryUtils.intFromLong(l) })
        .readProperty(BsExportBidKeyword.IS_SUSPENDED,
            ReaderBuilders.fromField(BIDS.IS_SUSPENDED)
                .by { RepositoryUtils.booleanFromLong(it) })
        .readProperty(BsExportBidKeyword.PARAMS, paramsReader)
        .build()

    override fun getBids(shard: Int, ids: Collection<Long>): List<BsExportBidKeyword> =
        getBidsByCondition(shard, BIDS.ID.`in`(ids))

    override fun getBidsByHrefParams(
        shard: Int,
        keys: Collection<BsExportBidsWithParamsRepositoryInterface.HrefParamKey>,
    ): List<BsExportBidKeyword> {
        val bidIds = keys.map { it.id }
        val condition = BIDS.ID.`in`(bidIds)
        return getBidsByCondition(shard, condition)
    }

    private fun getBidsByCondition(shard: Int, condition: Condition): List<BsExportBidKeyword> =
        dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(BIDS)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS.CID))
            .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.ID.eq(BIDS.ID)
                .and(BIDS_HREF_PARAMS.CID.eq(BIDS.CID)))
            .where(condition)
            .fetch { reader.fromDb(it) }

    // копипаста запроса, он высоконагруженный и производетильность критична
    override fun getBidsByCids(
        shard: Int,
        cids: List<Long>,
        limit: Int,
        from: BsExportBidKeyword?
    ): List<BsExportBidKeyword> {
        val filteredCids = cids.filter { from == null || it >= from.campaignId }

        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(CAMPAIGNS)
            .straightJoin(BIDS).on(BIDS.CID.eq(CAMPAIGNS.CID))
            .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.ID.eq(BIDS.ID)
                .and(BIDS_HREF_PARAMS.CID.eq(BIDS.CID)))
            .where(
                CAMPAIGNS.CID.`in`(filteredCids)
                    .and(
                        if (from == null) DSL.trueCondition()
                        else DSL.row(CAMPAIGNS.CID, BIDS.ID)
                            .gt(from.campaignId, from.id)
                    )
            )
            .orderBy(CAMPAIGNS.CID, BIDS.ID)
            .limit(limit)
            .fetch { reader.fromDb(it) }
    }

    override fun getCids(shard: Int, lastCampaignId: Long, bidsLimit: Int): List<Long> =
        dslContextProvider.ppc(shard)
            .selectDistinct(DSL.field(CAMPAIGNS.CID.name, Long::class.java))
            .from(
                DSL.select(CAMPAIGNS.CID)
                    .from(CAMPAIGNS)
                    .join(BIDS).on(BIDS.CID.eq(CAMPAIGNS.CID))
                    .where(CAMPAIGNS.CID.gt(lastCampaignId))
                    .orderBy(CAMPAIGNS.CID)
                    .limit(bidsLimit)
            )
            .fetch { it.value1() }

}
