package ru.yandex.direct.core.bsexport.repository.bids

import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.bsexport.model.BsExportBidOfferRetargeting
import ru.yandex.direct.core.entity.bids.repository.BidMappings
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType
import ru.yandex.direct.dbschema.ppc.tables.BidsBase.BIDS_BASE
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder
import ru.yandex.direct.jooqmapper.read.ReaderBuilders

@Repository
class BsExportBidsOfferRetargetingRepository(
    private val dslContextProvider: DslContextProvider
) : BsExportBidsRepositoryInterface<BsExportBidOfferRetargeting> {
    private val reader = JooqReaderWithSupplierBuilder.builder(::BsExportBidOfferRetargeting)
        .readProperty(
            BsExportBidOfferRetargeting.ID,
            ReaderBuilders.fromField(BIDS_BASE.BID_ID)
        )
        .readProperty(BsExportBidOfferRetargeting.AD_GROUP_ID, ReaderBuilders.fromField(BIDS_BASE.PID))
        .readProperty(BsExportBidOfferRetargeting.CAMPAIGN_ID, ReaderBuilders.fromField(BIDS_BASE.CID))
        .readProperty(BsExportBidOfferRetargeting.ORDER_ID, ReaderBuilders.fromField(CAMPAIGNS.ORDER_ID))
        .readProperty(BsExportBidOfferRetargeting.CLIENT_ID, ReaderBuilders.fromField(CAMPAIGNS.CLIENT_ID))
        .readProperty(
            BsExportBidOfferRetargeting.CURRENCY, ReaderBuilders.fromField(CAMPAIGNS.CURRENCY)
                .by { Currencies.getCurrency(it.literal) })
        .readProperty(
            BsExportBidKeyword.CAMPAIGN_TYPE, ReaderBuilders.fromField(CAMPAIGNS.TYPE)
                .by { CampaignType.fromSource(it) })
        .readProperty(
            BsExportBidOfferRetargeting.CAMPAIGN_STRATEGY_DATA,
            ReaderBuilders.fromField(CAMPAIGNS.STRATEGY_DATA)
                .by { CampaignMappings.strategyDataFromDb(it) })
        .readProperty(
            BsExportBidOfferRetargeting.CAMPAIGN_STRATEGY_NAME,
            ReaderBuilders.fromField(CAMPAIGNS.STRATEGY_NAME)
                .by { StrategyName.fromSource(it) })
        // todo пока ставки не читаем, переосмыслим тут: https://st.yandex-team.ru/DIRECT-163737
        // сейчас для ТК ставку берем из стратегии кампании
        // .readProperty(BsExportBidOfferRetargeting.PRICE_CPA, ReaderBuilders.fromField(BIDS_BASE.PRICE))
        // .readProperty(BsExportBidOfferRetargeting.PRICE_CPC, ReaderBuilders.fromField(BIDS_BASE.PRICE_CONTEXT))
        .readProperty(
            BsExportBidOfferRetargeting.IS_SUSPENDED,
            ReaderBuilders.fromField(BIDS_BASE.OPTS)
                .by { BidMappings.isSuspendedFromDbOpts(it) })
        .readProperty(
            BsExportBidOfferRetargeting.IS_DELETED,
            ReaderBuilders.fromField(BIDS_BASE.OPTS)
                .by { BidMappings.isDeletedFromDbOpts(it) })
        // Для ТК в качестве названия фильтра используем имя кампании
        .readProperty(
            BsExportBidOfferRetargeting.PARAMS,
            ReaderBuilders.fromField(CAMPAIGNS.NAME)
                .by { name: String? ->
                    mapOf(126 to name)
                })
        .build()

    override fun getBidsByCids(
        shard: Int, cids: List<Long>, limit: Int,
        from: BsExportBidOfferRetargeting?
    ): List<BsExportBidOfferRetargeting> {
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(CAMPAIGNS)
            .straightJoin(BIDS_BASE).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
            .where(
                CAMPAIGNS.CID.`in`(cids)
                    .and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
                    .and(
                        if (from == null) DSL.trueCondition()
                        else DSL.row(CAMPAIGNS.CID, BIDS_BASE.PID, BIDS_BASE.BID_ID)
                            .gt(from.campaignId, from.adGroupId, from.id)
                    )
            )
            .orderBy(CAMPAIGNS.CID, BIDS_BASE.PID, BIDS_BASE.BID_ID)
            .limit(limit)
            .fetch { reader.fromDb(it) }
    }

    override fun getBids(shard: Int, ids: Collection<Long>): List<BsExportBidOfferRetargeting> {
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(BIDS_BASE)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
            .where(
                BIDS_BASE.BID_ID.`in`(ids)
                    .and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
            )
            .fetch { reader.fromDb(it) }
    }

    override fun getCids(shard: Int, lastCampaignId: Long, bidsLimit: Int): List<Long> =
        dslContextProvider.ppc(shard)
            .selectDistinct(DSL.field(CAMPAIGNS.CID.name, Long::class.java))
            .from(
                DSL.select(CAMPAIGNS.CID)
                    .from(CAMPAIGNS)
                    .join(BIDS_BASE).on(BIDS_BASE.CID.eq(CAMPAIGNS.CID))
                    .where(
                        CAMPAIGNS.CID.gt(lastCampaignId)
                            .and(BIDS_BASE.BID_TYPE.eq(BidsBaseBidType.offer_retargeting))
                    )
                    .orderBy(CAMPAIGNS.CID)
                    .limit(bidsLimit)
            )
            .fetch { it.value1() }
}
