package ru.yandex.direct.core.bsexport.repository.bids

import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.bsexport.model.BsExportBidPerformance
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbschema.ppc.tables.BidsPerformance.BIDS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder
import ru.yandex.direct.jooqmapper.read.ReaderBuilders

@Repository
class BsExportBidsPerformanceRepository(
    private val dslContextProvider: DslContextProvider
) : BsExportBidsRepositoryInterface<BsExportBidPerformance> {

    private val reader = JooqReaderWithSupplierBuilder.builder(::BsExportBidPerformance)
        .readProperty(
            BsExportBidPerformance.ID,
            ReaderBuilders.fromField(BIDS_PERFORMANCE.PERF_FILTER_ID)
        )
        .readProperty(BsExportBidPerformance.AD_GROUP_ID, ReaderBuilders.fromField(BIDS_PERFORMANCE.PID))
        .readProperty(BsExportBidPerformance.CAMPAIGN_ID, ReaderBuilders.fromField(PHRASES.CID))
        .readProperty(BsExportBidPerformance.ORDER_ID, ReaderBuilders.fromField(CAMPAIGNS.ORDER_ID))
        .readProperty(BsExportBidPerformance.CLIENT_ID, ReaderBuilders.fromField(CAMPAIGNS.CLIENT_ID))
        .readProperty(BsExportBidPerformance.CURRENCY, ReaderBuilders.fromField(CAMPAIGNS.CURRENCY)
            .by { Currencies.getCurrency(it.literal) })
        .readProperty(BsExportBidKeyword.CAMPAIGN_TYPE, ReaderBuilders.fromField(CAMPAIGNS.TYPE)
            .by { CampaignType.fromSource(it) })
        .readProperty(BsExportBidPerformance.CAMPAIGN_STRATEGY_DATA,
            ReaderBuilders.fromField(CAMPAIGNS.STRATEGY_DATA)
                .by { CampaignMappings.strategyDataFromDb(it) })
        .readProperty(BsExportBidPerformance.CAMPAIGN_STRATEGY_NAME,
            ReaderBuilders.fromField(CAMPAIGNS.STRATEGY_NAME)
                .by { StrategyName.fromSource(it) })
        .readProperty(BsExportBidPerformance.PRICE_CPC, ReaderBuilders.fromField(BIDS_PERFORMANCE.PRICE_CPC))
        .readProperty(BsExportBidPerformance.PRICE_CPA, ReaderBuilders.fromField(BIDS_PERFORMANCE.PRICE_CPA))
        .readProperty(BsExportBidPerformance.TARGET_FUNNEL,
            ReaderBuilders.fromField(BIDS_PERFORMANCE.TARGET_FUNNEL)
                .by { TargetFunnel.fromSource(it) })
        .readProperty(BsExportBidPerformance.IS_SUSPENDED,
            ReaderBuilders.fromField(BIDS_PERFORMANCE.IS_SUSPENDED)
                .by { RepositoryUtils.booleanFromLong(it) })
        .readProperty(BsExportBidPerformance.IS_DELETED,
            ReaderBuilders.fromField(BIDS_PERFORMANCE.IS_DELETED)
                .by { RepositoryUtils.booleanFromLong(it) })
        .readProperty(BsExportBidPerformance.PARAMS,
            ReaderBuilders.fromField(BIDS_PERFORMANCE.NAME)
                .by { name: String? ->
                    mapOf(126 to name)
                })
        .build()

    override fun getBidsByCids(
        shard: Int, cids: List<Long>, limit: Int,
        from: BsExportBidPerformance?
    ): List<BsExportBidPerformance> {
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(CAMPAIGNS)
            .straightJoin(PHRASES).on(CAMPAIGNS.CID.eq(PHRASES.CID))
            .straightJoin(BIDS_PERFORMANCE).on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID))
            .where(
                CAMPAIGNS.CID.`in`(cids)
                    .and(
                        if (from == null) DSL.trueCondition()
                        else DSL.row(CAMPAIGNS.CID, PHRASES.PID, BIDS_PERFORMANCE.PERF_FILTER_ID)
                            .gt(from.campaignId, from.adGroupId, from.id)
                    )
            )
            .orderBy(CAMPAIGNS.CID, PHRASES.PID, BIDS_PERFORMANCE.PERF_FILTER_ID)
            .limit(limit)
            .fetch { reader.fromDb(it) }
    }

    override fun getBids(shard: Int, ids: Collection<Long>): List<BsExportBidPerformance> {
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(BIDS_PERFORMANCE)
            .join(PHRASES).on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID))
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
            .where(BIDS_PERFORMANCE.PERF_FILTER_ID.`in`(ids))
            .fetch { reader.fromDb(it) }
    }

    override fun getCids(shard: Int, lastCampaignId: Long, bidsLimit: Int): List<Long> =
        dslContextProvider.ppc(shard)
            .selectDistinct(DSL.field(CAMPAIGNS.CID.name, Long::class.java))
            .from(
                DSL.select(CAMPAIGNS.CID)
                    .from(CAMPAIGNS)
                    .join(PHRASES).on(PHRASES.CID.eq(CAMPAIGNS.CID))
                    .join(BIDS_PERFORMANCE).on(BIDS_PERFORMANCE.PID.eq(PHRASES.PID))
                    .where(CAMPAIGNS.CID.gt(lastCampaignId))
                    .orderBy(CAMPAIGNS.CID)
                    .limit(bidsLimit)
            )
            .fetch { it.value1() }

}
