package ru.yandex.direct.core.bsexport.repository.bids

import org.jooq.Condition
import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.bsexport.model.BsExportRelevanceMatch
import ru.yandex.direct.core.entity.bids.repository.BidMappings
import ru.yandex.direct.core.entity.bids.service.BidBaseOpt
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.relevancematch.repository.RelevanceMatchMapping
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbschema.ppc.enums.BidsBaseBidType
import ru.yandex.direct.dbschema.ppc.tables.BidsBase.BIDS_BASE
import ru.yandex.direct.dbschema.ppc.tables.BidsHrefParams.BIDS_HREF_PARAMS
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbutil.SqlUtils
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder
import ru.yandex.direct.jooqmapper.read.ReaderBuilders

@Repository
class BsExportBidsRelevanceMatchRepository(
    private val dslContextProvider: DslContextProvider
) : BsExportBidsWithParamsRepositoryInterface<BsExportRelevanceMatch> {

    companion object {
        private val BID_TYPES = listOf(
            BidsBaseBidType.relevance_match,
            BidsBaseBidType.relevance_match_search
        )
    }

    private val paramsReader = ReaderBuilders.fromFields(BIDS_HREF_PARAMS.PARAM1, BIDS_HREF_PARAMS.PARAM2)
        .by { param1: String?, param2: String? ->
            mapOf(
                1 to param1,
                2 to param2,
            )
        }

    private val reader = JooqReaderWithSupplierBuilder.builder(::BsExportRelevanceMatch)
        .readProperty(BsExportRelevanceMatch.ID, ReaderBuilders.fromField(BIDS_BASE.BID_ID))
        .readProperty(BsExportRelevanceMatch.AD_GROUP_ID, ReaderBuilders.fromField(BIDS_BASE.PID))
        .readProperty(BsExportRelevanceMatch.CAMPAIGN_ID, ReaderBuilders.fromField(BIDS_BASE.CID))
        .readProperty(BsExportRelevanceMatch.ORDER_ID, ReaderBuilders.fromField(CAMPAIGNS.ORDER_ID))
        .readProperty(BsExportRelevanceMatch.CLIENT_ID, ReaderBuilders.fromField(CAMPAIGNS.CLIENT_ID))
        .readProperty(BsExportRelevanceMatch.CURRENCY, ReaderBuilders.fromField(CAMPAIGNS.CURRENCY)
            .by { Currencies.getCurrency(it.literal) })
        .readProperty(BsExportBidKeyword.CAMPAIGN_TYPE, ReaderBuilders.fromField(CAMPAIGNS.TYPE)
            .by { CampaignType.fromSource(it) })
        .readProperty(BsExportRelevanceMatch.PRICE, ReaderBuilders.fromField(BIDS_BASE.PRICE))
        .readProperty(
            BsExportRelevanceMatch.PRICE_CONTEXT,
            ReaderBuilders.fromField(BIDS_BASE.PRICE_CONTEXT)
        )
        .readProperty(BsExportRelevanceMatch.IS_SUSPENDED,
            ReaderBuilders.fromField(BIDS_BASE.OPTS)
                .by { BidMappings.isSuspendedFromDbOpts(it) })
        .readProperty(BsExportRelevanceMatch.IS_DELETED,
            ReaderBuilders.fromField(BIDS_BASE.OPTS)
                .by { BidMappings.isDeletedFromDbOpts(it) })
        .readProperty(BsExportRelevanceMatch.RELEVANCE_MATCH_CATEGORIES, ReaderBuilders.fromField(BIDS_BASE.RELEVANCE_MATCH_CATEGORIES)
            .by { RelevanceMatchMapping.relevanceMatchCategoriesFromDbFormat(it) })
        .readProperty(BsExportRelevanceMatch.PARAMS, paramsReader)
        .build()

    override fun getBidsByCids(
        shard: Int, cids: List<Long>, limit: Int,
        from: BsExportRelevanceMatch?
    ): List<BsExportRelevanceMatch> {
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(CAMPAIGNS)
            .straightJoin(BIDS_BASE).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
            .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.ID.eq(BIDS_BASE.BID_ID)
                .and(BIDS_HREF_PARAMS.CID.eq(BIDS_BASE.CID)))
            .where(
                CAMPAIGNS.CID.`in`(cids)
                    .and(BIDS_BASE.BID_TYPE.`in`(BID_TYPES)) // TODO
                    .and(
                        if (from == null) DSL.trueCondition()
                        else DSL.row(CAMPAIGNS.CID, BIDS_BASE.BID_ID)
                            .gt(from.campaignId, from.id)
                    )
                    .and(
                        SqlUtils.findInSet(BidBaseOpt.DELETED.typedValue, BIDS_BASE.OPTS)
                            .eq(RepositoryUtils.booleanToLong(false))
                    )
            )
            .orderBy(CAMPAIGNS.CID, BIDS_BASE.BID_ID)
            .limit(limit)
            .fetch { reader.fromDb(it) }
    }

    override fun getBids(shard: Int, ids: Collection<Long>): List<BsExportRelevanceMatch> =
        getBidsByCondition(shard, BIDS_BASE.BID_ID.`in`(ids))

    override fun getBidsByHrefParams(
        shard: Int,
        keys: Collection<BsExportBidsWithParamsRepositoryInterface.HrefParamKey>,
    ): List<BsExportRelevanceMatch> {
        val bidIds = keys.map { it.id }
        val condition = BIDS_BASE.BID_ID.`in`(bidIds)
        return getBidsByCondition(shard, condition)
    }

    private fun getBidsByCondition(shard: Int, condition: Condition) =
        dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(BIDS_BASE)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_BASE.CID))
            .leftJoin(BIDS_HREF_PARAMS).on(BIDS_HREF_PARAMS.ID.eq(BIDS_BASE.BID_ID)
                .and(BIDS_HREF_PARAMS.CID.eq(BIDS_BASE.CID)))
            .where(condition.and(BIDS_BASE.BID_TYPE.`in`(BID_TYPES)))
            .fetch { reader.fromDb(it) }

    override fun getCids(shard: Int, lastCampaignId: Long, bidsLimit: Int): List<Long> =
        dslContextProvider.ppc(shard)
            .selectDistinct(DSL.field(CAMPAIGNS.CID.name, Long::class.java))
            .from(
                DSL.select(CAMPAIGNS.CID)
                    .from(CAMPAIGNS)
                    .join(BIDS_BASE).on(BIDS_BASE.CID.eq(CAMPAIGNS.CID))
                    .where(
                        CAMPAIGNS.CID.gt(lastCampaignId)
                            .and(BIDS_BASE.BID_TYPE.`in`(BID_TYPES))
                    )
                    .orderBy(CAMPAIGNS.CID)
                    .limit(bidsLimit)
            )
            .fetch { it.value1() }

}
