package ru.yandex.direct.core.bsexport.repository.bids

import java.math.BigInteger
import org.jooq.impl.DSL
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.bsexport.model.BsExportBidKeyword
import ru.yandex.direct.core.bsexport.model.BsExportBidRetargeting
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.retargeting.model.TargetingCategory
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionMappings
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingConditionRepository
import ru.yandex.direct.core.entity.retargeting.repository.TargetingCategoriesCache
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.dbschema.ppc.tables.BidsRetargeting.BIDS_RETARGETING
import ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS
import ru.yandex.direct.dbschema.ppc.tables.RetargetingConditions.RETARGETING_CONDITIONS
import ru.yandex.direct.dbschema.ppc.tables.RetargetingGoals.RETARGETING_GOALS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplierBuilder
import ru.yandex.direct.jooqmapper.read.ReaderBuilders

@Repository
class BsExportBidsRetargetingRepository(
    private val dslContextProvider: DslContextProvider,
    private val targetingCategoriesCache: TargetingCategoriesCache,
) : BsExportBidsRepositoryInterface<BsExportBidRetargeting> {

    private fun createBidWithParamsReader(): JooqReaderWithSupplier<BsExportBidRetargeting> {
        val targetingCategories = targetingCategoriesCache
            .targetingCategories
            .associateBy { it.importId }

        val paramsReader = ReaderBuilders
            .fromFields(
                BIDS_RETARGETING.RET_ID,
                RETARGETING_CONDITIONS.PROPERTIES,
                RETARGETING_CONDITIONS.CONDITION_JSON,
            )
            .by { retId, properties: String?, conditionJson: String? ->
                mapRowToHrefParams(targetingCategories, retId, properties, conditionJson)
            }

        return commonFieldsReaderBuilder()
            .readProperty(BsExportBidRetargeting.PARAMS, paramsReader)
            .build()
    }

    private fun mapRowToHrefParams(
        targetingCategories: Map<BigInteger, TargetingCategory>,
        retId: Long,
        properties: String?,
        conditionJson: String?,
    ): Map<Int, String?> {
        if (properties != null && RetargetingConditionRepository.PROPERTY_INTEREST in properties) {
            val rules = RetargetingConditionMappings.rulesFromJson(conditionJson)
            val goals = rules.flatMap { it.goals }
            for (goal in goals) {
                val goalId = goal.id.toBigInteger()
                val category = targetingCategories[goalId]
                if (category != null) {
                    return mapOf(
                        125 to goalId.toString(),
                        126 to category.name,
                    )
                }
            }
        }

        return mapOf(126 to retId.toString())
    }

    override fun getBidsByCids(
        shard: Int, cids: List<Long>,
        limit: Int, from: BsExportBidRetargeting?
    ): List<BsExportBidRetargeting> {
        val reader = createBidWithParamsReader()
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(CAMPAIGNS)
            .straightJoin(BIDS_RETARGETING).on(CAMPAIGNS.CID.eq(BIDS_RETARGETING.CID))
            .leftJoin(RETARGETING_CONDITIONS).on(
                RETARGETING_CONDITIONS.RET_COND_ID.eq(BIDS_RETARGETING.RET_COND_ID)
            )
            .where(CAMPAIGNS.CID.`in`(cids))
            .and(
                if (from == null) DSL.trueCondition()
                else DSL.row(CAMPAIGNS.CID, BIDS_RETARGETING.RET_ID)
                    .gt(from.campaignId, from.id)
            )
            .orderBy(CAMPAIGNS.CID, BIDS_RETARGETING.RET_ID)
            .limit(limit)
            .fetch { reader.fromDb(it) }
    }

    fun getBidsByPids(
        shard: Int, pids: Collection<Long>
    ): List<BsExportBidRetargeting> {
        val reader = commonFieldsReaderBuilder().build()
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(BIDS_RETARGETING)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_RETARGETING.CID))
            .where(BIDS_RETARGETING.PID.`in`(pids))
            .fetch { reader.fromDb(it) }
    }

    override fun getBids(shard: Int, ids: Collection<Long>): List<BsExportBidRetargeting> {
        val reader = createBidWithParamsReader()
        return dslContextProvider.ppc(shard)
            .select(reader.fieldsToRead)
            .from(BIDS_RETARGETING)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_RETARGETING.CID))
            .leftJoin(RETARGETING_CONDITIONS).on(
                RETARGETING_CONDITIONS.RET_COND_ID.eq(BIDS_RETARGETING.RET_COND_ID)
            )
            .where(BIDS_RETARGETING.RET_ID.`in`(ids))
            .fetch { createBidWithParamsReader().fromDb(it) }
    }

    /**
     * Лениво загружает ставки для данных условий ретаргетинга
     */
    fun getBidsByRetargetingConditions(
        shard: Int,
        retCondIds: Collection<Long>,
    ): Sequence<List<BsExportBidRetargeting>> = sequence {
        var lastBid: BsExportBidRetargeting? = null
        while (true) {
            val chunk = getChunkedBidsByRetargetingCondition(shard, retCondIds, lastBid)
            if (chunk.isEmpty()) {
                break
            }

            yield(chunk)
            lastBid = chunk.last()
        }
    }

    private fun getChunkedBidsByRetargetingCondition(
        shard: Int,
        retCondIds: Collection<Long>,
        from: BsExportBidRetargeting?,
    ): List<BsExportBidRetargeting> {
        val bidWithParamsReader = createBidWithParamsReader()
        return dslContextProvider.ppc(shard)
            .select(bidWithParamsReader.fieldsToRead)
            .from(BIDS_RETARGETING)
            .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BIDS_RETARGETING.CID))
            .leftJoin(RETARGETING_CONDITIONS).on(
                RETARGETING_CONDITIONS.RET_COND_ID.eq(BIDS_RETARGETING.RET_COND_ID)
            )
            .where(BIDS_RETARGETING.RET_COND_ID.`in`(retCondIds)
                .and(
                    when (from) {
                        null -> DSL.trueCondition()
                        else -> DSL.row(BIDS_RETARGETING.RET_ID).gt(from.id)
                    }
                ))
            .orderBy(BIDS_RETARGETING.RET_ID)
            .limit(BIDS_BY_RETARGETING_CONDITION_CHUNK_SIZE)
            .fetch { bidWithParamsReader.fromDb(it) }
    }

    override fun getCids(shard: Int, lastCampaignId: Long, bidsLimit: Int): List<Long> =
        dslContextProvider.ppc(shard)
            .selectDistinct(DSL.field(CAMPAIGNS.CID.name, Long::class.java))
            .from(
                DSL.select(CAMPAIGNS.CID)
                    .from(CAMPAIGNS)
                    .join(BIDS_RETARGETING).on(BIDS_RETARGETING.CID.eq(CAMPAIGNS.CID))
                    .where(CAMPAIGNS.CID.gt(lastCampaignId))
                    .orderBy(CAMPAIGNS.CID)
                    .limit(bidsLimit)
            )
            .fetch { it.value1() }

    companion object {
        private const val BIDS_BY_RETARGETING_CONDITION_CHUNK_SIZE: Int = 10_000

        private val IS_ACCESSIBLE_FIELD =
            DSL.select(DSL.case_(DSL.count()).`when`(0, 1L).else_(0L))
                .from(RETARGETING_GOALS)
                .where(
                    RETARGETING_GOALS.RET_COND_ID.eq(BIDS_RETARGETING.RET_COND_ID)
                        .and(RETARGETING_GOALS.IS_ACCESSIBLE.eq(0))
                )
                .asField<Long>("is_accessible")

        private fun commonFieldsReaderBuilder() =
            JooqReaderWithSupplierBuilder.builder { BsExportBidRetargeting() }
                .readProperty(BsExportBidRetargeting.ID, ReaderBuilders.fromField(BIDS_RETARGETING.RET_ID))
                .readProperty(BsExportBidRetargeting.RET_COND_ID, ReaderBuilders.fromField(BIDS_RETARGETING.RET_COND_ID))
                .readProperty(BsExportBidRetargeting.AD_GROUP_ID, ReaderBuilders.fromField(BIDS_RETARGETING.PID))
                .readProperty(BsExportBidRetargeting.CAMPAIGN_ID, ReaderBuilders.fromField(BIDS_RETARGETING.CID))
                .readProperty(BsExportBidRetargeting.ORDER_ID, ReaderBuilders.fromField(CAMPAIGNS.ORDER_ID))
                .readProperty(BsExportBidKeyword.CLIENT_ID, ReaderBuilders.fromField(CAMPAIGNS.CLIENT_ID))
                .readProperty(BsExportBidKeyword.CURRENCY, ReaderBuilders.fromField(CAMPAIGNS.CURRENCY)
                    .by { Currencies.getCurrency(it.literal) })
                .readProperty(BsExportBidKeyword.CAMPAIGN_TYPE, ReaderBuilders.fromField(CAMPAIGNS.TYPE)
                    .by { CampaignType.fromSource(it) })
                .readProperty(BsExportBidRetargeting.PRICE_CONTEXT, ReaderBuilders.fromField(BIDS_RETARGETING.PRICE_CONTEXT))
                .readProperty(BsExportBidRetargeting.IS_SUSPENDED,
                    ReaderBuilders.fromField(BIDS_RETARGETING.IS_SUSPENDED)
                        .by { RepositoryUtils.booleanFromLong(it) })
                .readProperty(BsExportBidRetargeting.IS_ACCESSIBLE,
                    ReaderBuilders.fromField(IS_ACCESSIBLE_FIELD)
                        .by { RepositoryUtils.booleanFromLong(it) })
    }
}
