package ru.yandex.direct.core.bsexport.repository.resources

import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository
import ru.yandex.direct.dbschema.ppc.tables.AdgroupsDynamic.ADGROUPS_DYNAMIC
import ru.yandex.direct.dbschema.ppc.tables.AdgroupsPerformance.ADGROUPS_PERFORMANCE
import ru.yandex.direct.dbschema.ppc.tables.AdgroupsText.ADGROUPS_TEXT
import ru.yandex.direct.dbschema.ppc.tables.Feeds.FEEDS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider

@Repository
open class BsExportFeedInfoRepository(
    private val dslContextProvider: DslContextProvider,
    private val bannerRelationsRepository: BannerRelationsRepository
) {
    fun getFeedInfoForDynamicFeedAdGroups(shard: Int, adGroupIds: Collection<Long>): List<AdGroupFeedInfo> {
        if (adGroupIds.isEmpty()) {
            return listOf()
        }
        return dslContextProvider.ppc(shard)
            .select(
                ADGROUPS_DYNAMIC.PID, FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID,
                FEEDS.MARKET_FEED_ID, FEEDS.FEED_ID
            )
            .from(ADGROUPS_DYNAMIC)
            .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_DYNAMIC.FEED_ID))
            .where(ADGROUPS_DYNAMIC.PID.`in`(adGroupIds))
            .and(FEEDS.MARKET_BUSINESS_ID.isNotNull)
            .and(FEEDS.MARKET_SHOP_ID.isNotNull)
            .and(FEEDS.MARKET_FEED_ID.isNotNull)
            .fetch {
                AdGroupFeedInfo(
                    adGroupId = it.get(ADGROUPS_DYNAMIC.PID)!!,
                    marketBusinessId = it.get(FEEDS.MARKET_BUSINESS_ID)!!,
                    marketShopId = it.get(FEEDS.MARKET_SHOP_ID)!!,
                    marketFeedId = it.get(FEEDS.MARKET_FEED_ID)!!,
                    directFeedId = it.get(FEEDS.FEED_ID)!!
                )
            }
    }

    fun getFeedInfoForPerformanceAdGroups(shard: Int, adGroupIds: Collection<Long>): List<AdGroupFeedInfo> {
        if (adGroupIds.isEmpty()) {
            return listOf()
        }
        return dslContextProvider.ppc(shard)
            .select(
                ADGROUPS_PERFORMANCE.PID, FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID,
                FEEDS.MARKET_FEED_ID, FEEDS.FEED_ID
            )
            .from(ADGROUPS_PERFORMANCE)
            .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_PERFORMANCE.FEED_ID))
            .where(ADGROUPS_PERFORMANCE.PID.`in`(adGroupIds))
            .and(FEEDS.MARKET_BUSINESS_ID.isNotNull)
            .and(FEEDS.MARKET_SHOP_ID.isNotNull)
            .and(FEEDS.MARKET_FEED_ID.isNotNull)
            .fetch {
                AdGroupFeedInfo(
                    adGroupId = it.get(ADGROUPS_PERFORMANCE.PID)!!,
                    marketBusinessId = it.get(FEEDS.MARKET_BUSINESS_ID)!!,
                    marketShopId = it.get(FEEDS.MARKET_SHOP_ID)!!,
                    marketFeedId = it.get(FEEDS.MARKET_FEED_ID)!!,
                    directFeedId = it.get(FEEDS.FEED_ID)!!,
                )
            }
    }

    fun getFeedInfoForTextAdGroups(shard: Int, adGroupIds: Collection<Long>): List<AdGroupFeedInfo> {
        if (adGroupIds.isEmpty()) {
            return listOf()
        }
        return dslContextProvider.ppc(shard)
            .select(
                ADGROUPS_TEXT.PID, FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID,
                FEEDS.MARKET_FEED_ID, FEEDS.FEED_ID
            )
            .from(ADGROUPS_TEXT)
            .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_TEXT.FEED_ID))
            .where(ADGROUPS_TEXT.PID.`in`(adGroupIds))
            .and(FEEDS.MARKET_BUSINESS_ID.isNotNull)
            .and(FEEDS.MARKET_SHOP_ID.isNotNull)
            .and(FEEDS.MARKET_FEED_ID.isNotNull)
            .fetch {
                AdGroupFeedInfo(
                    adGroupId = it.get(ADGROUPS_TEXT.PID)!!,
                    marketBusinessId = it.get(FEEDS.MARKET_BUSINESS_ID)!!,
                    marketShopId = it.get(FEEDS.MARKET_SHOP_ID)!!,
                    marketFeedId = it.get(FEEDS.MARKET_FEED_ID)!!,
                    directFeedId = it.get(FEEDS.FEED_ID)!!,
                )
            }
    }

    fun getBannerIdsByAdGroupIds(shard: Int, adGroupIds: Collection<Long>): List<Long> {
        if (adGroupIds.isEmpty()) {
            return listOf()
        }
        return bannerRelationsRepository.getBannerIdsByAdGroupIds(shard, adGroupIds)
    }

    fun getAdGroupIdsByFeedIds(shard: Int, feedIds: Collection<Long>): List<Long> {
        if (feedIds.isEmpty()) {
            return listOf()
        }
        val textAdGroups = dslContextProvider.ppc(shard)
            .select(ADGROUPS_TEXT.PID)
            .from(ADGROUPS_TEXT)
            .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_TEXT.FEED_ID))
            .where(FEEDS.FEED_ID.`in`(feedIds))
        val dynamicAdGroups = dslContextProvider.ppc(shard)
            .select(ADGROUPS_DYNAMIC.PID)
            .from(ADGROUPS_DYNAMIC)
            .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_DYNAMIC.FEED_ID))
            .where(FEEDS.FEED_ID.`in`(feedIds))
        val performanceAdGroups = dslContextProvider.ppc(shard)
            .select(ADGROUPS_PERFORMANCE.PID)
            .from(ADGROUPS_PERFORMANCE)
            .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_PERFORMANCE.FEED_ID))
            .where(FEEDS.FEED_ID.`in`(feedIds))
        return textAdGroups.unionAll(dynamicAdGroups).unionAll(performanceAdGroups)
            .fetch(ADGROUPS_TEXT.PID)
    }
}

data class AdGroupFeedInfo(
    val adGroupId: Long,
    val marketBusinessId: Long,
    val marketShopId: Long,
    val marketFeedId: Long,
    val directFeedId: Long,
)
