package ru.yandex.direct.core.configuration;

import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Arrays;
import java.util.OptionalInt;

import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.AnonymousAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.google.common.base.Strings;
import com.google.common.primitives.Ints;
import kotlin.TuplesKt;
import org.apache.commons.io.IOUtils;
import org.asynchttpclient.AsyncHttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.core.io.Resource;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.web.context.WebApplicationContext;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.autobudget.restart.AutobudgetRestartConfiguration;
import ru.yandex.direct.canvas.client.CanvasClient;
import ru.yandex.direct.canvas.client.CanvasClientConfiguration;
import ru.yandex.direct.common.configuration.CommonConfiguration;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.mds.MdsHolder;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.bids.interpolator.CapFactory;
import ru.yandex.direct.core.entity.campaign.container.AffectedCampaignIdsContainer;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.RequestCampaignAccessibilityCheckerProvider;
import ru.yandex.direct.core.entity.freelancer.service.AvatarsConfigNameConverter;
import ru.yandex.direct.core.entity.keyword.service.KeywordRecentStatisticsProvider;
import ru.yandex.direct.core.entity.notification.NotificationService;
import ru.yandex.direct.core.entity.notification.NotificationServiceDummyImpl;
import ru.yandex.direct.core.entity.notification.NotificationServiceIntapiImpl;
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator;
import ru.yandex.direct.core.entity.statistics.repository.OrderStatDayRepository;
import ru.yandex.direct.core.entity.statistics.repository.OrderStatDaySandboxRepository;
import ru.yandex.direct.core.entity.xiva.XivaPushesQueueService;
import ru.yandex.direct.core.entity.xiva.repository.XivaPushesQueueRepository;
import ru.yandex.direct.core.grut.api.DefaultGrutApiProperties;
import ru.yandex.direct.core.grut.api.GrutApiProperties;
import ru.yandex.direct.core.service.grut.DcTranslator;
import ru.yandex.direct.core.util.GrutTraceCallback;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbqueue.repository.DbQueueTypeMap;
import ru.yandex.direct.dbqueue.service.DbQueueService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.dssclient.DssClient;
import ru.yandex.direct.dssclient.DssClientCredentials;
import ru.yandex.direct.dssclient.DssUserCredentials;
import ru.yandex.direct.dssclient.http.HttpConnectorSettings;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.gemini.GeminiClient;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.geobasehelper.GeoBaseHttpApiHelper;
import ru.yandex.direct.geosearch.GeosearchClient;
import ru.yandex.direct.geosearch.GeosearchClientSettings;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.intapi.client.IntApiClientConfiguration;
import ru.yandex.direct.integrations.configuration.IntegrationsConfiguration;
import ru.yandex.direct.libs.keywordutils.helper.ClemmerCache;
import ru.yandex.direct.libs.keywordutils.helper.ParseKeywordCache;
import ru.yandex.direct.libs.keywordutils.helper.SingleKeywordsCache;
import ru.yandex.direct.libs.keywordutils.inclusion.model.KeywordWithLemmasFactory;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean;
import ru.yandex.direct.mail.LoggingMailSender;
import ru.yandex.direct.mail.MailSender;
import ru.yandex.direct.mail.RedirectingMailSender;
import ru.yandex.direct.mail.SmtpMailSender;
import ru.yandex.direct.mail.YServiceTokenCreator;
import ru.yandex.direct.pdfgen.PdfBuilder;
import ru.yandex.direct.queryrec.LanguageRecognizer;
import ru.yandex.direct.queryrec.QueryrecJni;
import ru.yandex.direct.queryrec.QueryrecService;
import ru.yandex.direct.queryrec.UzbekLanguageThresholds;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.turboapps.client.TurboAppsClient;
import ru.yandex.direct.turbolandings.client.TurboLandingsClient;
import ru.yandex.direct.turbolandings.client.TurboLandingsClientConfiguration;
import ru.yandex.direct.turbopages.client.TurbopagesClient;
import ru.yandex.direct.turbopages.client.TurbopagesClientConfiguration;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.direct.ugcdb.client.UgcDbClient;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.crypt.Encrypter;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.direct.ytcomponents.service.OrderStatDayDynContextProvider;
import ru.yandex.direct.ytcomponents.spring.YtComponentsConfiguration;
import ru.yandex.grut.client.GrutClient;
import ru.yandex.grut.client.GrutGrpcClient;
import ru.yandex.grut.client.ServiceHolder;
import ru.yandex.grut.client.yp_service_discovery.LivenessCheckParams;
import ru.yandex.grut.client.yp_service_discovery.YpServiceHolder;
import ru.yandex.grut.client.yp_service_discovery.YpServiceHolderWithFallback;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.configuration.CommonConfiguration.LIVE_CONFIG_CHECK_RATE;
import static ru.yandex.direct.common.db.PpcPropertyNames.CLIENTS_WITH_DEFAULT_VIE_LANGUAGE;
import static ru.yandex.direct.common.db.PpcPropertyNames.CLIENTS_WITH_ENABLED_UZBEK_LANGUAGE;
import static ru.yandex.direct.common.db.PpcPropertyNames.CYRILLIC_UZBEK_TO_RUSSIAN_THRESHOLD;
import static ru.yandex.direct.common.db.PpcPropertyNames.LATIN_UZBEK_TO_RUSSIAN_THRESHOLD;
import static ru.yandex.direct.common.db.PpcPropertyNames.RECOGNIZED_BANNER_LANGUAGES_SET;
import static ru.yandex.direct.common.db.PpcPropertyNames.UZBEK_TO_TURKISH_THRESHOLD;
import static ru.yandex.direct.common.db.PpcPropertyNames.UZBEK_TO_UNKNOWN_THRESHOLD;
import static ru.yandex.direct.config.EssentialConfiguration.CONFIG_SCHEDULER_BEAN_NAME;
import static ru.yandex.direct.solomon.SolomonUtils.GRUT_METRICS_REGISTRY;

@Configuration
@Import({
        AuthorizationConfiguration.class,
        CommonConfiguration.class,
        CoreIntegrationConfiguration.class,
        IntegrationsConfiguration.class,
        YtComponentsConfiguration.class,
        AutobudgetRestartConfiguration.class,
})
@ComponentScan(
        basePackages = "ru.yandex.direct.core",
        excludeFilters = {
                @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION),
        }
)
public class CoreConfiguration {

    public static final String TURBO_APPS_CLIENT = "turboAppsClient";
    public static final String GEOSEARCH_CLIENT = "geosearchClient";
    public static final String TRUSTED_COUNTER_DOMAINS = "trustedCounterDomains";
    public static final String TRUSTED_MOBILE_COUNTER_DOMAINS = "trustedMobileCounterDomains";
    public static final String DIRECT_FILES_MDS_HOLDER_BEAN_NAME = "directFilesMds";
    public static final String ACCESSIBLE_CAMPAIGN_CHEKER_PROVIDER = "accessibleCampaignCheckerProvider";
    public static final String GEO_BASE_HELPER = "geoBaseHelper";
    public static final String ROBOT_DIRECT_DAAS_STARTREK_TOKEN = "robotDirectDaasStartrekToken";
    public static final String DEFAULT_COPY_RECENT_STATISTICS_PROVIDER_BEAN_NAME =
            "defaultCopyRecentStatisticsProvider";
    public static final String GRUT_CLIENT_FOR_WATCHLOG = "grutClientForWatchlog";
    public static final String SERVICE_HOLDER_FOR_WATCHLOG = "serviceHolderForWatchlog";
    public static final String SERVICE_HOLDER = "serviceHolder";
    public static final String DIRECT_CONFIG_SERVICE_HOLDER = "service_holder";
    public static final String CONVERSION_CENTER_ENCRYPTER_BEAN_NAME = "ConversionCenterEncrypter";

    @Bean
    public ClemmerCache clemmerCache(DirectConfig config) {
        int cacheSize = config.getInt("clemmer.cache_size");
        return new ClemmerCache(cacheSize);
    }

    @Bean
    public ParseKeywordCache parseKeywordCache(DirectConfig config) {
        OptionalInt cacheSize = config.findInt("parsekeyword.cache_size");
        return cacheSize.isPresent() ? new ParseKeywordCache(cacheSize.getAsInt()) : new ParseKeywordCache();
    }

    @Bean
    public SingleKeywordsCache singleKeywordsCache(ClemmerCache clemmerCache, DirectConfig config) {
        OptionalInt cacheSize = config.findInt("singlekeywords.cache_size");
        return cacheSize.isPresent() ? new SingleKeywordsCache(clemmerCache, cacheSize.getAsInt()) :
                new SingleKeywordsCache(clemmerCache);
    }

    @Bean
    @Autowired
    @Lazy
    public KeywordWithLemmasFactory keywordWithLemmasFactory(SingleKeywordsCache singleKeywordsCache,
                                                             ParseKeywordCache parseKeywordCache) {
        return new KeywordWithLemmasFactory(singleKeywordsCache, parseKeywordCache);
    }

    @Bean
    @Lazy
    public RegionIdsValidator regionIdsValidator() {
        return new RegionIdsValidator();
    }

    @Bean
    @Lazy
    IntApiClient intApiClient(@Value("${direct_intapi.url}") String url,
                              @Value("${direct_intapi.tvm_app_id}") int tvmAppId,
                              AsyncHttpClient asyncHttpClient,
                              TvmIntegration tvmIntegration) throws URISyntaxException {
        TvmService dstIntapiService = TvmService.fromIdStrict(tvmAppId);
        return new IntApiClient(new IntApiClientConfiguration(url), asyncHttpClient,
                () -> tvmIntegration.getTicket(dstIntapiService));
    }


    @Bean
    @Lazy
    UgcDbClient ugcDbClient(@Value("${ugc_db.url}") String url,
                            @Value("${ugc_db.tvm_app_id}") int tvmAppId,
                            DirectConfig directConfig,
                            AsyncHttpClient asyncHttpClient,
                            TvmIntegration tvmIntegration) {
        TvmService tvmService = TvmService.fromIdStrict(tvmAppId);
        DirectConfig ugcDbConfig = directConfig.getBranch("ugc_db");
        FetcherSettings fetcherSettings = new FetcherSettings()
                .withRequestTimeout(ugcDbConfig.getDuration("timeout"))
                .withRequestRetries(ugcDbConfig.getInt("requestRetries"));
        ParallelFetcherFactory pff = new ParallelFetcherFactory(asyncHttpClient, fetcherSettings);
        return new UgcDbClient(url, tvmService, pff, tvmIntegration);
    }

    @Bean
    @Lazy
    TurboLandingsClient turboLandingsClient(
            @Value("${turbo_landings.url}") String url,
            @Value("${turbo_landings.token}") String token,
            @Value("${turbo_landings.file}") String fileWithToken,
            AsyncHttpClient asyncHttpClient
    ) {
        return new TurboLandingsClient(new TurboLandingsClientConfiguration(url, token, fileWithToken),
                new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()));
    }

    @Bean
    @Lazy
    TurbopagesClient turbopagesClient(
            @Value("${turbopages.url}") String url,
            @Value("${turbopages.header}") String header,
            AsyncHttpClient asyncHttpClient
    ) {
        return new TurbopagesClient(new TurbopagesClientConfiguration(url, header),
                new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()));
    }

    @Bean(TURBO_APPS_CLIENT)
    @Lazy
    TurboAppsClient turboappsClient(
            @Value("${turboapps.url}") String url,
            @Value("${turboapps.tvm_app_id}") int tvmAppId,
            @Value("${turboapps.retry_count}") int retryCount,
            @Value("${turboapps.parallel}") int parallel,
            @Value("${turboapps.chunk_size}") int chunkSize,
            AsyncHttpClient asyncHttpClient,
            TvmIntegration tvmIntegration,
            DirectConfig directConfig) {
        TvmService tvmService = TvmService.fromIdStrict(tvmAppId);
        DirectConfig turboapps = directConfig.getBranch("turboapps");
        ParallelFetcherFactory parallelFetcherFactory =
                new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()
                        .withRequestTimeout(turboapps.getDuration("request_timeout"))
                        .withSoftTimeout(turboapps.getDuration("soft_timeout"))
                        .withRequestRetries(retryCount)
                        .withParallel(parallel));
        return new TurboAppsClient(url, chunkSize, tvmIntegration, tvmService, parallelFetcherFactory);
    }

    @Bean
    @Lazy
    CanvasClient canvasClient(
            @Value("${canvas.video_url}") String videoUrl,
            @Value("${canvas.canvas_backend_url}") String canvasBackendUrl,
            @Value("${canvas.canvas_url}") String canvasUrl,
            @Value("${canvas.token}") String token,
            @Value("${canvas.file}") String fileWithToken,
            @Value("${canvas.chunk_size}") int chunkSize,
            AsyncHttpClient asyncHttpClient
    )
            throws URISyntaxException {
        var config = new CanvasClientConfiguration(videoUrl, canvasBackendUrl, canvasUrl,
                fileWithToken,
                token,
                chunkSize);

        var fetcherSettings = new FetcherSettings()
                .withRequestTimeout(config.getReadTimeout());
        var fetcherFactory = new ParallelFetcherFactory(
                asyncHttpClient,
                fetcherSettings);

        return new CanvasClient(config, fetcherFactory);
    }

    @Bean(name = GEOSEARCH_CLIENT)
    public GeosearchClient geosearchClient(GeosearchClientSettings settings, AsyncHttpClient asyncHttpClient,
                                           TvmIntegration tvmIntegration) {
        int tvmAppId = settings.getTvmAppId();
        TvmService tvmService = TvmService.fromIdStrict(tvmAppId);
        ParallelFetcherFactory pff = new ParallelFetcherFactory(
                asyncHttpClient,
                new FetcherSettings().withRequestTimeout(settings.getRequestTimeout()).withRequestRetries(
                        settings.getRequestRetries())
        );
        return new GeosearchClient(settings, tvmService, pff, tvmIntegration);
    }

    @Bean
    public LanguageRecognizer languageRecognizer() {
        return new LanguageRecognizer();
    }

    @Bean
    public UzbekLanguageThresholds uzbekLanguageThresholds(PpcPropertiesSupport ppcPropertiesSupport) {
        return new UzbekLanguageThresholds(
                ppcPropertiesSupport.get(CYRILLIC_UZBEK_TO_RUSSIAN_THRESHOLD, Duration.ofMinutes(2)),
                ppcPropertiesSupport.get(LATIN_UZBEK_TO_RUSSIAN_THRESHOLD, Duration.ofMinutes(2)),
                ppcPropertiesSupport.get(UZBEK_TO_TURKISH_THRESHOLD, Duration.ofMinutes(2)),
                ppcPropertiesSupport.get(UZBEK_TO_UNKNOWN_THRESHOLD, Duration.ofMinutes(2)));
    }

    @Bean
    public QueryrecJni queryrecJni() {
        return new QueryrecJni(true);
    }

    @Bean
    public QueryrecService queryrecService(LanguageRecognizer languageRecognizer,
                                           UzbekLanguageThresholds uzbekLanguageThresholds,
                                           PpcPropertiesSupport ppcPropertiesSupport,
                                           QueryrecJni queryrecJni) {
        return new QueryrecService(languageRecognizer,
                ppcPropertiesSupport.get(CLIENTS_WITH_ENABLED_UZBEK_LANGUAGE, Duration.ofMinutes(2)),
                ppcPropertiesSupport.get(CLIENTS_WITH_DEFAULT_VIE_LANGUAGE, Duration.ofMinutes(2)),
                ppcPropertiesSupport.get(RECOGNIZED_BANNER_LANGUAGES_SET, Duration.ofMinutes(2)),
                uzbekLanguageThresholds,
                queryrecJni);
    }

    @Bean
    @Lazy
    public NotificationService notificationService(EnvironmentType environmentType, IntApiClient intApiClient) {
        if (environmentType.isSandbox()) {
            return new NotificationServiceDummyImpl();
        } else {
            return new NotificationServiceIntapiImpl(intApiClient, environmentType);
        }
    }

    @Bean
    public CapFactory capFactory() {
        return new CapFactory();
    }

    @Bean
    public MailSender mailSender(@Value("${sendmail.suppress_mail}") boolean suppressMail,
                                 @Value("${sendmail.relay.hostname}") String relayHostname,
                                 @Value("${sendmail.relay.port}") int relayPort,
                                 @Value("${sendmail.redirect_address}") String redirectEmail,
                                 @Value("${sendmail.yservice_salt_file_url}") Resource yserviceSaltFile,
                                 @Value("${sendmail.hmac_salt_file_url}") Resource hmacSaltFile,
                                 @Value("${sendmail.bounce_address}") String bounceAddress,
                                 EnvironmentType environmentType)
            throws IOException {
        if (suppressMail) {
            return new LoggingMailSender();
        }

        String serviceSalt;
        try (InputStream saltFileInputStream = yserviceSaltFile.getInputStream()) {
            serviceSalt = IOUtils.toString(saltFileInputStream, StandardCharsets.US_ASCII).trim();
        }

        String hmacSalt;
        try (InputStream hmacSaltFileInputStream = hmacSaltFile.getInputStream()) {
            hmacSalt = IOUtils.toString(hmacSaltFileInputStream, StandardCharsets.US_ASCII).trim();
        }

        SmtpMailSender actualSender = new SmtpMailSender(bounceAddress,
                relayHostname, relayPort,
                new YServiceTokenCreator("yadirect", serviceSalt, hmacSalt));

        if (redirectEmail == null || redirectEmail.isEmpty()) {
            if (!environmentType.isProductionOrPrestable()) {
                throw new IllegalArgumentException("mail must be redirected while not in production or prestable");
            }

            return actualSender;
        }

        return new RedirectingMailSender(actualSender, redirectEmail);
    }

    @Bean
    DbQueueTypeMap dbQueueTypeMap(DslContextProvider dslContextProvider) {
        return new DbQueueTypeMap(dslContextProvider);
    }

    @Bean
    DbQueueRepository dbQueueRepository(DslContextProvider dslContextProvider,
                                        ShardHelper shardHelper,
                                        DbQueueTypeMap dbQueueTypeMap) {
        return new DbQueueRepository(dslContextProvider, shardHelper, dbQueueTypeMap);
    }

    @Bean
    DbQueueService dbQueueService(DbQueueRepository dbQueueRepository) {
        return new DbQueueService(dbQueueRepository);
    }

    @Bean(name = DIRECT_FILES_MDS_HOLDER_BEAN_NAME)
    public MdsHolder directFilesMds(
            @Value("${mds.direct_files.read_host_port}") String readHostPort,
            @Value("${mds.direct_files.write_host_port}") String writeHostPort,
            @Value("${mds.direct_files.namespace}") String namespace,
            @Value("${mds.direct_files.token_file_url}") String tokenFileUrl,
            @Value("${mds.direct_files.timeout_in_seconds}") int timeoutInSeconds,
            @Value("${mds.direct_files.max_connections}") int maxConnections,
            @Qualifier(CONFIG_SCHEDULER_BEAN_NAME) TaskScheduler liveConfigChangeTaskScheduler) {
        return MdsHolder.instance(readHostPort, writeHostPort, namespace, timeoutInSeconds,
                LiveResourceFactory.get(tokenFileUrl), liveConfigChangeTaskScheduler, LIVE_CONFIG_CHECK_RATE,
                maxConnections);
    }

    @Bean
    public HttpConnectorSettings dssClientHttpConnectorSettings(
            @Value("${dssclient.base_url}") String baseUrl,
            @Value("${dssclient.timeout}") int timeout) {
        return new HttpConnectorSettings(baseUrl, timeout);
    }

    @Bean
    public DssClient dssClient(HttpConnectorSettings settings,
                               @Value("${dssclient.user_credentials_file_url}") Resource userCredentialsJson,
                               @Value("${dssclient.client_credentials_file_url}") Resource clientCredentialsJson)
            throws IOException {
        DssUserCredentials userCredentials;
        try (InputStream inputStream = userCredentialsJson.getInputStream()) {
            userCredentials = JsonUtils.fromJson(inputStream, DssUserCredentials.class);
        }

        DssClientCredentials clientCredentials;
        try (InputStream inputStream = clientCredentialsJson.getInputStream()) {
            clientCredentials = JsonUtils.fromJson(inputStream, DssClientCredentials.class);
        }

        return new DssClient(settings, userCredentials, clientCredentials);
    }

    @Bean
    public XivaPushesQueueService xivaPushesQueueService(
            DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        XivaPushesQueueRepository repository = new XivaPushesQueueRepository(dslContextProvider);
        return new XivaPushesQueueService(repository, shardHelper);
    }

    @Bean
    public PdfBuilder pdfBuilder() {
        return new PdfBuilder();
    }

    @Bean
    @Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
    public AffectedCampaignIdsContainer affectedCampaignIdsContainer() {
        return new AffectedCampaignIdsContainer();
    }

    @Bean(name = ACCESSIBLE_CAMPAIGN_CHEKER_PROVIDER)
    @Scope(value = WebApplicationContext.SCOPE_REQUEST, proxyMode = ScopedProxyMode.TARGET_CLASS)
    public RequestCampaignAccessibilityCheckerProvider accessibilityCheckerProvider() {
        return new RequestCampaignAccessibilityCheckerProvider();
    }

    @Bean
    @Lazy
    public static AvatarsConfigNameConverter avatarsConfigNameConverter() {
        return new AvatarsConfigNameConverter();
    }

    @Bean
    @Lazy
    public AmazonS3 mdsS3(DirectConfig config) {
        DirectConfig s3Config = config.getBranch("mds-s3");

        AWSCredentials awsCredentials;
        String tokenFileUrl = s3Config.findString("token_file_url").orElse("");
        if (tokenFileUrl.isEmpty()) {
            awsCredentials = new AnonymousAWSCredentials();
        } else {
            String tokenString = FileUtils.slurp(Paths.get(tokenFileUrl));
            awsCredentials = JsonUtils.fromJson(tokenString, MdsS3Token.class);
            checkState(!Strings.isNullOrEmpty(awsCredentials.getAWSAccessKeyId()), "S3 AccessKeyId must be defined");
            checkState(!Strings.isNullOrEmpty(awsCredentials.getAWSSecretKey()), "S3 SecretKey must be defined");
        }

        AwsClientBuilder.EndpointConfiguration endpoint = new AwsClientBuilder.EndpointConfiguration(
                s3Config.getString("endpoint"),
                // из переписки в чатике MDS S3 API support: советуют использовать их регион по умолчанию
                Regions.US_EAST_1.getName());

        ClientConfiguration clientConfig = new ClientConfiguration()
                .withConnectionTimeout(Ints.checkedCast(s3Config.getDuration("connection-timeout").toMillis()))
                .withRequestTimeout(Ints.checkedCast(s3Config.getDuration("request-timeout").toMillis()))
                .withMaxConnections(s3Config.getInt("max-connections"))
                .withMaxErrorRetry(s3Config.getInt("num-retries"));

        return AmazonS3ClientBuilder.standard()
                .withPathStyleAccessEnabled(Boolean.TRUE)
                .withEndpointConfiguration(endpoint)
                .withCredentials(new AWSStaticCredentialsProvider(awsCredentials))
                .withClientConfiguration(clientConfig)
                .disableChunkedEncoding() // until https://st.yandex-team.ru/MDS-3584
                .build();
    }

    @Bean(GEO_BASE_HELPER)
    public GeoBaseHelper geoBaseHelper(GeoTreeFactory geoTreeFactory) {
        return new GeoBaseHttpApiHelper(geoTreeFactory);
    }

    @Bean
    @Lazy
    public OrderStatDayRepository orderStatDayRepository(EnvironmentType environmentType,
                                                         OrderStatDayDynContextProvider orderStatDayDynContextProvider,
                                                         ShardHelper shardHelper,
                                                         CampaignRepository campaignRepository) {
        return environmentType.isSandbox()
                ? new OrderStatDaySandboxRepository(shardHelper, campaignRepository)
                : new OrderStatDayRepository(orderStatDayDynContextProvider);
    }

    @Bean
    @Lazy
    public Encrypter encrypter(@Value("${crypt.secret}") String secret) {
        return new Encrypter(secret);
    }

    @Bean("MediascopeClientSettingsEncrypter")
    @Lazy
    public Encrypter mediascopeSettingsEncrypter(DirectConfig directConfig) {
        var secretPath = directConfig.getString("mediascope_client.settings_salt_token_path");
        var secret = LiveResourceFactory.get(secretPath).getContent();
        return new Encrypter(secret);
    }

    @Lazy
    @Bean(CONVERSION_CENTER_ENCRYPTER_BEAN_NAME)
    public Encrypter conversionCenterEncrypter(
            @Value("${conversion_center.encryption.secret}") String secretPath,
            LiveResourceFactoryBean liveResourceFactoryBean) {
        var secret = liveResourceFactoryBean.get(secretPath).getContent();
        return new Encrypter(secret);
    }


    @Bean
    GeminiClient geminiClient(@Value("${gemini.api_url}") String apiUrl,
                              @Value("${gemini.user}") String user,
                              AsyncHttpClient asyncHttpClient) {
        ParallelFetcherFactory parallelFetcherFactory =
                new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings());
        return new GeminiClient(apiUrl, user, parallelFetcherFactory);
    }

    @Bean(SERVICE_HOLDER)
    @Lazy
    @Primary
    ServiceHolder serviceHolder(LiveResourceFactoryBean liveResourceFactoryBean,
                                DirectConfig directConfig) {
        return createServiceHolder(liveResourceFactoryBean, directConfig.getBranch("object_api"));
    }

    @Bean
    public GrutApiProperties grutApiProperties() {
        return new DefaultGrutApiProperties();
    }

    @Bean
    @Lazy
    @Primary
    GrutClient grutClient(@Qualifier(SERVICE_HOLDER) ServiceHolder serviceHolder,
                          PpcPropertiesSupport ppcPropertiesSupport) {
        return new GrutGrpcClient(serviceHolder, new GrutTraceCallback(ppcPropertiesSupport), GRUT_METRICS_REGISTRY);
    }

    @Bean(SERVICE_HOLDER_FOR_WATCHLOG)
    @Lazy
    ServiceHolder serviceHolderForWatchlog(LiveResourceFactoryBean liveResourceFactoryBean,
                                           DirectConfig directConfig) {
        return createServiceHolder(liveResourceFactoryBean, directConfig.getBranch("object_api"),
                DIRECT_CONFIG_SERVICE_HOLDER, true);
    }

    @Bean(GRUT_CLIENT_FOR_WATCHLOG)
    @Lazy
    GrutClient grutClientForWatchlog(@Qualifier(SERVICE_HOLDER_FOR_WATCHLOG) ServiceHolder serviceHolder,
                                     PpcPropertiesSupport ppcPropertiesSupport) {
        return new GrutGrpcClient(serviceHolder, new GrutTraceCallback(ppcPropertiesSupport), GRUT_METRICS_REGISTRY);
    }

    @Bean(name = CoreConfiguration.DEFAULT_COPY_RECENT_STATISTICS_PROVIDER_BEAN_NAME)
    @Lazy
    public KeywordRecentStatisticsProvider keywordRecentStatisticsProvider() {
        return keywordRequests -> emptyMap();
    }

    private ServiceHolder createServiceHolder(LiveResourceFactoryBean liveResourceFactoryBean,
                                              DirectConfig directConfig) {
        return createServiceHolder(liveResourceFactoryBean, directConfig,
                DIRECT_CONFIG_SERVICE_HOLDER, false);
    }

    public static ServiceHolder createServiceHolder(LiveResourceFactoryBean liveResourceFactoryBean,
                                                    DirectConfig directConfig,
                                                    String serviceHolderPath,
                                                    boolean forWatchlog) {
        var ypEndpointSetId = directConfig.getString(serviceHolderPath + ".yp_endpointset_id");
        if (ypEndpointSetId == null) {
            throw new IllegalStateException(serviceHolderPath + ".yp_endpointset_id entry is required to create a" +
                    "service holder");
        }
        var tokenResource = directConfig.getString(serviceHolderPath + ".token");
        var channelsCount = directConfig.getInt("service_holder.channels_count");
        var keepAliveSeconds = directConfig.getLong("service_holder.keepalive_interval_seconds");
        var maxMessageSize = directConfig.getInt("service_holder.max_message_size");
        var maxMetadataSize = directConfig.getInt("service_holder.max_metadata_size");
        var timeout = forWatchlog ? directConfig.getLong("service_holder.watchlog_timeout_sec")
                : directConfig.getLong("service_holder.timeout_sec");
        var token = tokenResource != null && !tokenResource.isEmpty()
                ? liveResourceFactoryBean.get(tokenResource).getContent().trim()
                : null;

        ServiceHolder ypServiceHolder = null;
        final var currentDc = System.getProperty("direct.dc");
        final var clusterWithEndpointSets = DcTranslator.getClusterByDC(currentDc)
                .stream()
                .map(s -> TuplesKt.to(s, ypEndpointSetId))
                .collect(toList());
        final var fallbackDc = DcTranslator.getFallbackDC(currentDc);
        final var livenessCheckParams = new LivenessCheckParams();
        if (fallbackDc != null) {
            final var fallbackClusterWithEndpointSets = fallbackDc
                    .stream()
                    .map(s -> TuplesKt.to(s, ypEndpointSetId))
                    .collect(toList());
            ypServiceHolder = new YpServiceHolderWithFallback(
                    Arrays.asList(clusterWithEndpointSets, fallbackClusterWithEndpointSets),
                    token,
                    channelsCount,
                    null,
                    keepAliveSeconds,
                    maxMessageSize,
                    maxMetadataSize,
                    timeout,
                    GRUT_METRICS_REGISTRY,
                    livenessCheckParams);
        } else {
            ypServiceHolder = new YpServiceHolder(
                    clusterWithEndpointSets,
                    token,
                    channelsCount,
                    null,
                    keepAliveSeconds,
                    maxMessageSize,
                    maxMetadataSize,
                    timeout,
                    GRUT_METRICS_REGISTRY,
                    livenessCheckParams);
        }
        return ypServiceHolder;
    }
}
