package ru.yandex.direct.core.configuration;

import java.net.URI;

import org.asynchttpclient.AsyncHttpClient;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.balance.client.BalanceClient;
import ru.yandex.direct.common.configuration.CommonConfiguration;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.core.entity.freelancer.service.AvatarsClientPool;
import ru.yandex.direct.core.entity.freelancer.service.AvatarsClientPoolFactory;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.core.service.integration.passport.NonIdempotentHttpRequestRetryHandler;
import ru.yandex.direct.core.service.integration.passport.PassportService;
import ru.yandex.direct.core.service.integration.passport.TvmPassportClientBuilder;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;
import ru.yandex.inside.passport.internal.api.PassportClient;

@Configuration
@Lazy
@Import({
        CommonConfiguration.class,
})
public class CoreIntegrationConfiguration {
    public static final String FREELANCERS_AVATARS_CLIENT_POOL = "freelancersAvatarsClientPool";
    public static final String TEXT_BANNER_IMAGES_AVATARS_CLIENT_POOL = "textBannerImagesAvatarsClientPool";
    public static final String IMAGE_BANNER_IMAGES_AVATARS_CLIENT_POOL = "imageBannerImagesAvatarsClientPool";
    public static final String UAC_AVATARS_CLIENT_POOL = "uacAvatarsClientPool";
    public static final String CONVERSION_SOURCE_TYPE_AVATARS_CLIENT_POOL = "conversionSourceTypeAvatarsClientPool";

    @Bean
    public PassportClient passportClient(
            @Value("${passport.internal.api.serviceUrl}") String serviceUri,
            @Value("${passport.internal.api.connectionRequestTimeoutInMs:1000}") int connectionRequestTimeoutInMs,
            @Value("${passport.internal.api.connectTimeoutInMs:1000}") int connectTimeoutInMs,
            @Value("${passport.internal.api.socketTimeoutInMs:3000}") int socketTimeoutInMs,
            @Value("${passport.internal.api.maxConnectionCount:100}") int maxConnectionCount,
            @Value("${passport.internal.api.tvm_app_id}") int tvmServiceId,
            TvmIntegration tvmIntegration) {

        return new TvmPassportClientBuilder()
                .withServiceUri(URI.create(serviceUri))
                .withConnectionRequestTimeoutInMs(connectionRequestTimeoutInMs)
                .withConnectTimeoutInMs(connectTimeoutInMs)
                .withSocketTimeoutInMs(socketTimeoutInMs)
                .withMaxConnectionCount(maxConnectionCount)
                .withRetryHandler(new NonIdempotentHttpRequestRetryHandler())
                .withTvmIntegration(tvmIntegration)
                .withTvmServiceId(TvmService.fromIdStrict(tvmServiceId))
                .build();
    }

    @Bean
    public PassportService passportService(
            @Value("${passport.internal.api.consumer}") String consumer,
            @Value("${passport.internal.api.retryMaxAttempts:2}") int retryMaxAttempts,
            @Value("${passport.internal.api.retryBackoffDelayInMs:100}") int retryBackoffDelayInMs,
            PassportClient passportClient
    ) {
        return new PassportService(
                passportClient,
                consumer,
                retryMaxAttempts,
                retryBackoffDelayInMs);
    }

    @Bean
    public BalanceService balanceService(BalanceClient balanceClient,
                                         @Value("${balance.directServiceId}") Integer directServiceId,
                                         @Value("${balance.directServiceToken}") String directServiceToken) {
        return new BalanceService(balanceClient, directServiceId, directServiceToken);
    }

    @Bean(name = FREELANCERS_AVATARS_CLIENT_POOL)
    @Lazy
    public AvatarsClientPool freelancersAvatarsClientPool(DirectConfig directConfig,
                                                          AsyncHttpClient asyncHttpClient) {
        DirectConfig avatarsClientConfig = directConfig.getBranch("freelancer_avatars_client");
        AvatarsClientPoolFactory avatarsClientPoolFactory =
                new AvatarsClientPoolFactory(avatarsClientConfig, asyncHttpClient, null, null);
        return avatarsClientPoolFactory.getAvatarsClientPool();
    }

    @Bean(name = TEXT_BANNER_IMAGES_AVATARS_CLIENT_POOL)
    @Lazy
    public AvatarsClientPool textBannerImagesAvatarsClientPool(DirectConfig directConfig,
                                                               AsyncHttpClient asyncHttpClient) {
        DirectConfig avatarsClientConfig = directConfig.getBranch("text_ad_image_avatars_client");
        AvatarsClientPoolFactory avatarsClientPoolFactory =
                new AvatarsClientPoolFactory(avatarsClientConfig, asyncHttpClient, null, null);
        return avatarsClientPoolFactory.getAvatarsClientPool();
    }

    @Bean(name = IMAGE_BANNER_IMAGES_AVATARS_CLIENT_POOL)
    @Lazy
    public AvatarsClientPool imageBannerImagesAvatarsClientPool(DirectConfig directConfig,
                                                                AsyncHttpClient asyncHttpClient) {
        DirectConfig avatarsClientConfig = directConfig.getBranch("image_ad_image_avatars_client");
        AvatarsClientPoolFactory avatarsClientPoolFactory =
                new AvatarsClientPoolFactory(avatarsClientConfig, asyncHttpClient, null, null);
        return avatarsClientPoolFactory.getAvatarsClientPool();
    }

    @Bean(name = UAC_AVATARS_CLIENT_POOL)
    @Lazy
    public AvatarsClientPool uacAvatarsClientPool(DirectConfig directConfig,
                                                  AsyncHttpClient asyncHttpClient,
                                                  TvmIntegration tvmIntegration) {
        DirectConfig avatarsClientConfig = directConfig.getBranch("uac_avatars_client");
        AvatarsClientPoolFactory avatarsClientPoolFactory =
                new AvatarsClientPoolFactory(avatarsClientConfig, asyncHttpClient,
                        tvmIntegration, TvmService.AVATARS_PROD);
        return avatarsClientPoolFactory.getAvatarsClientPool();
    }

    @Bean(name = CONVERSION_SOURCE_TYPE_AVATARS_CLIENT_POOL)
    @Lazy
    public AvatarsClientPool conversionSourceTypeAvatarsClientPool(DirectConfig directConfig,
                                                                   AsyncHttpClient asyncHttpClient) {
        DirectConfig conversionSourceTypeClientConfig = directConfig.getBranch("conversion_source_type_avatars_client");
        AvatarsClientPoolFactory avatarsClientPoolFactory =
                new AvatarsClientPoolFactory(conversionSourceTypeClientConfig, asyncHttpClient, null, null);
        return avatarsClientPoolFactory.getAvatarsClientPool();
    }
}
