package ru.yandex.direct.core.copyentity

import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.model.Entity

data class CopyConfig<T : Entity<KeyT>, KeyT>(
    val clientIdFrom: ClientId,
    val clientIdTo: ClientId,

    val operatorUid: Long,

    val entityClass: Class<T>,
    val entityIds: List<KeyT>,

    val flags: CopyCampaignFlags = CopyCampaignFlags(),

    val copyMappings: Map<Class<out Entity<*>>, Map<*, *>> = mapOf(
        Client::class.java to mapOf(
            clientIdFrom.asLong() to clientIdTo.asLong(),
        ),
    ),
)

class CopyConfigBuilder<T : Entity<KeyT>, KeyT>(
    private val clientIdFrom: ClientId,
    private val clientIdTo: ClientId,
    private val operatorUid: Long,
    private val entityClass: Class<T>,
    private val entityIds: List<KeyT>,
) {
    private var flags = CopyCampaignFlags()
    private val copyMappings: MutableMap<Class<out Entity<*>>, MutableMap<*, *>> = mutableMapOf()

    init {
        withParentIdMapping(Client::class.java, clientIdFrom.asLong(), clientIdTo.asLong())
    }

    fun withFlags(flags: CopyCampaignFlags): CopyConfigBuilder<T, KeyT> {
        this.flags = flags
        return this
    }

    /**
     * Метод добавляем маппинг идентификаторов родителей копируемых объектов. Например, при копировании групп баннеров,
     * нужно указать для каждого идентификатора родительской кампании из списка копируемых групп, целевой идентификатор
     * кампании, в которую эта группа будет скопирована. Маппинг для клиентов задается по умолчанию.
     *
     * @param parentEntityClass класс родителя копируемых объектов
     * @param parentIdFrom исходный идентификатор родителя
     * @param parentIdTo целевой идентификатор родителя
     * @param <P> тип родителя копируемых объектов
     * @param <PKey> тип ключа родителя копируемых объектов
     */
    fun <P : Entity<PKey>, PKey> withParentIdMapping(
        parentEntityClass: Class<P>,
        parentIdFrom: PKey,
        parentIdTo: PKey,
    ): CopyConfigBuilder<T, KeyT> {
        @Suppress("UNCHECKED_CAST")
        val parentEntityMapping: MutableMap<PKey, PKey> = copyMappings
            .computeIfAbsent(parentEntityClass) { mutableMapOf<PKey, PKey>() }
                as MutableMap<PKey, PKey>

        parentEntityMapping[parentIdFrom] = parentIdTo
        return this
    }

    fun build() = CopyConfig(clientIdFrom, clientIdTo, operatorUid, entityClass, entityIds, flags, copyMappings)
}
