package ru.yandex.direct.core.copyentity;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.copyentity.mediators.CopyMediatorsFacade;
import ru.yandex.direct.core.copyentity.preprocessors.CopyPreprocessorFacade;
import ru.yandex.direct.core.copyentity.prevalidators.Prevalidator;
import ru.yandex.direct.model.Entity;
import ru.yandex.direct.model.Relationship;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Операция копирования сущностей и всего леса сущностей, из которых они состоят.
 * <p>
 * Дерево (а точнее, граф, так как в нем есть циклы), состоит из узлов - сущностей, и ребер - связей
 * между ними. Под сущностью имеется в виду сущность ядра (Banner, AdGroup, etc), вдобавок помеченная специальным
 * интерфейсом Entity. Под связкой понимается отношение a-la в базе foreign key -> primary key, в моделях это
 * реализуется новой моделью Relationship, генерирующейся из нового атрибута поля класса, соответствующего внешнему
 * ключу на ребенке.
 * <p>
 * Каждому классу сущности соответствует сервис сущности, который умеет производить основные операции, необходимые
 * для копирования -  получить сущности по id (get), добавить новую сущность в базу (add). Обычно этот интерфейс
 * реализует текущий ядровый сервис для этой сущности - т.е. BannerService для Banner, AdGroupService для AdGroup
 * и т.п. Для связок есть аналогичный сервис связи, который умеет получать id детей по id родителей (в обратном
 * направлении достаточно прочитать значение внешнего ключа на ребенке). Сейчас это обычно отдельный класс сервиса, так
 * как у сущности может быть/обычно много связей, а у связи два конца - две сущности. Отсутствие сервиса сущности для
 * сущности, так же как и отсутствие сервиса связи для связи является ошибкой.
 * <p>
 * Для каждой сущности из дерева копирование происходит по следующему алгоритму:
 * - получение списка сущностей для копирования по их id
 * - поиск родительских сущностей (на которые текущие указывают через foreign key) и копирование их, замена значений
 * foreign key в сущностях - это делается через поиск связок от текущей сущности к родителям
 * - обнуление id сущностей/приведение их к виду "новых" сущностей
 * - добавление сущностей в базу
 * - поиск сущностей-детей (которые указывают на эти сущности через foreign key), копирование их - это делаеттся через
 * поиск связок от текущей сущности к детям
 * <p>
 * Копирование родителей и детей производится по тому же самому алгоритму. Не копируются сущности, для которых уже
 * есть мэппинг "этот id считается скопированным в этот id". Этот механизм позволяет как задать, откуда куда будут
 * копироваться сущности (с одного id кампании на другой id кампании, например), так и позволяет избежать повторного/
 * бесконечного копирования сущностей при наличии колец в графе (сущности из этого мэппинга не копируются, после
 * копирования сущность в этот мэппинг немедленно попадает, то есть каждая сущность может быть скопирована только
 * один раз).
 * <p>
 * Также не копируются сущности, помеченные интерфейсом ClientLibraryObject, если копирование происходит в пределах
 * одного клиента (например, RetargetingCondition). Аналогично не копируются сущности, помеченные интерфейсом
 * CampaignLibraryObject, если копирование происходит внутри кампании (пример такого объекта - Vcard).
 * <p>
 * Таким образом, чтобы добавить какую-либо сущность к копированию, нужно
 * - проложить дорожку связки к нужной сущности от какой-либо уже копируемой сущности. Если такой дорожки в один шаг
 * нет, то надо добавить в копирование другую сущность, к которой такую дорожку можно проложить, и от нее уже
 * проложить дорожку к текущей сущности. Добавить соответствующий Relationship в нужную модель (если текущая
 * сущность - родитель в связке, то в ту сущность, откуда идет связь, если ребенок - то в текущую). Например, если
 * мы хотим добавить копирование Banner, и у нас уже копируются AdGroup, то мы в Banner на поле adGroupId добавляем
 * связку AdGroupContainsBanners
 * - добавить соответствующий новой связке RelationshipService, реализовать метод getIdsByParentIds. В нашем примере -
 * добавляем сервис BannersOfAdGroups, в нем реализуем метод getIdsByParentIds через получение bannerIds by
 * adGroupIds
 * - пометить нужную сущность интерфейсом Entity - в нашем примере в модель i_banner.conf добавляем implements: [Entity]
 * - прописать в ее сервис интерфейс EntityService, реализовать методы get и add. В нашем примере в BannerService
 * добавляем implements EntityService&lt;Banner&gt;, реализуем метод get через getBannersByIds, метод add через
 * addBanners
 * - если нужно, пометить сущность интерфейсами ClientLibraryObject (если сущность хранится на клиенте) или
 * CampaignLibraryObject (если на кампании). Например, Vcard нужно пометить интерфейсом CampaignLibraryObject, а
 * RetargetingCondition - интерфейсом ClientLibraryObject. В нашем примере Banner ни одним из этих интерфейсов
 * помечать не надо.
 * - для остальных связок этой сущности при необходимости добавить копирование их/связанных ими сущностей. В нашем
 * примере надо добавить связку с Vcard, и связку с BannerImage, но, поскольку BannerImage часть модели баннера,
 * то дальше связку с BannerImageFormat.
 *
 * @param <T>    тип копируемой сущности
 * @param <KeyT> тип первичного ключа копируемой сущности
 */
@ParametersAreNonnullByDefault
@SuppressWarnings("rawtypes")
public class CopyOperation<T extends Entity<KeyT>, KeyT> {
    private static final Logger logger = LoggerFactory.getLogger(CopyOperation.class);

    public static final int INSERT_CHUNK_SIZE = 1_000;

    private final EntityGraphNavigator navigator;
    private final EntityLoadService entityLoadService;

    private final Prevalidator prevalidator;
    private final CopyMediatorsFacade mediator;
    private final CopyPreprocessorFacade preprocessor;

    private final CopyOperationContainer copyContainer;

    CopyOperation(
            EntityGraphNavigator navigator,
            EntityLoadService entityLoadService,
            Prevalidator prevalidator,
            CopyMediatorsFacade mediator,
            CopyPreprocessorFacade preprocessor,
            CopyOperationContainer copyOperationContainer
    ) {
        this.navigator = navigator;
        this.entityLoadService = entityLoadService;
        this.prevalidator = prevalidator;
        this.mediator = mediator;
        this.preprocessor = preprocessor;
        this.copyContainer = copyOperationContainer;
    }

    public CopyOperationContainer getCopyContainer() {
        return copyContainer;
    }

    /**
     * Копирует сущности с указанными id заданного класса в другой объект, заданный мэппингами
     */
    @SuppressWarnings("unchecked")
    public CopyResult<KeyT> copy() {
        List<KeyT> ids = (List<KeyT>) copyContainer.getEntityIds();

        Class<T> entityClass = (Class<T>) navigator
                .getClosestToEntityAncestor(copyContainer.getEntityClass());

        EntityContext entityContext = new EntityContext(copyContainer);
        Map<Class, ValidationResult<?, Defect>> prefilterResults =
                entityLoadService.loadGraph(entityClass, ids, entityContext);

        Map<Class, MassResult> prevalidationResults = prevalidator.prevalidate(entityContext, copyContainer);
        if (!prevalidationResults.isEmpty()) {
            return new CopyResult<>(ids, entityClass, prevalidationResults,
                    prefilterResults, entityContext, ResultState.BROKEN);
        }

        Map<Class<?>, MassResult<?>> mediationResults = mediator.mediate(entityContext, copyContainer);
        if (!mediationResults.isEmpty()) {
            return new CopyResult(ids, entityClass, mediationResults,
                    prefilterResults, entityContext, ResultState.BROKEN);
        }

        Map<Class, MassResult> results = new HashMap<>();
        add(entityClass, entityContext, results);
        return new CopyResult<>(ids, entityClass, results, prefilterResults, entityContext, ResultState.SUCCESSFUL);
    }

    @SuppressWarnings("unchecked")
    private <P extends Entity<KeyP>, KeyP> void add(
            Class<P> entityClass,
            EntityContext entityContext,
            Map<Class, MassResult> results) {
        try (var ignored = Trace.current().profile("entityGraph:add", entityClass.getSimpleName())) {
            List<P> entities = entityContext.startCopying(entityClass);
            if (entities.isEmpty()) {
                return;
            }

            logger.info("Adding {} - {} entities", entityClass.getName(), entities.size());

            // добавить сущности-родители (thisEntity.parentEntityId = parentEntity.id)
            for (var relationship : entityLoadService.getChildToParentRelationships(entityClass, copyContainer)) {
                add(relationship.getParentEntityClass(), entityContext, results);
            }

            for (var relationship : entityLoadService.getRelationshipsToSetIds(entityClass, copyContainer)) {
                List<?> parentIdsBefore = StreamEx.of(entities).map(relationship::getParentId).toList();
                // проставляем в entities новые parentIds
                entityContext.setParentIds(entityClass, relationship);
                // отфильтровываем сущности с нескопированными родителями
                entities = StreamEx.of(entities).zipWith(parentIdsBefore.stream())
                        // parent id был, а теперь null - значит родитель почему-то не был скопирован, детей
                        // копировать не надо
                        .remove(e -> e.getValue() != null && relationship.getParentId(e.getKey()) == null)
                        .map(Map.Entry::getKey)
                        .toList();
            }

            preprocessor.preprocess(entities, copyContainer);
            entityContext.setIdsToNull(entityClass);

            EntityService<P, KeyP> service = navigator.getEntityService(entityClass);

            MassResult result = service.copyChunked(copyContainer, entities, Applicability.PARTIAL, INSERT_CHUNK_SIZE);
            results.put(entityClass, result);

            entityContext.finishCopying(entityClass);

            // добавить сущности-дети (thisEntity.id = childEntity.thisEntityId)
            for (Relationship relationship :
                    entityLoadService.getParentToChildRelationships(entityClass, copyContainer)) {
                add(relationship.getChildEntityClass(), entityContext, results);
            }
        }
    }

}
