package ru.yandex.direct.core.copyentity

import ru.yandex.direct.core.copyentity.model.CampaignPreprocessFlags
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.model.Entity
import java.math.BigDecimal
import java.util.Locale

data class CopyOperationContainer(
    private val config: CopyConfig<*, *>,
    val shardFrom: Int,
    val clientFrom: Client,
    val shardTo: Int,
    val clientTo: Client,
    val chiefUserTo: User,
    val locale: Locale,
) {
    // Продублированы поля из config, для более удобного использования
    val clientIdTo: ClientId = config.clientIdTo
    val clientIdFrom: ClientId = config.clientIdFrom
    val operatorUid: Long = config.operatorUid
    val entityClass: Class<out Entity<*>> = config.entityClass
    val entityIds: List<*> = config.entityIds
    val flags: CopyCampaignFlags = config.flags
    val copyMappings: Map<Class<out Entity<*>>, Map<*, *>> = config.copyMappings

    val isCopyingBetweenShards = shardFrom != shardTo
    val isCopyingBetweenClients = clientFrom.id != clientTo.id

    internal var fixedDestinationAdGroupAutoPrices: Map<Long, BigDecimal> = mapOf()
    var targetCampaignTypeByGroupId: Map<Long, CampaignType> = mapOf()

    private val campaignPreprocessFlagsByIds: MutableMap<Long, CampaignPreprocessFlags> = mutableMapOf()

    fun isCopyingBetweenCampaigns(): Boolean {
        if (BaseCampaign::class.java.isAssignableFrom(config.entityClass)) {
            return true
        }

        return config.copyMappings[BaseCampaign::class.java]
            ?.any { (key, value) -> key != value }
            ?: return false
    }

    fun getCampaignPreprocessFlagsById(campaignId: Long) =
        campaignPreprocessFlagsByIds.getOrPut(campaignId) { CampaignPreprocessFlags() }

    /**
     * Определяет, изменится ли после копирования идентификатор у сущности класса baseParentClass
     * с идентификатором parentKey. Метод полезен, например, при копировании групп или баннеров, когда нужно понять,
     * они копируются внутри своего же родителя, или копируются вместе с родителем (или прилинковываются
     * к другому родителю). От этого зависит, нужно ли их переименовывать или нет.
     */
    fun <T : Entity<TKey>, TKey> isCopyingToSameEntity(baseParentClass: Class<T>, parentKey: TKey): Boolean {
        // Если класс сущности среди копируемых, то идентификатор точно не останется тем же самым
        if (baseParentClass.isAssignableFrom(config.entityClass)) {
            return false
        }

        // Получаем мэппинги по классу сущности
        // Если их нет, значит идентификатор не изменится
        val entityMapping: Map<*, *> = config.copyMappings[baseParentClass]
            ?: return true

        // Если мэппинги есть, то сравниваем старый и новый идентификатор
        return parentKey == entityMapping[parentKey]
    }

    /**
     * Возвращает идентификатор родительского объекта после копирования, если он задан до копирования,
     * или текущий идентификатор родителя, если он будет копироваться вместе с потомками
     */
    fun <T : Entity<KeyT>, KeyT> getCopyToParentIdOrSame(baseParentClass: Class<T>, parentKey: KeyT): KeyT {
        val entityMapping: Map<*, *> = config.copyMappings[baseParentClass]
            ?: return parentKey
        return entityMapping.getOrDefault(parentKey, parentKey) as KeyT
    }

    fun <T : Entity<KeyT>, KeyT> onFinishCopying(entityClass: Class<T>, idsMappings: Map<KeyT, KeyT>) {
        if (entityClass == AdGroup::class.java) {
            remapAdGroupIdsAfterCopy(idsMappings as Map<Long, Long>)
        }
    }

    /**
     * Заменяет в карте ставок по-умолчанию и карте типов кампаний для групп после копирования
     * старые идентификаторы групп на новые
     */
    private fun remapAdGroupIdsAfterCopy(adGroupMappings: Map<Long, Long>) {
        val remappedFixedAdGroupAutoPrices: MutableMap<Long, BigDecimal> = mutableMapOf()
        fixedDestinationAdGroupAutoPrices.forEach {
            val newId: Long? = adGroupMappings[it.key]
            if (newId != null) {
                remappedFixedAdGroupAutoPrices[newId] = it.value
            }
        }
        fixedDestinationAdGroupAutoPrices = remappedFixedAdGroupAutoPrices;

        val remappedTargetCampaignTypeByGroupId: MutableMap<Long, CampaignType> = mutableMapOf()
        targetCampaignTypeByGroupId.forEach {
            val newId: Long? = adGroupMappings[it.key]
            if (newId != null) {
                remappedTargetCampaignTypeByGroupId[newId] = it.value
            }
        }
        targetCampaignTypeByGroupId = remappedTargetCampaignTypeByGroupId
    }

    fun getShowConditionFixedAutoPrices(): ShowConditionFixedAutoPrices {
        return ShowConditionFixedAutoPrices.ofPerAdGroupFixedPrices(fixedDestinationAdGroupAutoPrices)
    }
}
