package ru.yandex.direct.core.copyentity

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.mediators.DefaultCopyMediatorsFacade
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags
import ru.yandex.direct.core.copyentity.preprocessors.CopyPreprocessorFacade
import ru.yandex.direct.core.copyentity.prevalidators.CopyPrevalidator
import ru.yandex.direct.core.copyentity.translations.RenameProcessor
import ru.yandex.direct.core.entity.client.model.Client
import ru.yandex.direct.core.entity.client.service.ClientService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.core.entity.user.service.UserService
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.model.Entity

@Component
class CopyOperationFactory(
    private val entityGraphNavigator: EntityGraphNavigator,
    private val entityLoadService: EntityLoadService,
    private val copyPrevalidator: CopyPrevalidator,
    private val copyMediatorsFacade: DefaultCopyMediatorsFacade,
    private val copyPreprocessorFacade: CopyPreprocessorFacade,
    private val clientService: ClientService,
    private val userService: UserService,
) {

    fun <T : Entity<KeyT>, KeyT> build(
        config: CopyConfig<T, KeyT>,
    ): CopyOperation<T, KeyT> {
        val shardFrom = clientService.getShardByClientIdStrictly(config.clientIdFrom)
        val clientFrom = clientService.getClient(config.clientIdFrom)!!

        val shardTo = clientService.getShardByClientIdStrictly(config.clientIdTo)
        val clientTo = clientService.getClient(config.clientIdTo)!!

        val chiefUserTo: User = userService.getUser(clientTo.chiefUid)!!

        val copyContainer = CopyOperationContainer(
            config = config,
            shardFrom = shardFrom,
            clientFrom = clientFrom,
            shardTo = shardTo,
            clientTo = clientTo,
            chiefUserTo = chiefUserTo,
            locale = RenameProcessor.getCopyLocaleByChiefUser(chiefUserTo),
        )

        return CopyOperation(
            entityGraphNavigator,
            entityLoadService,
            copyPrevalidator,
            copyMediatorsFacade,
            copyPreprocessorFacade,
            copyContainer,
        )
    }

    fun <T : Entity<KeyT>, KeyT> build(
        shardFrom: Int,
        clientFrom: Client,
        shardTo: Int,
        clientTo: Client,
        operatorUid: Long,
        entityClass: Class<T>,
        entityIds: List<KeyT>,
        flags: CopyCampaignFlags,
    ): CopyOperation<T, KeyT> {
        val chiefUserTo: User = userService.getUser(clientTo.chiefUid)!!

        val copyContainer = CopyOperationContainer(
            config = CopyConfig(
                clientIdFrom = ClientId.fromLong(clientFrom.id),
                clientIdTo = ClientId.fromLong(clientTo.id),
                operatorUid = operatorUid,
                entityClass = entityClass,
                entityIds = entityIds,
                flags = flags,
            ),
            shardFrom = shardFrom,
            clientFrom = clientFrom,
            shardTo = shardTo,
            clientTo = clientTo,
            chiefUserTo = chiefUserTo,
            locale = RenameProcessor.getCopyLocaleByChiefUser(chiefUserTo),
        )

        return CopyOperation(
            entityGraphNavigator,
            entityLoadService,
            copyPrevalidator,
            copyMediatorsFacade,
            copyPreprocessorFacade,
            copyContainer,
        )
    }
}
