package ru.yandex.direct.core.copyentity;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.model.Entity;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.copyentity.CopyValidationResultUtils.logFailedResults;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Результат копирования дерева сущностей
 *
 * @param <KeyT> тип первичного ключа копируемой верхнеуровневой сущности
 */
@ParametersAreNonnullByDefault
@SuppressWarnings("rawtypes")
public class CopyResult<KeyT> {
    private static final Logger logger = LoggerFactory.getLogger(CopyResult.class);

    private final MassResult<KeyT> massResult;
    private final EntityContext entityContext;

    CopyResult(List<KeyT> entityIds, Class entityClass, Map<Class, MassResult> results,
               Map<Class, ValidationResult<?, Defect>> prefilterResults, EntityContext entityContext,
               ResultState resultState) {
        this.entityContext = entityContext;
        this.massResult = mergeMassResults(entityIds, results, prefilterResults, entityClass, entityContext,
                resultState);
    }

    /**
     * Возвращает массовый результат копирования дерева сущностей
     */
    public MassResult<KeyT> getMassResult() {
        return massResult;
    }

    /**
     * Для заданного класса возвращает информацию о скопированных сущностях, включая заданную исходно
     *
     * @param clazz класс сущности, для которого запрашивается информация
     * @return мэппинг "id источника" - "id копии"
     */
    public Map<Object, Object> getEntityMapping(Class clazz) {
        return entityContext.getCopyMappingsByClass(clazz);
    }

    public <T extends Entity<K>, K> Map<K, K> getEntityMappings(Class<T> clazz) {
        return entityContext.getCopyMappingByClass(clazz);
    }

    public EntityContext getEntityContext() {
        return entityContext;
    }

    public void logFailedResultsForMonitoring() {
        var vr = massResult.getValidationResult();
        logFailedResults(vr, logger);
    }

    public <T extends Entity<TKey>, TKey> List<Result<TKey>> getCopiedEntityResults(Class<T> clazz) {
        var entityMappings = getEntityMapping(clazz);
        List<Result<TKey>> newResults = new ArrayList<>();
        for (Result<?> result : massResult.getResult()) {
            Object id = result.getResult();
            TKey newId = (TKey) entityMappings.get(id);
            if (newId != null) {
                Result<TKey> newAdGroupResult = new Result<>(newId, result.getValidationResult(), result.getState());
                newResults.add(newAdGroupResult);
            }
        }
        return newResults;
    }

    @SuppressWarnings("unchecked")
    private MassResult<KeyT> mergeMassResults(List<KeyT> entityIds, Map<Class, MassResult> results,
                                              Map<Class, ValidationResult<?, Defect>> prefilterResults,
                                              Class entityClass,
                                              EntityContext entityContext, ResultState resultState) {
        var validationResults = EntryStream.of(results)
                .mapValues(mr -> mr.getValidationResult())
                .toMap();

        // Восстанавливаем старые идентификаторы объектов, вместо нулевых. Нулевыми они могут стать, только если
        // объекты дошли до копирования и их старые идентификаторы были сброшены на null, но на операции добавления
        // объекты не прошли валидацию. В этом случае нужно восстановить старые идентификаторы. Потому что,
        // во-первых, нам нужно корректно смержить результаты валидации, чтобы они все не слились в один
        // для разных объектов, а во-вторых, в логах с ошибками будут нулевые идентификаторы, и разобраться
        // на каком же объекте из БД сломалась валидация будет очень сложно.
        entityContext.restoreNullIds();

        // если нет результата валидации (такое бывает в превалидации, когда мы проверяем не все), делаем dummy
        // результат, чтобы он был корнем иерархии ошибок.
        // на данном этапе в entityVR лежат энтити, которые дошли непосредственно до стадии копирования
        // (здесь не будет энтити, которые отфильтровались на этапе prefilter),
        // при этом порядок энтитей как в entityIds НЕ гарантируется
        ValidationResult entityVR = nvl(validationResults.get(entityClass),
                ValidationResult.success(entityContext.getObjects(entityClass)));

        ValidationResult<?, Defect> prefilterEntityVR = prefilterResults.get(entityClass);
        // после мержа с prefilterEntityVR в entityVR лежат все энтити -
        // и которые отфильтровались, и которые попытались (возможно успешно, а возможно и нет) скопироваться.
        // при этом гарантируется, что порядок энтитей будет такой же, как в соответствующем им списке entityIds
        entityVR = mergeWithPrefilterResult(entityVR, prefilterEntityVR, entityIds, entityClass, entityContext);
        validationResults.put(entityClass, entityVR);

        var mergedValidationResult = mergeValidationResults(entityVR, validationResults, entityClass, entityContext);
        ValidationResult finalEntityVR = entityVR;
        List<Result<KeyT>> entityResults = EntryStream.of(entityIds)
                .mapKeyValue((index, entityId) -> getResult(index, entityId, finalEntityVR, resultState))
                .toList();
        return new MassResult<KeyT>(entityResults, mergedValidationResult, resultState);
    }

    @SuppressWarnings("unchecked")
    private ValidationResult mergeWithPrefilterResult(ValidationResult entityVR,
                                                      ValidationResult<?, Defect> prefilterEntityVR,
                                                      List<KeyT> entityIds,
                                                      Class entityClass,
                                                      EntityContext entityContext) {
        Collection<ValidationResult> subResults = entityVR.getSubResults().values();
        Map<Object, ValidationResult> subResultByEntity =
                StreamEx.of(subResults).toMap(ValidationResult::getValue, Function.identity());
        Map<Object, Entity> vrEntities =
                StreamEx.of((List<Entity>) entityVR.getValue()).toMap(Entity::getId, Function.identity());

        Map<Object, Entity> contextEntities = entityContext.getEntities(entityClass);

        Collection<ValidationResult<?, Defect>> perfilterSubResults = prefilterEntityVR.getSubResults().values();
        Map<Object, ValidationResult> prefilterSubResultByEntity =
                StreamEx.of(perfilterSubResults).toMap(ValidationResult::getValue, Function.identity());
        Map<Object, Entity> prefilterEntities =
                StreamEx.of((List<Entity>) prefilterEntityVR.getValue()).toMap(Entity::getId, Function.identity());

        List<Object> mergedValues = new ArrayList<>();
        var mergedVR = new ValidationResult(mergedValues, entityVR.getErrors(), entityVR.getWarnings());
        for (int i = 0; i < entityIds.size(); i++) {
            KeyT entityId = entityIds.get(i);
            Entity entity;
            ValidationResult subResult;
            if (vrEntities.get(entityId) != null) {
                // сначала смотрим в vrEntities для случая, когда копирование не прошло стадию превалидации.
                // в стадии превалидации может происходить подмена родительских или дочерних foreign keys,
                // но при этом в entityContext подмены не будет - расхождение
                // (например в AdGroupCopyPrevalidator#prevalidate происходит установка id кампании)
                entity = vrEntities.get(entityId);
                subResult = subResultByEntity.get(entity);
            } else if (contextEntities.get(entityId) != null) {
                // затем смотрим в contextEntities - маппинг для прошедших префильтрацию и превалидацию энтитей
                entity = contextEntities.get(entityId);
                subResult = subResultByEntity.get(entity);
            } else {
                // и затем смотрим в prefilterEntities - маппинг не прошедших префильтрацию энтитей (их нет в контексте)
                entity = prefilterEntities.get(entityId);
                subResult = prefilterSubResultByEntity.get(entity);
            }
            mergedValues.add(entity);
            if (subResult != null) {
                mergedVR.addSubResult(index(i), subResult);
            }
        }
        return mergedVR;
    }

    private ValidationResult mergeValidationResults(ValidationResult entityVR,
                                                    Map<Class, ValidationResult> validationResults,
                                                    Class entityClass,
                                                    EntityContext entityContext) {
        Deque<Class> processing = new ArrayDeque<>(Set.of(entityClass));
        Set<Class> processed = new HashSet<>();
        Deque<Map.Entry<Class, Class>> hierarchicalRelationships = new ArrayDeque<>();

        while (!processing.isEmpty()) {
            Class current = processing.removeFirst();
            processed.add(current);
            Set<Class> childEntityClasses = new HashSet<>(entityContext.getChildEntityClasses(current));
            childEntityClasses.removeAll(processed);

            for (var childEntityClass : childEntityClasses) {
                hierarchicalRelationships.push(Map.entry(current, childEntityClass));
            }
            processing.addAll(childEntityClasses);
        }

        while (!hierarchicalRelationships.isEmpty()) {
            Map.Entry<Class, Class> relationship = hierarchicalRelationships.pop();
            makeHierarchicalValidationResult(relationship.getKey(), relationship.getValue(),
                    entityContext, validationResults);
        }

        // вернуть ошибку вида adGroupIds[1].Banner[1]
        EntryStream.of(validationResults)
                .filter(e -> e.getKey() != entityClass)
                .filter(e -> !processed.contains(e.getKey()))
                .forEach(e -> entityVR.addSubResult(field(e.getKey().getSimpleName()), e.getValue()));
        return entityVR;
    }

    @SuppressWarnings("unchecked")
    private Result<KeyT> getResult(Integer idx, KeyT entityId, ValidationResult rootValidationResult,
                                   ResultState resultState) {
        ValidationResult entityVR = (ValidationResult) rootValidationResult.getSubResults().get(index(idx));
        var finalResultState = (entityVR == null || !entityVR.hasAnyErrors()) && resultState == ResultState.SUCCESSFUL
                ? ResultState.SUCCESSFUL : ResultState.BROKEN;
        return new Result<>(entityId, entityVR, finalResultState);
    }

    @SuppressWarnings("unchecked")
    private void makeHierarchicalValidationResult(Class parentEntityClass,
                                                  Class childEntityClass,
                                                  EntityContext entityContext,
                                                  Map<Class, ValidationResult> validationResults) {
        ValidationResult childEntitiesVR = validationResults.get(childEntityClass);
        if (childEntitiesVR == null) {
            return;
        }
        // если нет результата валидации (такое бывает в превалидации, когда мы проверяем не все), делаем dummy
        // результат, чтобы он был корнем иерархии ошибок
        ValidationResult parentEntitiesVR = nvl(validationResults.get(parentEntityClass),
                ValidationResult.success(entityContext.getObjects(parentEntityClass)));

        List<? extends Entity<?>> parentEntities = (List<? extends Entity<?>>) parentEntitiesVR.getValue();
        List<Object> parentEntityIds =
                entityContext.getOldIdsByOldEntitiesWithNewIds(parentEntityClass, parentEntities);

        Map<KeyT, List<Object>> parentIdsToChildEntities =
                entityContext.getChildEntitiesByOldParentIds(parentEntityClass, childEntityClass, parentEntityIds);

        Map<?, ValidationResult> childEntityToChildEntityVR = EntryStream.of((List<?>) childEntitiesVR.getValue())
                .invert()
                .mapValues(idx -> (ValidationResult) childEntitiesVR.getSubResults().get(index(idx)))
                .nonNullValues()
                .toCustomMap(IdentityHashMap::new); // тут бывают сущности, которые equals друг другу

        Map<KeyT, ValidationResult> parentIdToChildEntitiesVR = EntryStream.of(parentIdsToChildEntities)
                .mapValues(entities -> {
                    var vr = new ValidationResult(entities);
                    EntryStream.of(entities).filterValues(childEntityToChildEntityVR::containsKey)
                            .forKeyValue((idx, entity) -> vr.addSubResult(index(idx),
                                    childEntityToChildEntityVR.get(entity)));
                    return vr;
                })
                .nonNullValues()
                .toMap();

        EntryStream.of(parentEntityIds)
                .forKeyValue((idx, id) -> {
                    var idVR = parentIdToChildEntitiesVR.get(id);
                    if (idVR == null) {
                        return;
                    }
                    parentEntitiesVR.getOrCreateSubValidationResult(index(idx), parentEntities.get(idx))
                            .addSubResult(field(childEntityClass.getSimpleName()), idVR);
                });
    }
}
