package ru.yandex.direct.core.copyentity;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;

import ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefectIds;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefectIds;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefectIds;
import ru.yandex.direct.core.entity.feed.validation.FeedDefectIds;
import ru.yandex.direct.core.entity.keyword.service.validation.KeywordDefectIds;
import ru.yandex.direct.core.entity.organizations.validation.OrganizationDefectIds;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NET_IS_SWITCHED_OFF;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.Ids.BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NOT_DIFFERENT_PLACES;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.Ids.BID_FOR_SEARCH_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.Ids.BID_FOR_SEARCH_WONT_BE_ACCEPTED_SEARCH_IS_SWITCHED_OFF;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;

@ParametersAreNonnullByDefault
public class CopyValidationResultUtils {

    /**
     * Дефекты, варнинги которых ожидаемы, о таких варнингах в логи писать не хотим
     */
    private static final Set<DefectId<?>> EXPECTED_WARNING_DEFECTS = ImmutableSet.of(
            /*
             * Ставка не будет использоваться, так как на кампании установлена автобюджетная стратегия,
             * либо выключены показы в сетях
             */
            BID_FOR_SEARCH_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY,
            BID_FOR_SEARCH_WONT_BE_ACCEPTED_SEARCH_IS_SWITCHED_OFF,
            BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NET_IS_SWITCHED_OFF,
            BID_FOR_CONTEXT_WONT_BE_ACCEPTED_NOT_DIFFERENT_PLACES,
            BID_FOR_CONTEXT_WONT_BE_ACCEPTED_IN_CASE_OF_AUTOBUDGET_STRATEGY,

            /*
             * Существует кампании, в которых количество баннеров в группе превышает лимит
             * Также, возможно добавить баннер в группу, до того как копирование доберется до баннеров
             */
            BannerDefectIds.Num.MAX_BANNERS_IN_ADGROUP,

            /*
             * Группа могла быть удалена клиентом до того, как копирование добралось до баннеров/фраз/etc
             */
            BannerDefectIds.Gen.AD_GROUP_NOT_FOUND,
            KeywordDefectIds.Gen.AD_GROUP_NOT_FOUND,
            DynamicTextAdTargetDefectIds.Gen.AD_GROUP_NOT_FOUND,

            /*
             * Пользователь мог заказать копирование нескольких пачек кампаний,
             * либо создать кампаний пока копирование не завершилось
             */
            CampaignDefectIds.Nums.MAX_COUNT_OF_CAMPAIGNS_CREATED,

            /*
             * Нельзя скопировать группу, если фид находится в невалидном статусе
             */
            FeedDefectIds.String.FEED_STATUS_WRONG,

            /*
             * Организация могла быть распубликована/удалена
             */
            OrganizationDefectIds.Gen.ORGANIZATION_NOT_FOUND
    );

    private CopyValidationResultUtils() {
    }

    public static <T> ValidationResult<T, Defect> filterDefects(ValidationResult<T, Defect> vr,
                                                                Set<DefectId<?>> whiteList) {
        var errors = StreamEx.of(vr.getErrors()).filter(d -> whiteList.contains(d.defectId())).toList();
        var warnings = StreamEx.of(vr.getWarnings()).filter(d -> whiteList.contains(d.defectId())).toList();
        var subResults = EntryStream.of(vr.getSubResults())
                .mapValues(vri -> filterDefects(vri, whiteList))
                .toMap();

        return new ValidationResult(vr.getValue(), errors, warnings, subResults);
    }

    public static void logFailedResults(ValidationResult<?, Defect> vr, Logger logger) {
        List<List<? extends DefectInfo<Defect>>> failedResults = getFailedResults(vr);
        List<? extends DefectInfo<Defect>> warningDefects = failedResults.get(0);
        List<? extends DefectInfo<Defect>> errorDefects = failedResults.get(1);

        if (!warningDefects.isEmpty() && logger.isWarnEnabled()) {
            logger.warn("While copying entity graph got the following warnings:\n\t{}",
                    StreamEx.of(warningDefects).joining("\n\t"));
        }

        if (!errorDefects.isEmpty() && logger.isErrorEnabled()) {
            logger.error("While copying entity graph got the following errors:\n\t{}",
                    StreamEx.of(errorDefects).joining("\n\t"));
        }

    }

    public static List<List<? extends DefectInfo<Defect>>> getFailedResults(ValidationResult<?, Defect> vr) {
        List<? extends DefectInfo<Defect>> warningDefects = StreamEx.of(vr.flattenWarnings())
                .remove(isExpectedWarningDefect())
                .toList();

        List<? extends DefectInfo<Defect>> errorDefects = vr.flattenErrors();

        List<? extends DefectInfo<Defect>> errorToWarnings = StreamEx.of(errorDefects)
                .filter(isExpectedWarningDefect())
                .toList();

        warningDefects.addAll((List) errorToWarnings);
        errorDefects.removeIf(isExpectedWarningDefect());

        return Arrays.asList(warningDefects, errorDefects);
    }

    @NotNull
    private static Predicate<DefectInfo<Defect>> isExpectedWarningDefect() {
        return defect -> EXPECTED_WARNING_DEFECTS.contains(defect.getDefect().defectId());
    }

    /**
     * Так как сущности добавляются чанками, нам приходится объединять результаты добавления отдельных чанков в общий.
     * В каждом отдельном чанке, результат валидации содержит в себе индексы объектов внутри чанков. В общем
     * результате валидации индексы объектов должны соответствовать индексам в общем списке объектов. Предполагаем,
     * что сервис сущности возвращает результат валидации списка (моделей, индексов, etc). NB: из-за такого
     * объединения ошибки валидации в корне списка могут дублироваться, если они содержатся в нескольких чанках.
     *
     * @param massResults упорядоченный список результатов добавления отдельных чанков сущностей
     * @param <T>         тип результата, необязательно является ключом сущности
     * @return результат добавления всех сущностей
     */
    public static <T> MassResult<T> mergeMassResults(List<MassResult> massResults) {
        //noinspection unchecked
        List<MassResult<T>> typedMassResults = (List) massResults;

        massResults.forEach(massResult -> {
            Object value = massResult.getValidationResult().getValue();
            checkState(value instanceof List, String.format(
                    "Entity service is expected to return List as validation result value, but got %s",
                    ifNotNull(value, Object::getClass)));
        });

        List<?> mergedValidationResultValues = StreamEx.of(massResults)
                .flatCollection(massResult -> (List) massResult.getValidationResult().getValue())
                .toList();

        //noinspection unchecked
        ValidationResult<List<Object>, Defect> mergedValidationResult =
                (ValidationResult) new ValidationResult<>(mergedValidationResultValues);

        AtomicInteger previousElementCount = new AtomicInteger();
        typedMassResults.forEach((MassResult<T> massResult) -> {
            //noinspection unchecked
            ValidationResult<List<Object>, Defect> chunkValidationResult =
                    (ValidationResult<List<Object>, Defect>) massResult.getValidationResult();

            Map<Integer, Integer> destToSourceIndexMap = IntStreamEx.ofIndices(chunkValidationResult.getValue())
                    .mapToEntry(chunkElementIndex -> previousElementCount.get() + chunkElementIndex, i -> i)
                    .toMap();

            // Используем List<Object> вместо List<?> выше, потому что метод требует конкретных типов, которых у нас нет
            mergeSubListValidationResults(mergedValidationResult, chunkValidationResult, destToSourceIndexMap);

            previousElementCount.addAndGet(chunkValidationResult.getValue().size());
        });

        List<Result<T>> mergedResults;
        // Если хотя бы в одном из результатов операций нету результатов элементов, значит их не должно быть и в других
        if (typedMassResults.stream().anyMatch(massResult -> massResult.toResultList() == null)) {
            mergedResults = null;
        } else {
            mergedResults = flatMap(typedMassResults, MassResult::toResultList);
        }

        return new MassResult<>(mergedResults, mergedValidationResult, ResultState.SUCCESSFUL);
    }
}
