package ru.yandex.direct.core.copyentity;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.model.Entity;
import ru.yandex.direct.model.Relationship;

import static java.util.Collections.emptyMap;

/**
 * Содержит следующую информацию о скопированных объектах - класс объекта, id источника (копируемого объекта), id
 * результата (объекта-копии)
 */
@ParametersAreNonnullByDefault
@SuppressWarnings("rawtypes")
public class EntityContext {
    private final CopyOperationContainer copyContainer;

    private final Map<Class<? extends Entity<?>>, Map<Object, ? extends Entity<?>>>
            entities; // class: (old id: entity) у entity  во время копирования может измениться primary идентификатор,
                      // и идентификаторы родителей на новые, но ключ в карте так и останется old id
    private final Map<Class<? extends Entity<?>>, Map<Class<? extends Entity<?>>, Map<Object, Set<Object>>>>
            idRelationships; // parent class: (child class: (parentId: childIds))
    private final Map<Class<? extends Entity<?>>, Map<Object, Object>> copyMappings; // class: (old id: new id)

    private final Set<Class> copied;

    public EntityContext(CopyOperationContainer copyContainer) {
        this.copyContainer = copyContainer;

        copyMappings = EntryStream.of(copyContainer.getCopyMappings())
                .mapValues(m -> (Map<Object, Object>) new HashMap<Object, Object>(m))
                .toCustomMap(HashMap::new);

        entities = new HashMap<>();
        idRelationships = new HashMap<>();
        copied = new HashSet<>();
    }

    /**
     * Отфильтровать из списка уже помеченные id объектов-источников
     *
     * @param entityClass тип помеченных объектов
     * @param sourceIds   список id проверяемых объектов
     * @param <KeyT>      тип id проверяемого объекта
     * @return список еще не помеченных id объектов
     */
    <KeyT> List<KeyT> filterAlreadyPresent(Class entityClass, List<KeyT> sourceIds) {
        Set<Object> alreadyCopied = getCopyMapping(entityClass).keySet();
        return StreamEx.of(sourceIds).filter(id -> !alreadyCopied.contains(id)).toList();
    }


    Map<Object, Object> getCopyMappingsByClass(Class entityClass) {
        return copyMappings.getOrDefault(entityClass, emptyMap());
    }

    /**
     * Возвращает информацию о всех скопированных объектах данного типа
     *
     * @param entityClass тип объекта, для которого запрашивается мэппинг
     * @return мэппинг вида "id объекта-источника" - "id результата"
     */
    @SuppressWarnings("unchecked")
    public <T extends Entity<K>, K> Map<K, K> getCopyMappingByClass(Class<T> entityClass) {
        return (Map<K, K>) getCopyMappingsByClass(entityClass);
    }

    @SuppressWarnings("unchecked")
    <T extends Entity<KeyT>, KeyT> void addObjects(Class<T> entityClass, Collection<T> entitiesToAdd) {
        var mapping = getCopyMapping(entityClass);
        Map entitiesOfClass = getEntities(entityClass);
        for (T entity : entitiesToAdd) {
            KeyT id = entity.getId();
            mapping.put(id, null);
            entitiesOfClass.put(id, entity);
        }
    }

    /**
     * Возврщает список классов сущностей, которые содержатся в этом контексте
     */
    public Set<Class<? extends Entity<?>>> getEntityClasses() {
        return entities.keySet();
    }

    public Set<Class<? extends Entity<?>>> getChildEntityClasses(Class<? extends Entity<?>> parentEntityClass) {
        Map<Class<? extends Entity<?>>, Map<Object, Set<Object>>> relationshipsOfEntity =
                idRelationships.get(parentEntityClass);
        if (relationshipsOfEntity == null) {
            return Set.of();
        }
        return EntryStream.of(relationshipsOfEntity)
                .filterValues(r -> !r.isEmpty())
                .keys()
                .toSet();
    }

    /**
     * Возвращает все сущности в этом контексте определенного класса
     */
    @SuppressWarnings("unchecked")
    public <T extends Entity<K>, K> List<T> getObjects(Class<T> entityClass) {
        return EntryStream.of(getEntities(entityClass))
                .sortedBy(ke -> (Comparable) ke.getKey())
                .values()
                .toList();
    }

    /**
     * Получает скопированные сущности-дети по id родителей
     *
     * @param parentEntityClass класс родителя
     * @param childEntityClass  класс ребенка
     * @param parentEntityIds   id родителей
     * @param <ChildT>          тип ребенка
     * @param <ParentKeyT>      тип id родителя
     * @return мапа "id родителя - список сущностей копий его детей"
     */
    public <ChildT extends Entity<TChildKey>, ParentKeyT, TChildKey> Map<ParentKeyT, List<ChildT>> getChildEntitiesByOldParentIds(
            Class<? extends Entity<?>> parentEntityClass, Class<ChildT> childEntityClass,
            Collection<ParentKeyT> parentEntityIds) {
        Map<Object, Set<Object>> oldParentIdsToOldChildIds = getIdRelationship(parentEntityClass, childEntityClass);
        Map<TChildKey, ChildT> oldChildIdsToChildEntities = getEntities(childEntityClass);
        return StreamEx.of(parentEntityIds).mapToEntry(oldParentIdsToOldChildIds::get)
                .nonNullValues()
                .mapValues(childIds -> StreamEx.of(childIds)
                        .map(oldChildIdsToChildEntities::get)
                        .select(childEntityClass)
                        .toList())
                .toMap();
    }

    @SuppressWarnings("unchecked")
    public <ParentT extends Entity<ParentKeyT>, ChildT extends Entity<?>, ParentKeyT> void setParentIds(
            Class<ChildT> entityClass, Relationship<ParentT, ChildT, ParentKeyT> relationship) {
        entities.get(entityClass).values().forEach(e -> setForeignKeyId(relationship, (ChildT) e));
    }

    public <ParentT extends Entity<ParentKeyT>, ChildT extends Entity<?>, ParentKeyT> void setParentIds(
            Collection<ChildT> entities, Relationship<ParentT, ChildT, ParentKeyT> relationship) {
        entities.forEach(e -> setForeignKeyId(relationship, e));
    }

    <T extends Entity<K>, K> List<T> startCopying(Class<T> entityClass) {
        if (copied.contains(entityClass)) {
            return List.of();
        }

        copied.add(entityClass);
        return getObjects(entityClass);
    }

    <T extends Entity<K>, K> void setIdsToNull(Class<T> entityClass) {
        if (!entities.containsKey(entityClass)
                // не меняем imageHash у BannerImageFormat
                || BannerImageFormat.class.isAssignableFrom(entityClass)) {
            return;
        }
        getEntities(entityClass).values().forEach(entity -> entity.setId(null));
    }

    /**
     * Добавляет информацию о скопированных объектах
     *
     * @param entityClass класс скопированных объектов
     */
    <T extends Entity<TKey>, TKey> void finishCopying(Class<T> entityClass) {
        Map<TKey, TKey> copyMappingsToAdd = Maps.transformValues(getEntities(entityClass), Entity::getId);
        getCopyMapping(entityClass).putAll(copyMappingsToAdd);
        copyContainer.onFinishCopying(entityClass, copyMappingsToAdd);
        copied.add(entityClass);
    }

    /**
     * Получает id исходных объектов по объектам после копирования (это те же самые объекты, равные по ссылке,
     * но, возможно, с новыми primary идентификатором или идентификаторами родителей
     *
     * @param entityClass           класс копируемых объектов
     * @param oldEntitiesWithNewIds исходные объекты после копирования, с, возможно, с новыми primary идентификатором
     *                              или идентификаторами родителей
     * @return список id исходных объектов
     */
    public List<Object> getOldIdsByOldEntitiesWithNewIds(
            Class<? extends Entity<?>> entityClass, List<?> oldEntitiesWithNewIds) {
        Map<Object, ? extends Entity<?>> idToEntity = entities.get(entityClass);
        IdentityHashMap<? extends Entity<?>, Object> entityToId =
                EntryStream.of(idToEntity).invert().toCustomMap(IdentityHashMap::new);
        return StreamEx.of(oldEntitiesWithNewIds).map(entityToId::get).toList();
    }

    /**
     * Регистрирует в контексте отношение "id родителя - id потомоков"
     *
     * @param childEntities сущности-потомки (содержат и id родителя в виде внешнего ключа, и id потомка в виде
     *                      первичного ключа)
     * @param relationship  отношение, которое регистрируется в контексте
     * @param <T>           тип сущности-потомка
     */
    public <T extends Entity<?>> void registerIdRelationship(
            Collection<T> childEntities,
            Relationship<? extends Entity<?>, T, ?> relationship) {
        var idRelationship =
                getIdRelationship(relationship.getParentEntityClass(), relationship.getChildEntityClass());
        childEntities.forEach(e ->
                idRelationship.computeIfAbsent(relationship.getParentId(e), pid -> new HashSet<>()).add(e.getId()));
    }

    /**
     * Получает id объекта-копии, если он есть
     *
     * @param entityClass класс скопированного объекта
     * @param sourceId    id объекта-источника
     * @return id объекта-копии, или null, если его нет (источник еще не был скопирован)
     */
    private <ParentT extends Entity<ParentKeyT>, ParentKeyT> ParentKeyT
    getDestinationId(Class<ParentT> entityClass, ParentKeyT sourceId) {
        return (ParentKeyT) getCopyMapping(entityClass).get(sourceId);
    }

    private <ParentT extends Entity<ParentKeyT>, ChildT extends Entity<?>, ParentKeyT> void setForeignKeyId(
            Relationship<ParentT, ChildT, ParentKeyT> relationship, ChildT entity) {
        ParentKeyT oldId = relationship.getParentId(entity);
        ParentKeyT newId = getDestinationId(relationship.getParentEntityClass(), oldId);
        relationship.setParentId(entity, newId);
    }

    @SuppressWarnings("unchecked")
    public <T extends Entity<K>, K> Map<K, T> getEntities(Class<T> entityClass) {
        return (Map) entities.computeIfAbsent(entityClass, cls -> new HashMap<>());
    }

    public <T extends Entity> Map<Object, T> getEntitiesUntyped(Class<T> entityClass) {
        return getEntities(entityClass);
    }

    @SuppressWarnings("unchecked")
    public <K, V> Map<K, Set<V>> getIdRelations(
            Class<? extends Entity<?>> parentEntityClass, Class<? extends Entity<?>> childEntityClass) {
        return (Map) getIdRelationship(parentEntityClass, childEntityClass);
    }

    private Map<Object, Set<Object>> getIdRelationship(
            Class<? extends Entity<?>> parentEntityClass, Class<? extends Entity<?>> childEntityClass) {
        return idRelationships.computeIfAbsent(parentEntityClass, cls -> new HashMap<>())
                .computeIfAbsent(childEntityClass, cls -> new HashMap<>());
    }

    private <ParentT extends Entity<?>> Map<Object, Object> getCopyMapping(Class<ParentT> entityClass) {
        return copyMappings.computeIfAbsent(entityClass, meta -> new HashMap<>());
    }

    /**
     * Удаляет из графа объекты, не имеющие потомков. Если передать в этот метод идентификаторы объектов,
     * у которых потомки есть - выбросится исключение.
     *
     * @param entityClass класс сущности
     * @param idsToRemove идентификаторы сущностей, которые нужно удалить.
     */
    public void removeLeafs(Class<? extends Entity<?>> entityClass, Set<Object> idsToRemove) {
        // проверим, что все переданные объекты действительно не имеют потомков
        Map<Class<? extends Entity<?>>, Map<Object, Set<Object>>> parentMappings = idRelationships.get(entityClass);
        if (parentMappings != null) {
            Set<Object> emptySet = new HashSet<>();
            List<Object> idsWithChildren = idsToRemove.stream()
                    .filter(id -> parentMappings.entrySet().stream()
                            .anyMatch(entry -> !entry.getValue().getOrDefault(id, emptySet).isEmpty())
                    ).collect(Collectors.toSet()).stream()
                    .sorted()
                    .collect(Collectors.toList());
            if (!idsWithChildren.isEmpty()) {
                throw new IllegalArgumentException(String.format(
                        "Objects with passed ids must not have children, but object with ids: %s have them",
                        idsWithChildren));
            }
        }

        for (Map<Class<? extends Entity<?>>, Map<Object, Set<Object>>> childrenMap : idRelationships.values()) {
            Map<Object, Set<Object>> parentToChildrenMapping = childrenMap.get(entityClass);
            if (parentToChildrenMapping == null) {
                continue;
            }
            Iterator<Map.Entry<Object, Set<Object>>> parentToChildrenMappingIterator =
                    parentToChildrenMapping.entrySet().iterator();
            while (parentToChildrenMappingIterator.hasNext()) {
                Map.Entry<Object, Set<Object>> entry = parentToChildrenMappingIterator.next();
                Set<Object> childrenIds = entry.getValue();
                childrenIds.removeAll(idsToRemove);
                // Если удалили вообще всех детей, то удаляем и мэппинг с родителем
                if (childrenIds.isEmpty()) {
                    parentToChildrenMappingIterator.remove();
                }
            }
            // А если мы удалили все мэппинги родительского класса с entityClass, то удаляем и связь между ними из карты
            if (parentToChildrenMapping.isEmpty()) {
                childrenMap.remove(entityClass);
            }
        }

        // Удаляем объекты из маппировки по идентификаторам
        Map<Object, ? extends Entity<?>> idToEntityMap = entities.get(entityClass);
        if (idToEntityMap != null) {
            idsToRemove.forEach(idToEntityMap::remove);
            if (idToEntityMap.isEmpty()) {
                entities.remove(entityClass);
            }
        }

        // Удаляем объекты из маппировки старых идентификаторов на новые
        Map<Object, Object> oldIdToNewIdMap = copyMappings.get(entityClass);
        if (oldIdToNewIdMap != null) {
            idsToRemove.forEach(oldIdToNewIdMap::remove);
            if (oldIdToNewIdMap.isEmpty()) {
                copyMappings.remove(entityClass);
            }
        }
    }

    public void restoreNullIds() {
        for (Map.Entry<Class<? extends Entity<?>>, Map<Object, ? extends Entity<?>>> entry: entities.entrySet()) {
            Map<Object, ? extends Entity<?>> entityById = entry.getValue();
            for (Map.Entry<Object, ? extends Entity<?>> idAndEntity: entityById.entrySet()) {
                Entity entity = idAndEntity.getValue();
                if (entity.getId() == null) {
                    Object oldId = idAndEntity.getKey();
                    entity.setId(oldId);
                }
            }
        }
    }

    public CopyOperationContainer getCopyContainer() {
        return copyContainer;
    }
}
