package ru.yandex.direct.core.copyentity;

import java.lang.reflect.TypeVariable;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.reflect.TypeToken;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.copyentity.exception.MultipleServicesFoundException;
import ru.yandex.direct.core.copyentity.exception.NoServiceFoundException;
import ru.yandex.direct.core.copyentity.exception.NotDerivedFromEntityException;
import ru.yandex.direct.model.Entity;
import ru.yandex.direct.model.Relationship;

import static ru.yandex.direct.core.copyentity.TypeUtils.cast;

/**
 * Вспомогательный класс для обхода графа сущностей, содержит необходимые методы для получения сущностей, связей,
 * и их сервисов. Все сущности внутри графа приводятся к непосредственному потомку {@link Entity}
 * методом {@link EntityGraphNavigator#getClosestToEntityAncestor(Class)}. Все отношения (связи) между сущностями
 * проксируются классом {@link EntityRelationshipProxy}, который так же приводит оригинальные результаты методов
 * {@link Relationship#getParentEntityClass()} и {@link Relationship#getChildEntityClass()}
 * к непосредственным потомкам {@link Entity} (тем же методом
 * {@link EntityGraphNavigator#getClosestToEntityAncestor(Class)}). Таким образом, граф всех объектов и связей ядра
 * схлопывается до непосредственных потомков {@link Entity} и связей между ними.
 * При этом класс прокси-связи {@link EntityRelationshipProxy} учитывает в методах
 * {@link EntityRelationshipProxy#getParentId(Entity)} и {@link EntityRelationshipProxy#setParentId(Entity, Object)},
 * какие были классы объектов родителя и ребенка в оригинальной связи до схлопывания, и работает с идентификаторами,
 * только если переданные в методы объекты являются потомками изначальных классов. Учитывать это важно,
 * так как в эти методы могут передаваться предки изначальных классов (которые по иерархии находятся между
 * оригинальным классом и {@link Entity}), с которыми связь работать не должна.
 * Так же, если планируется работать со связями графа необходимо помнить, что они схлопнутые, а оригинальные связи
 * между сущностями можно получить методом {@link EntityRelationshipProxy#getInnerRelationship()}
 */
@ParametersAreNonnullByDefault
@Component
public class EntityGraphNavigator {
    private final LoadingCache<Class<? extends Entity<?>>, Class<? extends Entity<?>>>
            entitiesAncestorsCache; // класс сущности: класс её предка - непосредственного потомка Entity
    private final Map<Class<? extends Entity<?>>, EntityService<? extends Entity<?>, ?>>
            entityAncestorsServices; // класс непосредственного потомка Entity: сервис по работе с ним и его потомками
    private final Map<Class<? extends Relationship<? extends Entity<?>, ? extends Entity<?>, ?>>,
            RelationshipService<? extends Relationship<? extends Entity<?>, ? extends Entity<?>, ?>, ?, ?>>
            relationshipServices; // класс отношения (связи): сервис по работе с этим отношениями
    private final
            Map<Class<? extends Entity<?>>, List<EntityRelationshipProxy<? extends Entity<?>, ? extends Entity<?>, ?>>>
            parentToChildRelationshipProxies; // класс непосредственного потомка Entity:
                                              // список прокси-связей к непосредственным потомкам Entity его детей
    private final
            Map<Class<? extends Entity<?>>, List<EntityRelationshipProxy<? extends Entity<?>, ? extends Entity<?>, ?>>>
            childToParentRelationshipProxies; // класс непосредственного потомка Entity:
                                              // список прокси-связей к непосредственным потомкам Entity его родителей

    @Autowired
    public EntityGraphNavigator(Set<EntityService> entityBeans, Set<RelationshipService> relationshipBeans) {
        entitiesAncestorsCache = CacheBuilder.newBuilder()
                .build(new CacheLoader<>() {
                    @Override
                    public Class<? extends Entity<?>> load(Class<? extends Entity<?>> key) {
                        return calculateEntityClass(key);
                    }
                });

        // cast нужен, так как даже если указать полные типы в параметрах entityBeans и relationshipBeans,
        // то java все равно не может определить тип ключа карты, так как этот класс формируется через рефлексию
        // внутри метода getEntityGraphServices и снаружи не зацепляется
        this.entityAncestorsServices = cast(getEntityGraphServices(entityBeans, EntityService.class));
        this.relationshipServices = cast(getEntityGraphServices(relationshipBeans, RelationshipService.class));

        List<EntityRelationshipProxy<? extends Entity<?>, ? extends Entity<?>, ?>>
                entityRelationshipProxies = relationshipServices.keySet().stream()
                .map(TypeUtils::silentCreateInstance)
                .filter(Objects::nonNull)
                .map(relationship -> new EntityRelationshipProxy<>(
                        relationship,
                        getClosestToEntityAncestor(relationship.getParentEntityClass()),
                        getClosestToEntityAncestor(relationship.getChildEntityClass())))
                .collect(Collectors.toList());

        parentToChildRelationshipProxies = entityRelationshipProxies.stream()
                .collect(Collectors.groupingBy(Relationship::getParentEntityClass));

        childToParentRelationshipProxies = entityRelationshipProxies.stream()
                .collect(Collectors.groupingBy(Relationship::getChildEntityClass));
    }

    /**
     * Для заданного класса находит его предка, являющегося непосредственным потомком {@link Entity}.
     * Метод работает через кэш, кэш получает данные методом {@link EntityGraphNavigator#calculateEntityClass(Class)}
     *
     * @param derived класс, у которого в предках есть интерфейс {@link Entity}
     * @return непосредственный потомок {@link Entity} (тип, его реализующий или расширяющий)
     */
    public Class<? extends Entity<?>> getClosestToEntityAncestor(Class<? extends Entity<?>> derived) {
        return entitiesAncestorsCache.getUnchecked(derived);
    }

    /**
     * По классу сущности получает сервис, работающий с его непосредственным наследником {@link Entity}.
     * Например, для класса TextBanner (наследник BannerWithAdGroupId) будет получен BannerService,
     * реализующий интерфейс EntityService&lt;BannerWithAdGroupId&gt;
     */
    public <T extends Entity<TKey>, TKey> EntityService<T, TKey> getEntityService(
            Class<? extends Entity<?>> derivedEntityClass) {
        var entityClass = getClosestToEntityAncestor(derivedEntityClass);
        if (!entityAncestorsServices.containsKey(entityClass)) {
            throw new NoServiceFoundException("No entity service for " + entityClass + " found.");
        }
        // Приведение типов нужно, так как entityAncestorsServices по ключу с типом Class<T> совершенно не обязательно
        // должен лежать сервис с типом EntityService<T, TKey>. Сервис теоретически может быть наследником
        // типа EntityService с любыми внутренними классами-параметрами, а не только T и TKey.
        // Поэтому приведение типов необходимо.
        return (EntityService<T, TKey>) entityAncestorsServices.get(entityClass);
    }

    /**
     * По классу связи получает ее сервис
     */
    public <TParent extends Entity<TParentKey>, TChild extends Entity<?>, TParentKey>
    RelationshipService<? extends Relationship<TParent, TChild, TParentKey>, TParentKey, ?>
    getRelationshipServiceByClass(Class<? extends Relationship<TParent, TChild, TParentKey>> relationshipClass) {
        if (!relationshipServices.containsKey(relationshipClass)) {
            throw new NoServiceFoundException("No relationship service for " + relationshipClass + " found.");
        }
        // Приведение типов нужно, так как relationshipServices по ключу с типом Class<TRelationship> совершенно
        // не обязательно должен лежать сервис с возвращаемым типом. Сервис теоретически может быть наследником
        // RelationshipService с любыми внутренними классами-параметрами, поэтому приведение типов необходимо.
        return (RelationshipService<? extends Relationship<TParent, TChild, TParentKey>, TParentKey, ?>)
                relationshipServices.get(relationshipClass);
    }

    /**
     * По объекту связи получает ее сервис. Если передана объект прокси-связи {@link EntityRelationshipProxy},
     * то вернет сервис для оригинальной связи {@link EntityRelationshipProxy#getInnerRelationship()}
     */
    public <TParent extends Entity<TParentKey>, TChild extends Entity<?>, TParentKey>
    RelationshipService<? extends Relationship<TParent, TChild, TParentKey>, TParentKey, ?>
    getRelationshipServiceByInstance(Relationship<TParent, TChild, TParentKey> relationship) {
        Relationship<TParent, TChild, TParentKey> realRelationship =
                relationship instanceof EntityRelationshipProxy ?
                ((EntityRelationshipProxy<TParent, TChild, TParentKey>) relationship).innerRelationship
                : relationship;
        // getClass() округляет класс до Class<? extends Relationship>, поэтому нужно приведение типов
        Class<? extends Relationship<TParent, TChild, TParentKey>> relationshipClass =
                (Class<? extends Relationship<TParent, TChild, TParentKey>>) realRelationship.getClass();
        return getRelationshipServiceByClass(relationshipClass);
    }

    /**
     * Отдает прокси-связи родитель->ребенок для родительской сущности TParent. TParent должен быть
     * непосредственным потомком {@link Entity}
     *
     * @param parentType класс - непосредственный потомок {@link Entity}
     * @return набор прокси-связей родителя с детьми
     */
    public <TParent extends Entity<TKey>, TKey>
    List<EntityRelationshipProxy<TParent, ? extends Entity<?>, TKey>>
    getParentToChildEntityRelationships(Class<TParent> parentType) {
        checkIsClassClosestEntityAncestor(parentType);

        // Приведение типов нужно, так как parentToChildRelationshipProxies по ключу с типом Class<TParent> совершенно
        // не обязательно должен лежать список прокси-связей EntityRelationshipProxy с требуемыми классами-параметрами.
        // Прокси-связи теоретически могут быть наследниками типа EntityRelationshipProxy с любыми внутренними
        // классами-параметрами, поэтому приведение типов необходимо.
        return (List<EntityRelationshipProxy<TParent, ?, TKey>>)
                (List)parentToChildRelationshipProxies.getOrDefault(parentType, Collections.emptyList());
    }

    /**
     * Отдает прокси-связи ребенок->родитель для дочерней сущности TParent. TParent должен быть
     * непосредственным потомком {@link Entity}
     *
     * @param childType класс - непосредственный потомок {@link Entity}
     * @return набор связей с ребенка с родителями
     */
    public <TChild extends Entity<?>> List<? extends EntityRelationshipProxy<? extends Entity<?>, TChild, ?>>
    getChildToParentEntityRelationships(Class<TChild> childType) {
        checkIsClassClosestEntityAncestor(childType);
        // Приведение типов нужно, так как childToParentRelationshipProxies по ключу с типом Class<TChild> совершенно
        // не обязательно должен лежать список прокси-связей EntityRelationshipProxy с требуемыми классами-параметрами.
        // Прокси-связи теоретически могут быть наследниками типа EntityRelationshipProxy с любыми внутренними
        // классами-параметрами, поэтому приведение типов необходимо.
        return (List<? extends EntityRelationshipProxy<?, TChild, ?>>)
                (List)childToParentRelationshipProxies.getOrDefault(childType, Collections.emptyList());
    }

    private <T extends Entity<?>> void checkIsClassClosestEntityAncestor(Class<T> pretender) {
        Class<? extends Entity<?>> closestToEntityAncestor = getClosestToEntityAncestor(pretender);
        if (!pretender.equals(closestToEntityAncestor))
            throw new IllegalArgumentException(String.format(
                    "%s is not a closest to Entity ancestor class, closest to Entity ancestor class is %s",
                    pretender, closestToEntityAncestor));
    }

    /**
     * Для наследника параметризованного класса (дженерика) достает класс его первого типового параметра, и делает
     * карту класс первого параметра: наследник параметризованного класса. Например, для
     * <code>services = Set.of(ArrayList&lt;String&gt;, ArrayList&lt;Long&gt)</code> и
     * <code>serviceClass = List.class</code> вернет
     * <code>
     * Map.of(
     *     Class&lt;String&gt;, ArrayList&lt;String&gt,
     *     Class&lt;Long&gt;, ArrayList&lt;Long&gt)
     * </code>
     * @param services список наследников параметризованного класса
     * @param serviceClass конкретный параметризованный класс
     * @param <T> Тип наследника параметризованного класса
     * @param <TService> Тип параметризованного класса
     * @param <TInner> тип его первого типового параметра
     * @return карта отображения объектов на класс первого параметра дженерика
     */
    private <T extends TService, TInner, TService> Map<Class<TInner>, T>
    getEntityGraphServices(Set<T> services, Class<TService> serviceClass) {
        TypeVariable<Class<TService>> serviceFirstTypedParameter = serviceClass.getTypeParameters()[0];
        Map<Class<TInner>, List<T>> entityServiceGroupings = services.stream()
                .collect(Collectors.groupingBy(
                        service -> getServiceFirstTypeParameterClass(service, serviceFirstTypedParameter)));

        Map<Class<TInner>, List<Object>> entityMultiServices = EntryStream.of(entityServiceGroupings)
                .filterValues(ms -> ms.size() > 1)
                .mapValues(List::<Object>copyOf)
                .toMap();
        if (!entityMultiServices.isEmpty()) {
            throw new MultipleServicesFoundException(formMessage(entityMultiServices));
        }

        return EntryStream.of(entityServiceGroupings).mapValues(esg -> esg.get(0)).toMap();
    }

    /**
     * Формирует сообщение о том, что для одного класса сущности есть несколько сервисов с ней работающих
     */
    private <T> String formMessage(Map<Class<T>, List<Object>> multiServicesMap) {
        return "Several services for the following entities or relationships found: \n" +
                EntryStream.of(multiServicesMap)
                        .map(e -> e.getKey().getName()
                                + " : "
                                + StreamEx.of(e.getValue()).map(es -> es.getClass().getName()).joining(", "))
                        .joining(";\n");
    }

    /**
     * Для заданного класса находит его предка, являющегося непосредственным потомком {@link Entity},
     * (то есть классом, напрямую реализующим или расширяющим интерфейс {@link Entity}).
     *
     * @param derivedEntityClass класс, у которого в предках есть интерфейс {@link Entity}
     * @return непосредственный потомок {@link Entity} (тип, его реализующий или расширяющий)
     */
    private static Class<? extends Entity<?>> calculateEntityClass(Class<? extends Entity<?>> derivedEntityClass) {
        List<Class<? extends Entity<?>>> classesImplementingEntity =
                cast(TypeUtils.findAncestorsImplementing(derivedEntityClass, Entity.class));
        if (classesImplementingEntity.isEmpty()) {
            throw new NotDerivedFromEntityException(derivedEntityClass);
        }

        return classesImplementingEntity.get(0);
    }

    /**
     * Возвращает основной класс сущности сервиса. Для сервиса реализующего {@link EntityService} -
     * реализацию {@link Entity}, с которой он работает. Для сервиса, реализующего {@link RelationshipService} -
     * реализацию {@link Relationship} с которой он работает
     */
    @SuppressWarnings("UnstableApiUsage")
    private static <TService, TCastClass> Class<TCastClass> getServiceFirstTypeParameterClass(
            TService service, TypeVariable<Class<TService>> firstTypedParameter) {
        TypeToken<?> typeToken = TypeToken.of(service.getClass());
        return (Class<TCastClass>) typeToken.resolveType(firstTypedParameter).getType();
    }

    /**
     * Класс прокси-связи, схлопывающий оригинальное отношение между сущностями до связи между их предками,
     * являющимися непосредственными потомками {@link Entity}. Внутри себя содержит оригинальную связь
     * @param <TParent> тип сущности класса-родителя
     * @param <TChild> тип сущности класса-ребенка
     * @param <TParentKey> тип первичного ключа родителя
     */
    public static class
    EntityRelationshipProxy<TParent extends Entity<TParentKey>, TChild extends Entity<?>, TParentKey>
            implements Relationship<TParent, TChild, TParentKey> {
        private final Class<? extends Entity<?>> parentEntityClass;
        private final Class<? extends Entity<?>> childEntityClass;
        private final Relationship<TParent, TChild, TParentKey> innerRelationship;

        /**
         * Создает прокси-связь
         *
         * @param relationship - оригинальное отношение между сущностями
         * @param parentEntityClass непосредственный потомок {@link Entity} для типа TParent. Должен получаться вызовом
         *           <code>EntityGraphNavigator.getClosestToEntityAncestor(relationship.getParentEntityClass())</code>
         * @param childEntityClass непосредственный потомок {@link Entity} для типа TChild. Должен получаться вызовом
         *           <code>EntityGraphNavigator.getClosestToEntityAncestor(relationship.getChildEntityClass())</code>
         */
        public EntityRelationshipProxy(
                Relationship<TParent, TChild, TParentKey> relationship,
                Class<? extends Entity<?>> parentEntityClass,
                Class<? extends Entity<?>> childEntityClass) {
            this.innerRelationship = relationship;
            this.parentEntityClass = parentEntityClass;
            this.childEntityClass = childEntityClass;
        }

        /**
         * Получает идентификатор сущности-родителя из экземпляра сущности потомка. Так как все сущности в графе
         * схлопнуты до непосредственных потомков {@link Entity}, может так получиться, что в метод будет передан
         * экземпляр класса предка {@link TChild} (класс в иерархии выше {@link Entity}, но ниже {@link TChild}).
         * Поэтому метод проверяет, действительно ли класс переданного экземпляра является наследником оригинального
         * класса ребенка {@link TChild}, и только в этом случае возвращает идентификатор родителя, иначе null
         * (в противном случае возникла бы ошибка времени выполнения).
         *
         * @param child объект сущности-ребенка
         * @return идентифкатор сущности-родителя, или null, если класс сущности-ребенка является
         * предком {@link TChild}, а не его наследником.
         */
        @Override
        public TParentKey getParentId(TChild child) {
            if (innerRelationship.getChildEntityClass().isAssignableFrom(child.getClass())) {
                return innerRelationship.getParentId(child);
            }
            return null;
        }

        /**
         * Устанавливает идентификатор сущности-родителя в экземпляре сущности-ребенка. Так как все сущности в графе
         * схлопнуты до непосредственных потомков {@link Entity}, может так получиться, что в метод будет передан
         * экземпляр класса предка {@link TChild} (класс в иерархии выше {@link Entity}, но ниже {@link TChild}).
         * Поэтому метод проверяет, действительно ли класс переданного экземпляра является наследником оригинального
         * класса ребенка {@link TChild}, и только в этом случае проставляет идентификатор родителя, (в противном
         * случае возникла бы ошибка времени выполнения, так как методов установки идентификатора родителя
         * у предков {@link TChild} может просто не быть).
         *
         * @param child сущность-потомок
         * @param id идентифкатор сущности-родителя
         */
        @Override
        public void setParentId(TChild child, TParentKey id) {
            if (innerRelationship.getChildEntityClass().isAssignableFrom(child.getClass())) {
                innerRelationship.setParentId(child, id);
            }
        }

        /**
         * Вместо оригинального класса {@link TParent} отдает его предка - непосредственного потомка {@link Entity}.
         * Так как граф сущностей содержит только непосредственных потомков {@link Entity} и отношения между ними,
         * имело смысл вынести всю логику по приведению в отдельный класс. Если возвращать оригинальные классы,
         * то каждый раз в клиентах {@link EntityGraphNavigator} пришлось бы подменять их на непосредственных
         * потомков {@link Entity}, что неудобно. При этом, оригинальный класс родителя можно получить методом
         * <code>getInnerRelationship().getParentEntityClass()</code>
         *
         * @return Предок {@link TParent}, являющийся непосредственным потомком {@link Entity}. Нужно учитывать,
         * что возвращаемый класс не соответствует сигнатуре метода.
         */
        @Override
        public Class<TParent> getParentEntityClass() {
            return cast(parentEntityClass);
        }

        /**
         * Вместо оригинального класса {@link TChild} отдает его предка - непосредственного потомка {@link Entity}.
         * Так как граф сущностей содержит только непосредственных потомков {@link Entity} и отношения между ними,
         * имело смысл вынести всю логику по приведению в отдельный класс. Если возвращать оригинальные классы,
         * то каждый раз в клиентах {@link EntityGraphNavigator} пришлось бы подменять их на непосредственных
         * потомков {@link Entity}, что неудобно. При этом, оригинальный класс ребенка можно получить методом
         * <code>getInnerRelationship().getChildEntityClass()</code>
         *
         * @return Предок {@link TChild}, являющийся непосредственным потомком {@link Entity}. Нужно учитывать,
         * что возвращаемый класс не соответствует сигнатуре метода.
         */
        @Override
        public Class<TChild> getChildEntityClass() {
            return cast(childEntityClass);
        }

        /**
         * Возвращает оригинальное отношение между сущностями
         * @return оригинальное отношение между сущностями
         */
        public Relationship<TParent, TChild, TParentKey> getInnerRelationship() {
            return innerRelationship;
        }

    }
}
