package ru.yandex.direct.core.copyentity;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.prefilters.Prefilter;
import ru.yandex.direct.core.entity.CampaignLibraryObject;
import ru.yandex.direct.core.entity.ClientLibraryObject;
import ru.yandex.direct.core.entity.ShardLibraryObject;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.model.Entity;
import ru.yandex.direct.model.Relationship;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.copyentity.TypeUtils.cast;

@Service
@ParametersAreNonnullByDefault
public class EntityLoadService {

    private static final Logger logger = LoggerFactory.getLogger(EntityLoadService.class);

    public static final int SELECT_CHUNK_SIZE = SqlUtils.TYPICAL_SELECT_CHUNK_SIZE;

    private final EntityGraphNavigator navigator;
    private final Prefilter prefilter;

    @Autowired
    public EntityLoadService(EntityGraphNavigator navigator, Prefilter prefilter) {
        this.navigator = navigator;
        this.prefilter = prefilter;
    }

    public <T extends Entity<KeyT>, KeyT> Map<Class, ValidationResult<?, Defect>> loadGraph(
            Class<T> entityClass,
            List<KeyT> ids,
            EntityContext entityContext
    ) {
        Map<Class, ValidationResult<?, Defect>> prefilterResults = new HashMap<>();
        get(entityClass, ids, entityContext, prefilterResults);
        return prefilterResults;
    }

    private <T extends Entity<KeyT>, KeyT> void get(
            Class<T> entityClass,
            List<KeyT> ids,
            EntityContext entityContext,
            Map<Class, ValidationResult<?, Defect>> prefilterResults) {
        List<KeyT> toGet = entityContext.filterAlreadyPresent(entityClass, ids);
        if (toGet.isEmpty()) {
            return;
        }

        CopyOperationContainer copyContainer = entityContext.getCopyContainer();

        try (var ignored = Trace.current().profile("entityGraph:get", entityClass.getSimpleName(), ids.size())) {
            logger.info("Getting {} - {} entities", entityClass.getName(), ids.size());

            EntityService<T, KeyT> service = navigator.getEntityService(entityClass);
            List<T> entities = service.getChunked(copyContainer.getClientIdFrom(), copyContainer.getOperatorUid(), toGet, SELECT_CHUNK_SIZE);
            ValidationResult<List<T>, Defect> vr = this.prefilter.prefilter(entityClass, entities, copyContainer);
            entities = ValidationResult.getValidItems(vr);
            prefilterResults.put(entityClass, vr);
            if (entities.isEmpty()) {
                return;
            }
            Set<KeyT> filteredIds = StreamEx.of(entities).map(Entity::getId).toSet();

            entityContext.addObjects(entityClass, entities);

            // получить сущности-родители (thisEntity.parentEntityId = parentEntity.id)
            for (var relationship : getChildToParentRelationships(entityClass, copyContainer)) {
                entityContext.registerIdRelationship(entities, relationship);
                relationship.consumeNotNullParentIds(entities, (parentEntityClass, parentIds) ->
                        get(parentEntityClass, parentIds, entityContext, prefilterResults));
            }

            // получить сущности-дети (thisEntity.id = childEntity.thisEntityId)
            for (var relationship : getParentToChildRelationships(entityClass, copyContainer)) {
                RelationshipService<? extends Relationship<T, ? extends Entity<?>, KeyT>, KeyT, ?> relationshipService =
                        navigator.getRelationshipServiceByInstance(relationship);
                Set<?> childEntityIds =
                        relationshipService.getChildEntityIdsByParentIds(copyContainer.getClientIdFrom(),
                                copyContainer.getOperatorUid(), filteredIds);
                List<?> sortedChildEntityIds = StreamEx.of(childEntityIds).sorted().toList();
                // TODO: обойтись без каста можно попробовать, добавив класс ключа ребенка
                //  ChildKeyT в Relationship, связав его с классом ребенка как ChildT extends Entity<ChildKeyT>,
                //  и затем обернув вызов get() в лямбду, как в примере выше, только не в связи, а в сервисе связи
                get(cast(relationship.getChildEntityClass()), sortedChildEntityIds, entityContext, prefilterResults);
            }
        }
    }

    public <TChild extends Entity<?>> List<EntityGraphNavigator.EntityRelationshipProxy<? extends Entity<?>, TChild, ?>>
    getChildToParentRelationships(Class<TChild> entityClass, CopyOperationContainer copyConfig) {
        // не копируем по связям от библиотечных объектов
        if (isLibraryObject(entityClass)) {
            return List.of();
        }
        return navigator.getChildToParentEntityRelationships(entityClass).stream()
                .filter(r -> isCopyAllowed(copyConfig, r.getParentEntityClass())).collect(Collectors.toList());
    }

    public <TChild extends Entity<?>> List<? extends Relationship<? extends Entity<?>, TChild, ?>>
    getRelationshipsToSetIds(Class<TChild> entityClass, CopyOperationContainer copyConfig) {
        return navigator.getChildToParentEntityRelationships(entityClass).stream()
                .filter(r -> isCopyAllowed(copyConfig, r.getParentEntityClass())).collect(Collectors.toList());
    }

    public <TParent extends Entity<TKey>, TKey> List<? extends Relationship<TParent, ? extends Entity<?>, TKey>>
    getParentToChildRelationships(Class<TParent> entityClass, CopyOperationContainer copyConfig) {
        // не копируем по связям от библиотечных объектов
        if (isLibraryObject(entityClass)) {
            return List.of();
        }
        return navigator.getParentToChildEntityRelationships(entityClass).stream()
                .filter(r -> isCopyAllowed(copyConfig, r.getChildEntityClass())).collect(Collectors.toList());
    }

    private static boolean isLibraryObject(Class<?> entityClass) {
        return ShardLibraryObject.class.isAssignableFrom(entityClass)
                || ClientLibraryObject.class.isAssignableFrom(entityClass)
                || CampaignLibraryObject.class.isAssignableFrom(entityClass);
    }

    private static boolean isCopyAllowed(CopyOperationContainer copyContainer, Class<?> otherClass) {
        // закрыто копирование библиотечных объектов на шарде, не копируем их
        boolean canCopyInsideShard =
                copyContainer.isCopyingBetweenShards() || !ShardLibraryObject.class.isAssignableFrom(otherClass);
        // закрыто копирование библиотечных объектов на клиенте, не копируем их
        boolean canCopyInsideClient =
                copyContainer.isCopyingBetweenClients() || !ClientLibraryObject.class.isAssignableFrom(otherClass);
        // закрыто копирование библиотечных объектов на кампании, не копируем их
        boolean canCopyInsideCampaign =
                copyContainer.isCopyingBetweenCampaigns() || !CampaignLibraryObject.class.isAssignableFrom(otherClass);
        return canCopyInsideShard
                && canCopyInsideClient
                && canCopyInsideCampaign;
    }

}
