package ru.yandex.direct.core.copyentity;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.Entity;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

/**
 * Сервис сущности, предоставляет необходимый набор операций для работы с сущностью,
 * получение сущности по id, добавление новой сущности и копирование сущности. Дальше можно расширить до полного CRUD
 *
 * @param <T>    тип сущности
 * @param <KeyT> тип первичного ключа сущности
 */
@ParametersAreNonnullByDefault
public interface EntityService<T extends Entity<KeyT>, KeyT> {
    /**
     * Получает сущности по их id
     *
     * @param clientId    id клиента - владельца сущностей
     * @param operatorUid id оператора, выполняющего операцию
     * @param ids         коллекция id сущностей
     * @return список сущностей
     */
    List<T> get(ClientId clientId, Long operatorUid, Collection<KeyT> ids);

    /**
     * Получает сущности по их id, вызывает {@code get} для не более чем {@code chunkSize} сущностей за раз
     *
     * @return список сущностей
     * @see EntityService#get(ClientId, Long, Collection)
     */
    default List<T> getChunked(ClientId clientId, Long operatorUid, List<KeyT> ids, int chunkSize) {
        return StreamEx.ofSubLists(ids, chunkSize)
                .flatCollection(toGetChunk -> get(clientId, operatorUid, toGetChunk))
                .toList();
    }

    /**
     * Добавляет новые сущности в систему/базу
     *
     * @param clientId      id клиента - владельца сущностей
     * @param operatorUid   id оператора - представителя клиента
     * @param entities      список сущностей для добавления
     * @param applicability тип операции - частичная или полная
     * @return массовый результат добавления
     */
    MassResult add(ClientId clientId, Long operatorUid, List<T> entities, Applicability applicability);

    /**
     * Копирует набор сущностей. Логика копирования может отличаться от логики добавления новых сущностей
     *
     * @param copyContainer контейнер операции копирования
     * @param entities      список сущностей для добавления
     * @param applicability тип операции - частичная или полная
     * @return массовый результат добавления
     */
    MassResult copy(CopyOperationContainer copyContainer, List<T> entities, Applicability applicability);

    /**
     * Копирует набор сущностей, вызывая операцию копирования для не больше чем {@code chunkSize} сущностей за раз
     *
     * @return объединенный массовый результат добавления
     * @see EntityService#copy(CopyOperationContainer, List, Applicability)
     */
    default MassResult copyChunked(
            CopyOperationContainer copyContainer, List<T> entities,
            Applicability applicability, int chunkSize) {

        List<MassResult> massResults = StreamEx.ofSubLists(entities, chunkSize)
                .map(entityChunk -> copy(copyContainer, entityChunk, applicability))
                .toList();

        return CopyValidationResultUtils.mergeMassResults(massResults);
    }
}
