package ru.yandex.direct.core.copyentity;

import java.lang.reflect.InvocationTargetException;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

@ParametersAreNonnullByDefault
public class TypeUtils {

    private TypeUtils() {
    }

    /**
     * Находит ближайших предков заданного класса, реализующих заданный интерфейс
     *
     * @param successor     класс-потомок, для которого ищутся предки, реализующие заданный интерфейс
     * @param baseInterface интерфейс, который должны реализовывать предки заданного класса
     * @return Список предков заданного класса на одном уровне иерархии наследования, реализующих заданный интерфейс.
     * Их может быть несколько.
     */
    static <T> List<Class<? extends T>> findAncestorsImplementing(Class<?> successor, Class<T> baseInterface) {
        Set<TypeUnderCheck<T>> checking = Set.of(new TypeUnderCheck<>(successor, baseInterface));
        Set<TypeUnderCheck<T>> checked = new HashSet<>();

        while (!checking.isEmpty()) {
            Set<TypeUnderCheck<T>> implementingInterface = StreamEx.of(checking)
                    .filter(TypeUnderCheck::implementsInterface)
                    .toSet();

            if (!implementingInterface.isEmpty()) {
                return implementingInterface.stream()
                        .map(TypeUnderCheck::getType).collect(Collectors.toList());
            }
            checked.addAll(checking);

            Set<TypeUnderCheck<T>> parents = StreamEx.of(checking)
                    .flatMap(TypeUnderCheck::getSupertypes)
                    .map(st -> new TypeUnderCheck<>(st, baseInterface))
                    .toSet();

            parents.removeAll(checked);

            checking = parents;
        }

        return List.of();
    }

    @SuppressWarnings("unchecked")
    public static <T> T cast(Object elem) {
        return (T) elem;
    }

    public static <T> T silentCreateInstance(Class<T> type) {
        try {
            return type.getDeclaredConstructor().newInstance();
        } catch (NoSuchMethodException | IllegalAccessException | InstantiationException | InvocationTargetException ex) {
            return null;
        }
    }

    private static class TypeUnderCheck<T> {
        private final Class<?> type;
        private final Class<T> check;
        private final Set<Class<?>> interfaces;

        private TypeUnderCheck(Class<?> type, Class<T> check) {
            this.type = type;
            this.check = check;
            this.interfaces = Set.of(type.getInterfaces());
        }

        private Class<? extends T> getType() {
            return (Class<? extends T>) type;
        }

        private Set<Class<?>> getInterfaces() {
            return interfaces;
        }

        private Class<?> getSuperclass() {
            return type.getSuperclass();
        }

        private StreamEx<Class<?>> getSupertypes() {
            return StreamEx.of(getInterfaces()).append(getSuperclass()).nonNull();
        }

        private boolean implementsInterface() {
            return interfaces.contains(check);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            TypeUnderCheck<T> that = (TypeUnderCheck<T>) o;
            return Objects.equals(type, that.type) && Objects.equals(check, that.check);
        }

        @Override
        public int hashCode() {
            return Objects.hash(type);
        }
    }
}
