package ru.yandex.direct.core.copyentity.mediators

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.EntityContext
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignWithCampaignType
import ru.yandex.direct.core.entity.campaign.service.CampaignService
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService
import ru.yandex.direct.core.entity.keyword.model.Keyword
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.result.MassResult
import java.math.BigDecimal

/**
 * 1. Получает типы всех целевых кампаний. Они нужны для правильной установки нулевых ставок на дефолтные и наоборот
 * 2. Для каждой группы рассчитывает дефолтную ставку. Дефолтная ставка будет одинакова для всех групп
 *    в одной и той же целевой кампании:
 *    - для CPM-групп в CPM_YNDX_FRONTPAGE кампании ставка берется из ограничений по кампании,
 *      возвращаемых сервисом CpmYndxFrontpageCurrencyService, либо clientToCurrency.minCpmPrice,
 *      если таких ограничений не нашлось.
 *    - для CPM-групп в других типах кампаний - либо первая ненулевая ставка из Keywords копируемых групп внутри
 *      каждой кампании, либо clientToCurrency.minCpmPrice, если ставок на Keywords не нашлось.
 *    - дла остальных групп - clientToCurrency.minPrice
 */
@Component
class AdGroupsBidFieldsDefaultValuesResolverMediator(
    val campaignService: CampaignService,
    val cpmYndxFrontpageCurrencyService: CpmYndxFrontpageCurrencyService,
) : CopyMediator {
    override fun mediate(context: EntityContext, copyContainer: CopyOperationContainer): Map<Class<*>, MassResult<*>> {
        val adGroups: List<AdGroup> = context.getObjects(AdGroup::class.java)
        if (adGroups.isEmpty()) {
            return mapOf()
        }

        // Получаем имеющиеся типы кампаний.
        val campaignTypesByCampaignId: MutableMap<Long, CampaignType> = context.getEntities(BaseCampaign::class.java)
            .mapValues { (it.value as CampaignWithCampaignType).type }
            .toMutableMap()

        // Для получения ограничений по цене для групп кампаний CPM_YNDX_FRONTPAGE, мы должны проставить в группах
        // идентификатор целевой кампании, а потом, перед копированием, вернуть идентификаторы
        // кампаний в группах обратно.
        val oldCampaignIdsByAdGroupId: Map<Long, Long> = adGroups.associateBy({ it.id }, { it.campaignId })
        adGroups.forEach {
            it.campaignId = copyContainer.getCopyToParentIdOrSame(BaseCampaign::class.java, it.campaignId)
        }

        //Список кампаний, для которых мы пока не знаем тип
        val campaignIdsWithMissingTypes: List<Long> = adGroups
            .filter { it.campaignId !in campaignTypesByCampaignId }
            .map { it.campaignId }

        // Получаем недостающие типы кампаний
        if (campaignIdsWithMissingTypes.isNotEmpty()) {
            campaignTypesByCampaignId += campaignService.getCampaignsTypes(campaignIdsWithMissingTypes)
        }

        // Группируем группы по типу получения ставки по-умолчанию: CPM_YNDX_FRONTPAGE, CPM или все остальные (OTHER)
        val adGroupsByAdGroupMinBidPriceType: Map<AdGroupBidPriceType, List<AdGroup>> =
            adGroups.groupBy { getAdGroupBidPriceType(it.type, campaignTypesByCampaignId[it.campaignId]!!) }

        val clientToCurrency: Currency = Currencies.getCurrency(copyContainer.clientTo.workCurrency)

        // Получаем ограничения для групп кампаний CPM_YNDX_FRONTPAGE
        val cpmYandexFrontpageAdGroups: List<AdGroup> =
            adGroupsByAdGroupMinBidPriceType[AdGroupBidPriceType.CPM_YNDX_FRONTPAGE].orEmpty()

        // Считаем ставки по-умолчанию для групп кампаний CPM_YNDX_FRONTPAGE, это будет либо минимальная цена из
        // полученных ограничений, либо clientToCurrency.minCpmPrice
        val cmpYandexFrontpagePriceByAdGroupId: Map<Long, BigDecimal> =
            getDefaultFixedPriceForYandexFrontpageAdGroups(
                cpmYandexFrontpageAdGroups,
                copyContainer.shardTo,
                clientToCurrency,
            )

        // Для cpm-групп для возьмем все ключевые слова, сгруппируем их по кампаниям, и в каждой кампании
        // возьмем минимальную цену ключевого слова, или clientToCurrency.minCpmPrice,
        // если ключевых слов с ценой не найдется
        val cpmAdGroups = adGroupsByAdGroupMinBidPriceType[AdGroupBidPriceType.CPM].orEmpty()

        val cpmPriceByAdGroupId: Map<Long, BigDecimal> =
            getDefaultFixedPriceForCpmAdGroups(cpmAdGroups, context, clientToCurrency)

        // Для остальных групп ставку по-умолчанию устанавливаем в clientToCurrency.minPrice
        val otherAdGroups: List<AdGroup> = adGroupsByAdGroupMinBidPriceType[AdGroupBidPriceType.OTHER].orEmpty()

        val otherPriceByAdGroupId = otherAdGroups.associateBy({ it.id }, { clientToCurrency.minPrice })

        copyContainer.fixedDestinationAdGroupAutoPrices =
            cmpYandexFrontpagePriceByAdGroupId + cpmPriceByAdGroupId + otherPriceByAdGroupId

        copyContainer.targetCampaignTypeByGroupId =
            adGroups.associateBy({ it.id }, { campaignTypesByCampaignId[it.campaignId]!! })

        // Возвращаем оригинальные идентификаторы кампаний
        adGroups.forEach {
            it.campaignId = oldCampaignIdsByAdGroupId[it.id]
        }

        return mapOf()
    }

    private fun getDefaultFixedPriceForYandexFrontpageAdGroups(
        cpmYandexFrontpageAdGroups: List<AdGroup>,
        shard: Int,
        clientToCurrency: Currency,
    ): Map<Long, BigDecimal> {
        val cpmYandexFrontpageAdGroupsById: Map<Long, AdGroup> = cpmYandexFrontpageAdGroups.associateBy { it.id }

        val cpmYandexFrontpagePriceByAdGroupIdRestrictions: Map<Long, CpmYndxFrontpageAdGroupPriceRestrictions> =
            cpmYndxFrontpageCurrencyService.getAdGroupIdsToPriceDataMapByAdGroups(
                cpmYandexFrontpageAdGroups,
                shard,
                clientToCurrency
            )

        // Строим карту ставок по-умолчанию по идентификаторам кампании для CPM_YNDX_FRONTPAGE кампаний.
        // Берем минимальную ставку из ограничений или clientToCurrency.minCpmFrontpagePrice
        val cpmYandexFrontpagePriceByCampaignId: Map<Long, BigDecimal> =
            cpmYandexFrontpagePriceByAdGroupIdRestrictions.entries
                .groupBy(
                    { cpmYandexFrontpageAdGroupsById[it.key]!!.campaignId },
                    { it.value })
                .mapValues { restrictions ->
                    restrictions.value.map { it.cpmYndxFrontpageMinPrice }.minOf { it }
                }

        // Строим карту ставок по-умолчанию по идентификаторам cpm-групп, копируемых в CPM_YNDX_FRONTPAGE кампании
        return cpmYandexFrontpageAdGroups.associateBy(
            { it.id },
            { cpmYandexFrontpagePriceByCampaignId[it.campaignId] ?: clientToCurrency.minCpmPrice })
    }

    private fun getDefaultFixedPriceForCpmAdGroups(
        cpmAdGroups: List<AdGroup>,
        context: EntityContext,
        clientToCurrency: Currency
    ): Map<Long, BigDecimal> {
        val cpmAdGroupsById: Map<Long, AdGroup> = cpmAdGroups.associateBy { it.id }

        val keywords: List<Keyword> = context.getObjects(Keyword::class.java)

        // Строим карту ставок по-умолчанию по идентификаторам кампаниям cpm-групп
        val cpmPriceByCampaignId: Map<Long, BigDecimal> = keywords
            .filter { cpmAdGroupsById.containsKey(it.adGroupId) }
            .groupBy { cpmAdGroupsById[it.adGroupId]!!.campaignId }
            .mapValues { campaignAdGroupsKeywordsEntry ->
                campaignAdGroupsKeywordsEntry.value.mapNotNull { it.price }.minOfOrNull { it }
                    ?: clientToCurrency.minCpmPrice
            }

        // Строим карту ставок по-умолчанию по идентификаторам cpm-групп
        return cpmAdGroups.associateBy(
            { it.id },
            { cpmPriceByCampaignId[it.campaignId] ?: clientToCurrency.minCpmPrice })
    }

    fun getAdGroupBidPriceType(adGroupType: AdGroupType, campaignType: CampaignType): AdGroupBidPriceType {
        return if (campaignType == CampaignType.CPM_YNDX_FRONTPAGE) {
            AdGroupBidPriceType.CPM_YNDX_FRONTPAGE
        } else {
            when (adGroupType) {
                AdGroupType.CPM_BANNER,
                AdGroupType.CPM_VIDEO,
                AdGroupType.CPM_OUTDOOR,
                AdGroupType.CPM_YNDX_FRONTPAGE,
                AdGroupType.CPM_INDOOR,
                AdGroupType.CPM_GEOPRODUCT,
                AdGroupType.CPM_AUDIO,
                AdGroupType.CPM_GEO_PIN -> AdGroupBidPriceType.CPM
                else -> AdGroupBidPriceType.OTHER
            }
        }
    }

    enum class AdGroupBidPriceType {
        CPM_YNDX_FRONTPAGE,
        CPM,
        OTHER
    }
}
