package ru.yandex.direct.core.copyentity.mediators

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.EntityContext
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags
import ru.yandex.direct.core.entity.campaign.converter.CampaignConverter
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignWithForbiddenStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignWithMeaningfulGoals
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientAdapter
import ru.yandex.direct.core.entity.campaign.service.RequestBasedMetrikaClientFactory
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.ENGAGED_SESSION_GOAL_ID
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.metrikaReturnsResultWithErrors
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.strategy.CampaignWithCustomStrategyValidator.STRATEGY_TYPES_TO_CHECK_MEANINGFUL_GOALS_IN_CAMPAIGN
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.metrika.container.CampaignTypeWithCounterIds
import ru.yandex.direct.core.entity.metrika.service.campaigngoals.CampaignGoalsService
import ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoal
import ru.yandex.direct.core.entity.retargeting.model.CampMetrikaGoalId
import ru.yandex.direct.core.entity.retargeting.model.Goal
import ru.yandex.direct.metrika.client.MetrikaClientException
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.direct.validation.builder.ListValidationBuilder
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult

/**
 * Вычисляет для каждой кампании, нужно ли сбрасывать ее стратегию на недельный бюджет при копировании между клиентами,
 * а также удаляет из графа цели метрики, привязанные в таким кампаниям
 */
@Component
class CampaignWithStrategyCheckAccessToMetrikaGoalsMediator(
    private val featureService: FeatureService,
    private val metrikaClientFactory: RequestBasedMetrikaClientFactory,
    private val campaignGoalsService: CampaignGoalsService,
) : CopyMediator {

    private val logger = LoggerFactory.getLogger(CampaignWithStrategyCheckAccessToMetrikaGoalsMediator::class.java)
    private val conversionStrategyNames: Set<StrategyName> = setOf(
        StrategyName.AUTOBUDGET,
        StrategyName.AUTOBUDGET_AVG_CPA,
        StrategyName.AUTOBUDGET_AVG_CPI,
        StrategyName.AUTOBUDGET_ROI,
        StrategyName.AUTOBUDGET_CRR,
    )

    override fun mediate(
        context: EntityContext,
        copyContainer: CopyOperationContainer
    ): Map<Class<*>, MassResult<*>> {
        // Получаем список кампаний с целями, для которых нужно проверить доступ в метрике, а заодно проставляем
        // флаг сброса стратегии для strategy.goalId = BY_ALL_GOALS_GOAL_ID = 0 (все цели)
        val campaignsWithGoalsToCheck: MutableMap<CampaignWithStrategy, Set<Long>> =
            getCampaignsWithGoalsCheckMap(context, copyContainer)

        // Если копируем внутри клиента, больше никаких действий не нужно
        if (!copyContainer.isCopyingBetweenClients) {
            return mapOf()
        }

        // Проверяем права на цели метрики, если это нужно, а заодно проставляем флаги сброса конверсионной стратегии
        val vr: ValidationResult<List<CampaignWithStrategy>, Defect<*>> =
            checkMetrikaGoalsAndSetDropStrategyFlag(campaignsWithGoalsToCheck, copyContainer)

        if (vr.hasAnyErrors()) {
            return mapOf(
                BaseCampaign::class.java to MassResult.brokenMassAction<Long, CampaignWithStrategy>(null, vr))
        }

        if (copyContainer.flags.isDoNotCheckRightsToMetrikaGoals) {
            return mapOf()
        }
        // Если ошибок нет, и права на цели в метрике мы проверяем, то чистим граф от целей метрики,
        // привязанных к кампаниям, для которых нужно сбросить стратегии на недельный бюджет
        val campMetrikaGoals: List<CampMetrikaGoal> = context.getObjects(CampMetrikaGoal::class.java)
        // Если копируем конверсионные стратегии, то удаляем только цели, привязанные к кампаниям, для которых нужно
        // сбросить стратегии на недельный бюджет, а если не копируем, то удаляем все цели вообще
        val campMetrikaGoalsIdsToRemove: Set<CampMetrikaGoalId> = if (copyContainer.flags.isCopyConversionStrategies) {
            campMetrikaGoals.filter {
                copyContainer.getCampaignPreprocessFlagsById(it.campaignId).isDropStrategyToDefault
            }.map { it.id }.toSet()
        } else {
            campMetrikaGoals.map { it.id }.toSet()
        }
        context.removeLeafs(CampMetrikaGoal::class.java, campMetrikaGoalsIdsToRemove)
        return mapOf()
    }

    /**
     * Определяет список кампаний и целей из их конверсионных стратегий, доступ к которым нужно проверить у целевого
     * клиента при копировании между клиентами. Если мы безусловно будем копировать цели, тогда целевой тогда
     * целевой клиент будет иметь стратегию с недоступными целями, что неправильно.
     * @param context загруженный граф копирования
     * @param copyContainer параметры копирования
     * @return карта компаний и идентификаторов их целей, доступ к которым нужно проверить у целевого клиента.
     * При копировании внутри клиента возвращает пустую карту.
     */
    private fun getCampaignsWithGoalsCheckMap(
        context: EntityContext,
        copyContainer: CopyOperationContainer
    ): MutableMap<CampaignWithStrategy, Set<Long>> {
        val campaignsWithGoalsCheckMap: MutableMap<CampaignWithStrategy, Set<Long>> = mutableMapOf()

        val campaignsByIds: Map<Long, BaseCampaign> = context.getEntities(BaseCampaign::class.java)

        val copyFlags: CopyCampaignFlags = copyContainer.flags

        for ((campaignId, baseCampaign) in campaignsByIds) {
            if (baseCampaign !is CampaignWithStrategy) {
                continue
            }
            val campaign: CampaignWithStrategy = baseCampaign
            // Если кампания без стратегии, то пропускаем ее
            if (campaign is CampaignWithForbiddenStrategy) {
                continue
            }
            val strategy = campaign.strategy
            // Нет цели - пропускаем
            if (strategy.strategyData.goalId == null) {
                continue
            }
            val strategyGoalId: Long = strategy.strategyData.goalId
            // Идентификатор цели равный BY_ALL_GOALS_GOAL_ID = 0 (все цели) - устаревшая мета цель, поэтому
            // сбрасываем стратегию, настроенную на нее на недельный бюджет, даже если копируем внутри клиента
            if (strategyGoalId == CampaignConstants.BY_ALL_GOALS_GOAL_ID) {
                copyContainer.getCampaignPreprocessFlagsById(campaignId).isDropStrategyToDefault = true
                continue
            }
            // стратегия не конверсионная, то проверять дальше смысла нет
            if (!conversionStrategyNames.contains(strategy.strategyName)) {
                continue
            }

            // Если копируем между клиентами и не копируем конверсионные стратегии, то устанавливаем
            // флаг сброса на дефолтную стратегию сразу
            if (copyContainer.isCopyingBetweenClients && !copyFlags.isCopyConversionStrategies) {
                copyContainer.getCampaignPreprocessFlagsById(campaignId).isDropStrategyToDefault = true
                continue
            }

            // Найдем идентификаторы целей, к которым нужно проверить доступ
            lateinit var goalsIds: MutableSet<Long>
            // Проверим, что в качестве основной цели не указаны вовлеченные сессии, их ставить как главную цель нельзя,
            // можно только в meaningfulGoals
            if (strategyGoalId == ENGAGED_SESSION_GOAL_ID) {
                copyContainer.getCampaignPreprocessFlagsById(campaignId).isDropStrategyToDefault = true
                continue
            }

            if (strategyGoalId != CampaignConstants.MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID) {
                goalsIds = mutableSetOf(strategyGoalId)
            } else {
                // Если идентификатор цели равен MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID = 13 (по ключевым целям),
                // значит нужно оптимизировать стратегию по всем ключевым целям, поэтому проверим доступ к ним всем
                if (campaign is CampaignWithMeaningfulGoals) {
                    goalsIds = campaign.meaningfulGoals.map { it.goalId }.toMutableSet()
                    goalsIds.remove(ENGAGED_SESSION_GOAL_ID)
                    // Проверяем, что для некоторых типов кампаний эта цель была не единственная,
                    // иначе сбрасываем стратегию
                    if (goalsIds.isEmpty() &&
                        STRATEGY_TYPES_TO_CHECK_MEANINGFUL_GOALS_IN_CAMPAIGN.contains(campaign.strategy.strategyName)) {
                        copyContainer.getCampaignPreprocessFlagsById(campaignId).isDropStrategyToDefault = true
                        continue
                    }
                } else {
                    // Если goalId = 13, но кампания не потомок MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID - это
                    // какой-то сбой, и стратегию нужно сбросить.
                    copyContainer.getCampaignPreprocessFlagsById(campaignId).isDropStrategyToDefault = true
                    continue
                }
            }
            // Добавляем кампании для проверки доступности целей в карту, если доступность целей нужно проверять
            // и они есть. Флаг isDoNotCheckRightsToMetrikaGoals не проверяем раньше, так все рано нужно проставить
            // флаг isDropStrategyToDefault = true для невалидных кампаний
            if (copyContainer.isCopyingBetweenClients &&
                !copyFlags.isDoNotCheckRightsToMetrikaGoals && goalsIds.isNotEmpty()) {
                campaignsWithGoalsCheckMap[campaign] = goalsIds
            }
        }
        return campaignsWithGoalsCheckMap
    }

    /**
     * Для каждой переданной кампании проверяет, есть ли у целевого клиента доступ ко всем ее целям, и если доступа нет,
     * то устанавливает для таких кампаний флаг сброса конверсионной стратегии на недельный бюджет
     * @param campaignsWithGoalsCheckMap карта целей метрики, требующих проверки по кампаниям
     * @param copyContainer параметры копирования
     * @return результат валидации. Может быть с ошибками, только если запрос в метрику кинул эксепшн,
     * иначе всегда успешный
     */
    private fun checkMetrikaGoalsAndSetDropStrategyFlag(
        campaignsWithGoalsCheckMap: Map<CampaignWithStrategy, Set<Long>>, copyContainer: CopyOperationContainer
    ): ValidationResult<List<CampaignWithStrategy>, Defect<*>> {
        val vr: ValidationResult<List<CampaignWithStrategy>, Defect<*>> =
            ValidationResult(campaignsWithGoalsCheckMap.keys.toList())
        val vb: ListValidationBuilder<CampaignWithStrategy, Defect<*>> = ListValidationBuilder(vr)

        if (campaignsWithGoalsCheckMap.isEmpty()) {
            return vb.result
        }

        val clientId = copyContainer.clientIdTo

        try {
            val enabledFeatures: Set<String> = featureService.getEnabledForClientId(clientId)
            // Создаем кэширующего клиента метрики
            val metrikaClientAdapter: RequestBasedMetrikaClientAdapter =
                metrikaClientFactory.createMetrikaClient(
                    copyContainer.clientIdTo, campaignsWithGoalsCheckMap.keys.toList()
                )
            // Подготавливаем данные для запроса в метрику
            val campaignTypesWithCountersByCampaignId: MutableMap<Long, CampaignTypeWithCounterIds> = mutableMapOf()
            for (entry in campaignsWithGoalsCheckMap) {
                val campaign: CampaignWithStrategy = entry.key
                campaignTypesWithCountersByCampaignId[campaign.id] =
                    CampaignConverter.toCampaignTypeWithCounterIdsFromCampaignWithStrategy(campaign, enabledFeatures)
            }
            // Получаем из метрики список доступных целей для каждой кампании
            val availableGoalsForCampaignId: Map<Long, MutableSet<Goal>> =
                campaignGoalsService.getAvailableGoalsForCampaignId(
                    copyContainer.operatorUid,
                    clientId,
                    campaignTypesWithCountersByCampaignId,
                    metrikaClientAdapter
                )
            // Проверяем для каждой кампании, все ли ее цели в списке доступных, и если нет, то выставляем флаг сброса
            // стратегии на недельный бюджет.
            for (entry in campaignsWithGoalsCheckMap) {
                val campaign: CampaignWithStrategy = entry.key
                val campaignId: Long = campaign.id
                val goalsIds: Set<Long> = entry.value
                val availableGoalsIds: Set<Long> =
                    availableGoalsForCampaignId.getOrDefault(campaignId, mutableSetOf()).map { it.id }.toSet()

                if (!availableGoalsIds.containsAll(goalsIds)) {
                    copyContainer.getCampaignPreprocessFlagsById(campaignId)
                        .isDropStrategyToDefault = true
                }
            }
        } catch (e: MetrikaClientException) {
            logger.warn("Got an MetrikaClientException when querying for metrika goals for clientId: $clientId", e)
            vb.checkEach(metrikaReturnsResultWithErrors(), When.notNull())
        } catch (e: InterruptedRuntimeException) {
            logger.warn("Got an InterruptedRuntimeException when querying for metrika goals for clientId: $clientId", e)
            vb.checkEach(metrikaReturnsResultWithErrors(), When.notNull())
        }
        return vb.result
    }
}
