package ru.yandex.direct.core.copyentity.postprocessors

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyConfig
import ru.yandex.direct.core.copyentity.CopyResult
import ru.yandex.direct.core.copyentity.EntityContext
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationStatuses
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.InternalBanner
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository
import ru.yandex.direct.core.entity.banner.service.BannerService
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerateService
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges

@Component
class BannerStatusesCopier(
    private val shardHelper: ShardHelper,
    private val bannerService: BannerService,
    private val bannerModerateService: BannerModerateService,
    private val bannerModifyRepository: BannerModifyRepository,
) {
    private val updateChunkSize = 1000

    fun copyBannerStatuses(copyConfig: CopyConfig<*, *>, copyResult: CopyResult<*>) {
        val oldBannerIds: Set<Long> = copyResult.getEntityMappings(BannerWithAdGroupId::class.java)
            .filterValues { it != null }
            .keys

        if (oldBannerIds.isEmpty()) {
            return
        }

        val oldBanners: List<BannerWithAdGroupId> =
            bannerService.get(copyConfig.clientIdFrom, copyConfig.operatorUid, oldBannerIds)
        setBannerStatuses(copyConfig, copyResult.entityContext, oldBanners)
    }

    private fun setBannerStatuses(
        copyConfig: CopyConfig<*, *>,
        entityContext: EntityContext,
        oldBanners: List<BannerWithAdGroupId>,
    ) {
        val shardTo = shardHelper.getShardByClientIdStrictly(copyConfig.clientIdTo)

        val oldBannerIdToNewBanner: Map<Long, BannerWithAdGroupId> = entityContext
            .getEntities(BannerWithAdGroupId::class.java)
            .filterValues { banner -> banner.id != null }

        // moderate banners
        val bannersIdsToModerate: List<Long> = oldBanners.asSequence()
            .filterIsInstance(BannerWithModerationStatuses::class.java)
            .filter { it.statusModerate != BannerStatusModerate.NEW }
            .mapNotNull { banner -> oldBannerIdToNewBanner[banner.id] }
            .map { it.id }
            .distinct()
            .toList()

        bannerModerateService.moderateBanners(copyConfig.clientIdTo, copyConfig.operatorUid, bannersIdsToModerate)

        // set statusShow and statusArchived
        val appliedChanges: List<AppliedChanges<BannerWithSystemFields>> = oldBanners
            .filterIsInstance(BannerWithSystemFields::class.java)
            .map { banner -> toStatusesAppliedChanges(banner, oldBannerIdToNewBanner[banner.id]!!) }
            .filter { ac -> ac.hasActuallyChangedProps() }

        val container = BannerRepositoryContainer(shardTo)
        appliedChanges
            .chunked(updateChunkSize)
            .forEach { chunk -> bannerModifyRepository.update(container, chunk) }
    }

    private fun toStatusesAppliedChanges(
        oldBanner: BannerWithSystemFields,
        banner: BannerWithAdGroupId,
    ): AppliedChanges<BannerWithSystemFields> {
        val modelChanges = ModelChanges(banner.id, BannerWithSystemFields::class.java)

        val isStoppedByUrlMonitoring = banner is InternalBanner && banner.isStoppedByUrlMonitoring
        val statusShow = oldBanner.statusShow && oldBanner.statusModerate != BannerStatusModerate.NO
            && !isStoppedByUrlMonitoring

        modelChanges.process(statusShow, BannerWithSystemFields.STATUS_SHOW)
        modelChanges.process(oldBanner.statusArchived, BannerWithSystemFields.STATUS_ARCHIVED)

        return modelChanges.applyTo(banner as BannerWithSystemFields)
    }
}
