package ru.yandex.direct.core.copyentity.postprocessors

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyConfig
import ru.yandex.direct.core.copyentity.CopyResult
import ru.yandex.direct.core.copyentity.EntityContext
import ru.yandex.direct.core.entity.keyword.model.Keyword
import ru.yandex.direct.core.entity.keyword.service.KeywordOperationFactory
import ru.yandex.direct.core.entity.keyword.service.KeywordService
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.operation.Applicability

@Component
class KeywordStatusesCopier(
    private val keywordOperationFactory: KeywordOperationFactory,
    private val keywordService: KeywordService,
) {
    private val updateChunkSize = 1000

    fun copyKeywordStatuses(
        copyConfig: CopyConfig<*, *>,
        copyResult: CopyResult<*>,
    ) {
        val oldKeywordIds: Set<Long> = copyResult.getEntityMappings(Keyword::class.java)
            .filterValues { it != null }
            .keys

        if (oldKeywordIds.isEmpty()) {
            return
        }

        val oldKeywords: List<Keyword> =
            keywordService.get(copyConfig.clientIdFrom, copyConfig.operatorUid, oldKeywordIds)
        setKeywordStatuses(copyConfig, copyResult.entityContext, oldKeywords)
    }

    private fun setKeywordStatuses(
        copyConfig: CopyConfig<*, *>,
        context: EntityContext,
        oldKeywords: List<Keyword>,
    ) {
        val oldKeywordIdToNewKeyword: Map<Long, Keyword> = context.getEntities(Keyword::class.java)
            .filterValues { keyword -> keyword.id != null }

        val keywordIdsToSuspend: Set<Long> = oldKeywords
            .filter { keyword -> keyword.isSuspended }
            .mapNotNull { keyword -> oldKeywordIdToNewKeyword[keyword.id] }
            .map { keyword -> keyword.id }
            .toSet()

        val keywordChanges: List<ModelChanges<Keyword>> = keywordIdsToSuspend
            .map { id ->
                ModelChanges(id, Keyword::class.java)
                    .process(true, Keyword.IS_SUSPENDED)
            }

        keywordChanges
            .chunked(updateChunkSize)
            .forEach { chunk -> updateKeywordsChunk(copyConfig, chunk) }
    }

    private fun updateKeywordsChunk(copyConfig: CopyConfig<*, *>, chunk: List<ModelChanges<Keyword>>) {
        val updateOperation = keywordOperationFactory.createKeywordsUpdateOperation(
            Applicability.PARTIAL, chunk,
            copyConfig.operatorUid, copyConfig.clientIdTo, copyConfig.operatorUid,
            false, false,
        )
        updateOperation.prepareAndApply()
    }
}
