package ru.yandex.direct.core.copyentity.prefilters

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.model.AdGroupStates
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupStatusArchived
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.adGroupStatusStopped
import ru.yandex.direct.validation.builder.Constraint.fromPredicate
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder

@Component
class AdGroupPrefilter(
    private val adGroupRepository: AdGroupRepository,
) : EntityPrefilter<AdGroup> {

    companion object {
        private val DISALLOWED_ADGROUP_TYPES = setOf(
            AdGroupType.CPM_AUDIO,
            AdGroupType.CPM_INDOOR,
            AdGroupType.CPM_OUTDOOR,
            AdGroupType.CPM_GEO_PIN,
            AdGroupType.CPM_GEOPRODUCT,
        )
    }

    override fun prefilter(
        entities: List<AdGroup>,
        copyContainer: CopyOperationContainer,
    ): ValidationResult<List<AdGroup>, Defect<*>> {
        return validateList(entities) {
            checkEachBy { adGroup ->
                checkValidAdGroupsTypes(adGroup)
            }

            if (!copyContainer.flags.isCopyArchived || !copyContainer.flags.isCopyStopped) {
                val idToStatuses = getAdGroupStates(entities, copyContainer)
                checkEach(
                    fromPredicate({ adGroup -> !isArchived(idToStatuses[adGroup.id]) }, adGroupStatusArchived()),
                    When.isFalse(copyContainer.flags.isCopyArchived),
                )
                checkEach(
                    fromPredicate({ adGroup -> !isStopped(idToStatuses[adGroup.id]) }, adGroupStatusStopped()),
                    When.isFalse(copyContainer.flags.isCopyStopped),
                )
            }
        }
    }

    private fun getAdGroupStates(
        entities: Collection<AdGroup>,
        copyContainer: CopyOperationContainer,
    ): Map<Long, AdGroupStates> {
        val adGroupIds = entities.map { it.id }
        return adGroupRepository.getAdGroupStatuses(copyContainer.shardFrom, adGroupIds)
    }

    private fun isArchived(states: AdGroupStates?): Boolean {
        return states != null && states.archived
    }

    private fun isStopped(states: AdGroupStates?): Boolean {
        return states != null && !states.showing
    }

    /**
     * Запрещает копировать закрытые продукты
     */
    private fun checkValidAdGroupsTypes(adGroup: AdGroup): ValidationResult<AdGroup, Defect<*>> {
        val vb = ModelItemValidationBuilder.of(adGroup)
        vb.item(AdGroup.TYPE)
            .check(CommonConstraints.notInSet(DISALLOWED_ADGROUP_TYPES), AdGroupDefects.adGroupTypeNotSupported())
        return vb.result
    }
}
