package ru.yandex.direct.core.copyentity.prefilters

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign
import ru.yandex.direct.core.entity.campaign.model.CampaignSource
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported
import ru.yandex.direct.core.entity.campaign.service.validation.CopyCampaignDefects.yndFixedCurrency
import ru.yandex.direct.core.validation.defects.Defects.badStatusCampaignArchived
import ru.yandex.direct.currency.CurrencyCode
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.inSet
import ru.yandex.direct.validation.constraint.CommonConstraints.notEqual
import ru.yandex.direct.validation.constraint.CommonConstraints.notInSet
import ru.yandex.direct.validation.constraint.CommonConstraints.notTrue
import ru.yandex.direct.validation.result.Defect
import ru.yandex.direct.validation.result.ValidationResult
import ru.yandex.direct.validation.util.item
import ru.yandex.direct.validation.util.validateList
import ru.yandex.direct.validation.util.validateModel
import java.util.Set

@Component
class CampaignPrefilter : EntityPrefilter<BaseCampaign> {

    companion object {
        @JvmField
        val COPYABLE_TYPES = setOf(
            CampaignType.TEXT,
            CampaignType.DYNAMIC,
            CampaignType.PERFORMANCE,
            CampaignType.MOBILE_CONTENT,
            CampaignType.MCBANNER,
            CampaignType.CPM_BANNER,
            CampaignType.CPM_YNDX_FRONTPAGE,
            CampaignType.INTERNAL_AUTOBUDGET,
            CampaignType.INTERNAL_DISTRIB,
            CampaignType.INTERNAL_FREE,
        )

        val INTERNAL_CAMPAIGNS = setOf(
            CampaignType.INTERNAL_AUTOBUDGET,
            CampaignType.INTERNAL_DISTRIB,
            CampaignType.INTERNAL_FREE,
        )

        @JvmField
        val COPYABLE_SOURCES = setOf(
            CampaignSource.DIRECT,
            CampaignSource.DC,
            CampaignSource.API,
            CampaignSource.XLS,
        )
    }

    override fun prefilter(
        entities: List<BaseCampaign>,
        copyContainer: CopyOperationContainer,
    ): ValidationResult<List<BaseCampaign>, Defect<*>> {
        return validateList(entities) {
            checkEachBy(
                baseCampaignValidator(
                    copyContainer.flags.isCopyArchivedCampaigns,
                    copyContainer.isCopyingBetweenClients,
                )
            )
        }
    }

    private fun baseCampaignValidator(
        copyArchivedCampaigns: Boolean,
        isCopyingBetweenClients: Boolean,
    ): Validator<BaseCampaign, Defect<*>> {
        @Suppress("UNCHECKED_CAST")
        return campaignValidator(
            copyArchivedCampaigns,
            isCopyingBetweenClients
        ) as Validator<BaseCampaign, Defect<*>>
    }

    fun campaignValidator(
        copyArchivedCampaigns: Boolean,
        isCopyingBetweenClients: Boolean,
    ) = Validator { campaign: CommonCampaign ->
        validateModel(campaign) {
            item(CommonCampaign.TYPE) {
                check(inSet(COPYABLE_TYPES), campaignTypeNotSupported())
                check(
                    notEqual(CampaignType.PERFORMANCE, campaignTypeNotSupported()),
                    When.isTrue(isCopyingBetweenClients)
                )
                check(
                    notInSet(INTERNAL_CAMPAIGNS), campaignTypeNotSupported(),
                    When.isTrue(isCopyingBetweenClients)
                )
            }
            item(CommonCampaign.SOURCE) {
                check(inSet(COPYABLE_SOURCES), campaignTypeNotSupported())
            }
            item(CommonCampaign.IS_TOUCH) {
                check(notTrue(), campaignTypeNotSupported())
            }
            item(CommonCampaign.CURRENCY) {
                check(notEqual(CurrencyCode.YND_FIXED, yndFixedCurrency()))
            }
            item(CommonCampaign.STATUS_ARCHIVED) {
                check(notTrue(), badStatusCampaignArchived(), When.isFalse(copyArchivedCampaigns))
            }
        }
    }
}
