package ru.yandex.direct.core.copyentity.prefilters;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Префильто, обеспечивает предварительную фильтрацию копируемых сущностей в зависимости от типа
 */
@ParametersAreNonnullByDefault
public class Prefilter {
    private HashMap<Class, List<EntityPrefilter>> prefilters;

    public Prefilter() {
        this.prefilters = new HashMap<>();
    }

    /**
     * Добавляет новый типизированный префильтр в пул фильтров
     *
     * @param entityClass какой тип сущности обрабатывает новый типизированный префильтр
     * @param prefilter   добавляемый типизированный префильтр
     */
    protected <T> Prefilter addPrefilter(Class<T> entityClass, EntityPrefilter<T> prefilter) {
        prefilters.computeIfAbsent(entityClass, clazz -> new ArrayList<>()).add(prefilter);
        return this;
    }

    /**
     * Осуществляет предобработку переданных сущностей всеми имеющимися препроццессорами
     *
     * @param entityClass    тип переданных сущностей
     * @param entities       объекты, который надо обработать перед копированием
     * @param copyContainer  конфигурация копирования
     */
    @SuppressWarnings("unchecked")
    public <T> ValidationResult<List<T>, Defect> prefilter(Class<T> entityClass, List<T> entities,
                                                           CopyOperationContainer copyContainer) {
        List<EntityPrefilter<T>> filters = EntryStream.of(this.prefilters)
                .filterKeys(k -> k.isAssignableFrom(entityClass))
                .flatMap(e -> e.getValue().stream().map(p -> (EntityPrefilter<T>) p))
                .toList();
        ValidationResult<List<T>, Defect> result = new ValidationResult<>(entities);
        for (var filter : filters) {
            ValidationResult<List<T>, Defect> vr = (ValidationResult) filter.prefilter(entities, copyContainer);
            result.merge(vr);
        }
        return result;
    }
}
