package ru.yandex.direct.core.copyentity.preprocessors

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.translations.RenameProcessor
import ru.yandex.direct.core.entity.adgroup.model.AdGroup
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettings
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettingsProvider
import ru.yandex.direct.core.entity.adgroup.service.bstags.AllowedBsTagsValidator
import ru.yandex.direct.core.entity.feature.service.FeatureService

@Component
class AdGroupCopyPreprocessor(
    private val renameProcessor: RenameProcessor,
    private val adGroupBsTagsSettingsProvider: AdGroupBsTagsSettingsProvider,
    private val featureService: FeatureService,
) : CopyPreprocessorTypeSupport<AdGroup> {
    override fun getTypeClass() = AdGroup::class.java

    override fun preprocess(adGroups: List<AdGroup>, copyContainer: CopyOperationContainer) {
        adGroups.forEach { preprocess(it, copyContainer) }

        clearNonAccessibleTags(adGroups, copyContainer)
    }

    override fun preprocess(adGroup: AdGroup, copyContainer: CopyOperationContainer) {
        renameAdGroup(adGroup, copyContainer)
        clearTagsIfCopyingBetweenCampaigns(adGroup, copyContainer.isCopyingBetweenCampaigns())
        clearMinusKeywordsId(adGroup)

        adGroup.lastChange = null
    }

    private fun renameAdGroup(adGroup: AdGroup, copyContainer: CopyOperationContainer) {
        adGroup.name = renameProcessor.generateAdGroupCopyNameIfNeeded(adGroup, copyContainer)
    }

    /**
     * Метки принадлежат кампании и не копируются вместе с кампанией
     */
    private fun clearTagsIfCopyingBetweenCampaigns(adGroup: AdGroup, copyCampaignObjects: Boolean) {
        if (copyCampaignObjects) {
            adGroup.tags = null
        }
    }

    private fun clearMinusKeywordsId(adGroup: AdGroup) {
        // минус-слова добавляются списком в поле minusKeywords, minusKeywordsId делается сам, указывать его нельзя
        adGroup.minusKeywordsId = null
    }

    /**
     * Сбрасываем недоступные оператору теги ПИ
     * @see AllowedBsTagsValidator
     */
    private fun clearNonAccessibleTags(adGroups: List<AdGroup>, copyContainer: CopyOperationContainer) {
        val settings = adGroupBsTagsSettingsProvider.getAdGroupBsTagsSettings(adGroups, copyContainer.clientIdFrom)

        val validator =
            AllowedBsTagsValidator(settings, featureService, copyContainer.clientIdTo, copyContainer.operatorUid)

        adGroups.forEach { adGroup ->
            val adGroupSettings: AdGroupBsTagsSettings = settings[adGroup]!!

            if (adGroup.pageGroupTags != null) {
                adGroup.pageGroupTags = adGroup.pageGroupTags.filter { tag ->
                    validator.isBsTagAllowed(tag, adGroupSettings.allowedPageGroupTags)
                }
            }

            if (adGroup.targetTags != null) {
                adGroup.targetTags = adGroup.targetTags.filter { tag ->
                    validator.isBsTagAllowed(tag, adGroupSettings.allowedTargetTags)
                }
            }
        }
    }
}
