package ru.yandex.direct.core.copyentity.preprocessors;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.service.CalloutService;
import ru.yandex.direct.core.entity.banner.model.BannerMulticard;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithButton;
import ru.yandex.direct.core.entity.banner.model.BannerWithCallouts;
import ru.yandex.direct.core.entity.banner.model.BannerWithFlags;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithMulticardSet;
import ru.yandex.direct.core.entity.banner.model.BannerWithPhone;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.image.model.BannerImageFromPool;
import ru.yandex.direct.core.entity.image.repository.BannerImagePoolRepository;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.operation.Applicability;

import static ru.yandex.direct.core.copyentity.CopyValidationResultUtils.logFailedResults;
import static ru.yandex.direct.core.entity.banner.model.ButtonAction.CUSTOM_TEXT;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Component
@ParametersAreNonnullByDefault
public class BannerPreprocessor implements CopyPreprocessorTypeSupport<BannerWithAdGroupId> {
    private static final Logger logger = LoggerFactory.getLogger(BannerPreprocessor.class);

    private static final int INSERT_CHUNK_SIZE = 1000;

    private final BannerImagePoolRepository bannerImagePoolRepository;
    private final CalloutService calloutService;

    public BannerPreprocessor(BannerImagePoolRepository bannerImagePoolRepository,
                              CalloutService calloutService) {
        this.bannerImagePoolRepository = bannerImagePoolRepository;
        this.calloutService = calloutService;
    }

    @Override
    public Class<BannerWithAdGroupId> getTypeClass() {
        return BannerWithAdGroupId.class;
    }

    @Override
    public void preprocess(List<BannerWithAdGroupId> banners, CopyOperationContainer copyContainer) {
        preinsertBannerImagePool(banners, copyContainer);
        preinsertCallouts(banners, copyContainer);
        purgePhoneId(banners, copyContainer);
        removeButtonCaptions(banners);
        removeMulticardIds(banners);
        clearFlagsOnCpmYndxFrontpageCampaigns(banners, copyContainer);
    }

    @Override
    public void preprocess(BannerWithAdGroupId banner, CopyOperationContainer copyContainer) {
        preprocess(List.of(banner), copyContainer);
    }

    /**
     * Сбрасывает флаги у баннеров CpmYndxFrontpage кампаний так же как это делает perl
     */
    private void clearFlagsOnCpmYndxFrontpageCampaigns(
            List<BannerWithAdGroupId> banners, CopyOperationContainer copyContainer) {
        Map<Long, CampaignType> targetCampaignTypeByGroupId = copyContainer.getTargetCampaignTypeByGroupId();
        banners.forEach(banner -> {
            if (banner instanceof BannerWithFlags) {
                BannerWithFlags bannerWithFlags = (BannerWithFlags) banner;
                CampaignType bannerCampaignType = targetCampaignTypeByGroupId.get(banner.getAdGroupId());
                if (bannerCampaignType == CampaignType.CPM_YNDX_FRONTPAGE) {
                    bannerWithFlags.setFlags(null);
                }
            }
        });
    }

    /**
     * Копируем ручками BannerImagePool, не можем сделать этого стандартными средствами, так как в BannerImagePool два
     * id, id:Long, которым никто не пользуется, и image_hash:String, которым пользуются все. Даже alias не можем
     * сделать с image_hash в id, так как id уже занят. Фрустрация.
     */
    private void preinsertBannerImagePool(List<BannerWithAdGroupId> bannersWithAnyImages,
                                          CopyOperationContainer copyContainer) {
        if (!copyContainer.isCopyingBetweenClients()) {
            return;
        }

        var bannerImageHashes = StreamEx.of(bannersWithAnyImages)
                .select(BannerWithBannerImage.class)
                .map(BannerWithBannerImage::getImageHash)
                .nonNull();
        var imageHashes = StreamEx.of(bannersWithAnyImages)
                .select(BannerWithImage.class)
                .map(BannerWithImage::getImageHash)
                .nonNull();
        var multicardImageHashes = StreamEx.of(bannersWithAnyImages)
                .select(BannerWithMulticardSet.class)
                .flatCollection(BannerWithMulticardSet::getMulticards)
                .map(BannerMulticard::getImageHash);

        var anyImageHashes = bannerImageHashes
                .append(imageHashes)
                .append(multicardImageHashes)
                .toList();

        if (anyImageHashes.isEmpty()) {
            return;
        }

        List<BannerImageFromPool> poolImages = StreamEx.ofSubLists(anyImageHashes, SqlUtils.TYPICAL_SELECT_CHUNK_SIZE)
                .flatCollection(imageHashChunk -> bannerImagePoolRepository
                        .getBannerImageFromPoolsByHashes(copyContainer.getShardFrom(), copyContainer.getClientIdFrom(),
                                imageHashChunk)
                        .values())
                .toList();

        poolImages.forEach(img -> {
            img.setId(null);
            img.setClientId(copyContainer.getClientIdTo().asLong());
            img.setCreateTime(LocalDateTime.now());
        });

        StreamEx.ofSubLists(poolImages, INSERT_CHUNK_SIZE).forEach(imageChunk -> bannerImagePoolRepository
                .addOrUpdateImagesToPool(copyContainer.getShardTo(), copyContainer.getClientIdTo(), imageChunk));
    }

    /**
     * Не можем копировать уточнения обычным способом, так как они хранятся на баннере в виде списка id, который при
     * копировании между клиентами нужно заменять на список скопированных уточнений.
     */
    private void preinsertCallouts(List<BannerWithAdGroupId> entities, CopyOperationContainer copyContainer) {
        if (!copyContainer.isCopyingBetweenClients()) {
            return;
        }

        var calloutIds = StreamEx.of(entities)
                .select(BannerWithCallouts.class)
                .flatMap(b -> b.getCalloutIds().stream())
                .toList();

        List<Callout> callouts = calloutService.getChunked(copyContainer.getClientIdFrom(),
                copyContainer.getOperatorUid(), calloutIds, SqlUtils.TYPICAL_SELECT_CHUNK_SIZE);

        Map<Long, Callout> oldIdToCallout = listToMap(callouts, Callout::getId);

        callouts.forEach(c -> {
            c.setId(null);
            c.setClientId(copyContainer.getClientIdTo().asLong());
        });

        var result = calloutService.copyChunked(
                copyContainer, callouts, Applicability.PARTIAL, INSERT_CHUNK_SIZE);

        logFailedResults(result.getValidationResult(), logger);

        var oldIdToNewId = EntryStream.of(oldIdToCallout).mapValues(Callout::getId).toMap();
        StreamEx.of(entities).select(BannerWithCallouts.class)
                .filter(bwc -> bwc.getCalloutIds() != null && !bwc.getCalloutIds().isEmpty())
                .forEach(bwc -> bwc.setCalloutIds(
                        StreamEx.of(bwc.getCalloutIds()).map(oldIdToNewId::get).nonNull().toList()));
    }

    /**
     * При добавлении заголовок у кнопки должен быть пустым, для всех действий кроме CUSTOM_TEXT.
     */
    private void removeButtonCaptions(List<BannerWithAdGroupId> entities) {
        StreamEx.of(entities)
                .select(BannerWithButton.class)
                .filter(b -> b.getButtonAction() != CUSTOM_TEXT)
                .forEach(b -> b.setButtonCaption(null));
    }

    /**
     * Сбрасываем PhoneId при копировании между клиентами.
     * TODO: убрать при реализации DIRECT-136949
     */
    private void purgePhoneId(List<BannerWithAdGroupId> entities, CopyOperationContainer copyContainer) {
        if (!copyContainer.isCopyingBetweenClients()) {
            return;
        }

        StreamEx.of(entities)
                .select(BannerWithPhone.class)
                .forEach(b -> b.setPhoneId(null));
    }

    /**
     * Очищает id карточек мультибаннера, так как при создании баннера должны создаваться новые карточки
     */
    private void removeMulticardIds(List<BannerWithAdGroupId> entities) {
        StreamEx.of(entities)
                .select(BannerWithMulticardSet.class)
                .flatCollection(BannerWithMulticardSet::getMulticards)
                .forEach(multicard -> multicard.withMulticardId(null));
    }
}
