package ru.yandex.direct.core.copyentity.preprocessors

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.entity.domain.service.DomainService
import ru.yandex.direct.core.entity.mobileapp.model.MobileApp

@Component
class MobileAppCopyPreprocessor(
    private val domainService: DomainService,
) : CopyPreprocessorTypeSupport<MobileApp> {
    override fun getTypeClass() = MobileApp::class.java

    override fun preprocess(mobileApps: List<MobileApp>, copyContainer: CopyOperationContainer) {
        if (copyContainer.isCopyingBetweenClients) {
            mobileApps.forEach {
                it.mobileContentId = null
                it.domainId = null
            }
            if (copyContainer.isCopyingBetweenShards) {
                // Можно было бы сделать связь от мобильных приложений к мобильному контенту и доменам,
                // мобильный контент сделать наследником Entity<Long> и протянуть от него связь к доменам тоже,
                // однако операция добавления мобильных приложений MobileAppAddOperation уже сама ищет и добавляет
                // несуществующие домены (связанные с приложением) и мобильный контент, поэтому проще скопировать
                // домены мобильного контента между шардами тут, в препроцессоре, чем серьезно переделывать граф,
                // и дорабатывать сервисы и операции добавления доменов, мобильного контента и приложений.
                //
                // В саму операцию MobileAppAddOperation этот код не добавляем, потому что он нужен только при
                // копировании мобильных приложений. Операция же, в подавляющем большинстве случаев, используется для
                // обычного создания мобильных приложений в рамках одного шарда, и каждый раз выполнять там лишний
                // код - нет смысла. При этом понять в самой операции, что исходное мобильное приложение пришло из
                // другого шарда, никакой возможности нет. Поэтому копируем домены мобильного контента при межшардовом
                // копировании в препроцессоре.
                val domainIds: List<Long> = mobileApps.mapNotNull { it.mobileContent?.publisherDomainId }
                domainService.copyDomainsBetweenShards(copyContainer.shardFrom, copyContainer.shardTo, domainIds)
            }
        }
    }

    override fun preprocess(mobileApp: MobileApp, copyContainer: CopyOperationContainer) {
        preprocess(listOf(mobileApp), copyContainer)
    }
}
