package ru.yandex.direct.core.copyentity.preprocessors

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperation
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.EntityLoadService.SELECT_CHUNK_SIZE
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.currency.service.CurrencyConverterFactory
import ru.yandex.direct.core.entity.retargeting.model.ConditionType
import ru.yandex.direct.core.entity.retargeting.model.Retargeting
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.operation.Applicability

@Component
class RetargetingCopyPreprocessor(
    private val currencyConverterFactory: CurrencyConverterFactory,
    private val retargetingConditionService: RetargetingConditionService,
) : CopyPreprocessorTypeSupport<Retargeting> {

    private val logger: Logger = LoggerFactory.getLogger(RetargetingCopyPreprocessor::class.java)

    override fun getTypeClass() = Retargeting::class.java

    override fun preprocess(entities: List<Retargeting>, copyContainer: CopyOperationContainer) {
        entities.forEach { preprocess(it, copyContainer) }

        copyUserProfilesSameClient(entities, copyContainer)
    }

    override fun preprocess(entity: Retargeting, copyContainer: CopyOperationContainer) {
        fixPriceForCpmPriceCampaign(entity, copyContainer)
        convertPrice(entity, copyContainer)
    }

    private fun fixPriceForCpmPriceCampaign(entity: Retargeting, copyContainer: CopyOperationContainer) {
        if (entity.priceContext != null
            && copyContainer.targetCampaignTypeByGroupId[entity.adGroupId] == CampaignType.CPM_PRICE
        ) {
            entity.priceContext = null
        }
    }

    /**
     * Смена валюты возможна только при копировании кампании, то есть стратегия либо не меняется, либо меняется на
     * автобюджет, при котором задавать ставки по-умолчанию не нужно, поэтому просто конвертируем те что есть
     */
    private fun convertPrice(entity: Retargeting, copyContainer: CopyOperationContainer) {
        val currencyCodeFrom = copyContainer.clientFrom.workCurrency
        val currencyCodeTo = copyContainer.clientTo.workCurrency
        val currencyTo = Currencies.getCurrency(currencyCodeTo)

        if (currencyCodeFrom != currencyCodeTo) {
            val converter = currencyConverterFactory.createConverter(currencyCodeFrom, currencyCodeTo)
            entity.priceContext = converter.convert(entity.priceContext, currencyTo.minPrice, currencyTo.maxPrice)
        }
    }

    // Правильней было бы копировать объекты, добавив в граф объектов связь между ретаргетингами
    // и условиями ретаргетинга, а так же вершины с нужными условиями ретаргетинга.
    // Однако профили пользователя - это объекты класса RetargetingCondition, наследника ClientLibraryObject.
    // При копировании внутри клиента такие объекты не будут загружены сервисом EntityLoadService, а если их
    // принудительно догрузить в медиаторе, то их все равно не скопирует CopyOperation по той же причине.
    // Кроме того, в таком случае CopyOperation выбросит все ретаргетинги, для которых родительские условия
    // ретаргетинга будут другого типа. Потому что при копировании выбрасываются все потомки, у которых родителей
    // не было в графе объектов, а связь между родителями и потомками была. А если добавить в граф объектов
    // условия ретаргетинга без связи с ретаргетингами, тогда до копирования условий ретаргетинга
    // CopyOperation вообще не дойдет.
    //
    // Поэтому отставим копирование профилей пользователя внутри препроцессора до тех пор, пока кто-нибудь не решится
    // доработать CopyOperation так, чтобы она копировала весь загруженный граф, невзирая на ограничения и
    // не выбрасывала потомков тех родителей, которых изначально в графе копирования не было.
    private fun copyUserProfilesSameClient(entities: List<Retargeting>, copyContainer: CopyOperationContainer) {
        if (copyContainer.isCopyingBetweenClients) {
            return
        }

        val retCondIds = entities
            .mapNotNull { it.retargetingConditionId }
            .toSet()

        val retConditions = retargetingConditionService
            .getChunked(copyContainer.clientIdFrom, copyContainer.operatorUid, retCondIds.toList(), SELECT_CHUNK_SIZE)

        val userProfiles = retConditions
            .filter { it.type == ConditionType.interests }

        val userProfilesByOldId = userProfiles
            .associateBy { it.id }

        userProfiles
            .onEach { it.id = null }

        val result = retargetingConditionService
            .copyChunked(copyContainer, userProfiles, Applicability.PARTIAL, CopyOperation.INSERT_CHUNK_SIZE)

        if (result.errorCount > 0) {
            logger.error("The following errors occured while copying UserProfiles:\n{}",
            result.validationResult.flattenErrors())
        }

        entities
            .filter { it.retargetingConditionId in userProfilesByOldId }
            .onEach { it.retargetingConditionId = userProfilesByOldId[it.retargetingConditionId]?.id }
    }
}
