package ru.yandex.direct.core.copyentity.preprocessors.campaign

import org.springframework.stereotype.Component
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.preprocessors.CopyPreprocessorTypeSupport
import ru.yandex.direct.core.entity.campaign.model.CampaignType
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CampaignsAutobudget
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform
import ru.yandex.direct.core.entity.campaign.model.DbStrategy
import ru.yandex.direct.core.entity.campaign.model.StrategyData
import ru.yandex.direct.core.entity.campaign.model.StrategyName
import ru.yandex.direct.core.entity.currency.service.CurrencyConverter
import ru.yandex.direct.core.entity.currency.service.CurrencyConverterFactory
import ru.yandex.direct.currency.Currencies
import ru.yandex.direct.currency.Currency
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStrategyName
import java.math.BigDecimal
import java.time.LocalDateTime

@Component
class CampaignWithStrategyCopyPreprocessor(
    private val currencyConverterFactory: CurrencyConverterFactory,
) : CopyPreprocessorTypeSupport<CampaignWithStrategy> {

    companion object {
        // Список взят отсюда:
        // https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Direct/Strategy.pm?rev=r8244976#L336
        private val strategiesWithBidderRestartTimeOptionNames: Set<StrategyName> = setOf(
            StrategyName.AUTOBUDGET_AVG_CPA,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP,
            StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER,
            StrategyName.AUTOBUDGET,
            StrategyName.AUTOBUDGET_AVG_CPI,
            StrategyName.AUTOBUDGET_CRR,
        )

        // Список взят отсюда:
        // https://a.yandex-team.ru/arc_vcs/direct/perl/protected/Campaign/Copy.pm?rev=r9103806#L590
        private val strategiesToClearChangesCounter: Set<StrategyName> = setOf(
            StrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
            StrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
        )
    }

    override fun getTypeClass() = CampaignWithStrategy::class.java

    override fun preprocess(campaign: CampaignWithStrategy, copyContainer: CopyOperationContainer) {
        if (campaign.id != null && copyContainer.getCampaignPreprocessFlagsById(campaign.id).isDropStrategyToDefault) {
            dropStrategyToAutobudgetWeekSum(campaign)
        }

        setCampaignStrategyOptions(campaign)

        if (!copyContainer.isCopyingBetweenClients ||
            copyContainer.clientFrom.workCurrency == copyContainer.clientTo.workCurrency) {
            return
        }

        val converter = currencyConverterFactory.createConverter(
            copyContainer.clientFrom.workCurrency,
            copyContainer.clientTo.workCurrency
        )

        preprocessStrategy(campaign, converter)
    }

    private fun dropStrategyToAutobudgetWeekSum(campaign: CampaignWithStrategy) {
        val oldStrategyData: StrategyData = campaign.strategy.strategyData

        campaign.strategy = DbStrategy()
            .withStrategyName(StrategyName.AUTOBUDGET)
            .withPlatform(CampaignsPlatform.BOTH)
            .withAutobudget(CampaignsAutobudget.YES)
            .withStrategyData(
                StrategyData()
                    .withName(CampaignsStrategyName.autobudget.literal)
                    .withLastBidderRestartTime(LocalDateTime.now())
                    .withSum(oldStrategyData.sum)
                    .withBid(oldStrategyData.bid)
            ) as DbStrategy
    }

    private fun preprocessStrategy(
        campaign: CampaignWithStrategy,
        converter: CurrencyConverter,
    ) {
        val noMaxValue: BigDecimal? = null
        val currencyTo: Currency = Currencies.getCurrency(converter.currencyTo)
        val isPerformance = campaign.type == CampaignType.PERFORMANCE

        val strategy: StrategyData = campaign.strategy.strategyData

        if (strategy.avgBid != null) {
            val minAvgBid: BigDecimal = getMinAvgBid(currencyTo, isPerformance)
            val maxAvgBid = currencyTo.maxAutobudgetBid
            strategy.avgBid = converter.convert(strategy.avgBid, minAvgBid, maxAvgBid)
        }

        val minAvgCpa: BigDecimal = getMinAvgCpa(currencyTo, isPerformance)
        if (strategy.avgCpa != null) {
            strategy.avgCpa = converter.convert(strategy.avgCpa, minAvgCpa, noMaxValue)
        }

        if (strategy.avgCpi != null) {
            strategy.avgCpi = converter.convert(strategy.avgCpi, minAvgCpa, noMaxValue)
        }

        if (strategy.avgCpm != null) {
            val minAvgCpm = currencyTo.minAutobudgetAvgCpm
            val maxAvgCpm = currencyTo.maxCpmPrice
            strategy.avgCpm = converter.convert(strategy.avgCpm, minAvgCpm, maxAvgCpm)
        }

        if (strategy.filterAvgBid != null) {
            val minFilterAvgBid = currencyTo.minCpcCpaPerformance
            val maxFilterAvgBid = currencyTo.maxAutobudgetBid
            val filterAvgBid: BigDecimal = converter.convert(strategy.filterAvgBid, minFilterAvgBid, maxFilterAvgBid)
            strategy.filterAvgBid = filterAvgBid
        }

        if (strategy.filterAvgCpa != null) {
            val minFilterAvgCpa = currencyTo.minCpcCpaPerformance
            strategy.filterAvgCpa = converter.convert(strategy.filterAvgCpa, minFilterAvgCpa, noMaxValue)
        }

        if (strategy.bid != null) {
            val minBid: BigDecimal = getMinBid(currencyTo, isPerformance, strategy)
            val maxBid = currencyTo.maxAutobudgetBid
            strategy.bid = converter.convert(strategy.bid, minBid, maxBid)
        }

        if (strategy.sum != null) {
            val minSum = currencyTo.minAutobudget
            val maxSum = currencyTo.maxAutobudget
            strategy.sum = converter.convert(strategy.sum, minSum, maxSum)
        }

        if (strategy.budget != null) {
            val minBudget = currencyTo.minDailyBudgetForPeriod
            val maxBudget = currencyTo.maxDailyBudgetForPeriod
            strategy.budget = converter.convert(strategy.budget, minBudget, maxBudget)
        }

    }

    private fun setCampaignStrategyOptions(campaign: CampaignWithStrategy) {
        val now: LocalDateTime = LocalDateTime.now()

        if (strategiesWithBidderRestartTimeOptionNames.contains(campaign.strategy.strategyName)) {
            campaign.strategy.strategyData.lastBidderRestartTime = now
        }

        if (strategiesToClearChangesCounter.contains(campaign.strategy.strategyName)) {
            campaign.strategy.strategyData.dailyChangeCount = 1
            campaign.strategy.strategyData.lastUpdateTime = now
        }
    }

    private fun getMinAvgBid(currencyTo: Currency, isPerformance: Boolean): BigDecimal {
        return if (isPerformance) currencyTo.minCpcCpaPerformance else currencyTo.minAutobudgetAvgPrice
    }

    private fun getMinAvgCpa(currencyTo: Currency, isPerformance: Boolean): BigDecimal {
        return if (isPerformance) currencyTo.minCpcCpaPerformance else currencyTo.minAutobudgetAvgCpa
    }

    private fun getMinBid(currencyTo: Currency, isPerformance: Boolean, strategyData: StrategyData): BigDecimal {
        var minBid = currencyTo.minAutobudgetBid
        if (isPerformance) {
            minBid = currencyTo.minCpcCpaPerformance
        } else if (strategyData.avgCpi != null && strategyData.avgCpa > BigDecimal.ZERO) {
            minBid = currencyTo.minAutobudgetAvgCpa
        }
        return minBid
    }
}
