package ru.yandex.direct.core.copyentity.preprocessors.campaign

import org.springframework.stereotype.Component
import ru.yandex.direct.common.net.IpUtils
import ru.yandex.direct.common.net.NetAcl
import ru.yandex.direct.core.copyentity.CopyOperationContainer
import ru.yandex.direct.core.copyentity.preprocessors.CopyPreprocessorTypeSupport
import ru.yandex.direct.core.copyentity.translations.RenameProcessor
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate
import ru.yandex.direct.core.entity.campaign.model.CampaignWithStrategy
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign
import ru.yandex.direct.core.entity.campaign.model.StrategyData
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CampaignWithOptionalRequireFiltrationByDontShowDomainsValidator.canRequireFiltrationByDontShowDomains
import ru.yandex.direct.core.entity.campaign.service.validation.type.bean.CommonCampaignBeanValidator
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.user.model.User
import ru.yandex.direct.feature.FeatureName
import java.time.LocalDate

@Component
class CommonCampaignCopyPreprocessor(
    private val renamePreprocessor: RenameProcessor,
    private val featureService: FeatureService,
    private val netAcl: NetAcl,
) : CopyPreprocessorTypeSupport<CommonCampaign> {
    override fun getTypeClass() = CommonCampaign::class.java

    override fun preprocess(entity: CommonCampaign, copyContainer: CopyOperationContainer) {
        entity.name = renamePreprocessor.generateCampaignCopyName(entity, copyContainer)
        entity.agencyId = null
        entity.agencyUid = null
        entity.walletId = null
        entity.clientId = null
        entity.isServiceRequested = null
        entity.copiedFrom = entity.id

        removeInternalDisabledIps(entity)

        val shouldStopCampaignByEndDateChange = resetCampaignStartAndEndTime(entity)

        val shouldForceStop = copyContainer.flags.isStopCopiedCampaigns
            || shouldStopCampaignByEndDateChange
            || entity.statusModerate == CampaignStatusModerate.NO

        entity.statusShow =
            if (shouldForceStop) {
                false
            } else if (copyContainer.flags.isCopyCampaignStatuses) {
                entity.statusShow
            } else {
                true
            }

        entity.statusModerate =
            if (copyContainer.flags.isCopyCampaignStatuses
                && entity.statusModerate != CampaignStatusModerate.NEW
            ) {
                CampaignStatusModerate.READY
            } else {
                CampaignStatusModerate.NEW
            }

        val canRequireFiltrationByDontShowDomainsFeatureEnabled = featureService
            .isEnabledForClientId(copyContainer.clientIdTo, FeatureName.CAN_REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS)
        val canRequireFiltrationByDontShowDomainsInCpmFeatureEnabled = featureService
            .isEnabledForClientId(
                copyContainer.clientIdTo,
                FeatureName.CAN_REQUIRE_FILTRATION_BY_DONT_SHOW_DOMAINS_IN_CPM
            )
        if (!canRequireFiltrationByDontShowDomains(
                entity.type,
                canRequireFiltrationByDontShowDomainsFeatureEnabled,
                canRequireFiltrationByDontShowDomainsInCpmFeatureEnabled
            )
        ) {
            entity.requireFiltrationByDontShowDomains = null
        }

        if (copyContainer.isCopyingBetweenClients) {
            entity.currency = copyContainer.clientTo.workCurrency
            if (!copyContainer.flags.isCopyNotificationSettings) {
                setNotificationSettingsFromUser(entity, copyContainer.chiefUserTo)
            }
        }
    }

    /**
     * Сбрасываем время начала на текущую дату. Если дата окончания кампании была задана, и оказалась меньше чем дата
     * старта, то тоже сбрасываем на текущую дату
     *
     * @return `true`, если кампания должна быть принудительно остановлена из-за смены `endDate` на сегодня
     */
    private fun resetCampaignStartAndEndTime(campaign: CommonCampaign): Boolean {
        val today = LocalDate.now()
        val campaignEndInPast = campaign.endDate != null && campaign.endDate < today

        if (campaign.startDate < today) {
            campaign.startDate = today

            // Можно было бы вставить в CampaignWithStrategyCopyPreprocessor, но порядок выполнения препроцессоров
            // не определен, а нам нужно гарантировать, что код корректировки дат стратегии выполнится строго после
            // кода корректировки даты начала кампании
            val minCampaignEndDate = if (campaign is CampaignWithStrategy) {
                fixCampaignStrategyDatesAndGetCampaignMinEndDateIfNeeded(campaign) ?: today
            } else {
                today
            }

            if (campaign.endDate != null && campaign.endDate < minCampaignEndDate) {
                campaign.endDate = minCampaignEndDate
            }
        }

        return campaignEndInPast
    }

    private fun fixCampaignStrategyDatesAndGetCampaignMinEndDateIfNeeded(
        campaign: CampaignWithStrategy,
    ): LocalDate? {
        val strategyData: StrategyData? = campaign.strategy?.strategyData

        if (strategyData != null && strategyData.start != null && strategyData.start < campaign.startDate) {
            strategyData.start = campaign.startDate

            if (strategyData.finish != null && strategyData.finish <= strategyData.start) {
                // Нужно обойти дефект StrategyDefects.strategyPeriodDaysCountLessThanMin()
                strategyData.finish = strategyData.start.plusDays(1)
                // Нужно обойти дефект StrategyDefects.strategyEndDateIsAfterCampaignEndDate(),
                // поэтому возвращаем новую минимальную дату окончания кампании
                return strategyData.finish
            }
        }
        return null
    }

    private fun setNotificationSettingsFromUser(campaign: CommonCampaign, chiefUserTo: User) {
        // fio задается в chiefUserTo.fio в операции добавления кампании CommonCampaignAddOperationSupport,
        // здесь не дублируем
        campaign.fio = null
        campaign.email = chiefUserTo.email
        campaign.enableSendAccountNews = chiefUserTo.sendAccNews
    }

    /**
     * Среди отключенных ip-адресов кампании могут встречаться внутренние адреса яндекса, а такие кампании
     * не пройдут валидацию. Поэтому убираем такие ip-адреса
     *
     * @see CommonCampaignBeanValidator
     */
    private fun removeInternalDisabledIps(entity: CommonCampaign) {
        if (entity.disabledIps != null) {
            entity.disabledIps = entity.disabledIps
                .filter { !netAcl.isInternalIp(IpUtils.ipFromString(it)) }
                .toList()
                .ifEmpty { null }
        }
    }
}
