package ru.yandex.direct.core.copyentity.prevalidators;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityContext;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.CampaignContainsAdGroups;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettingsProvider;
import ru.yandex.direct.core.entity.adgroup.service.geotree.AdGroupGeoTreeProviderFactory;
import ru.yandex.direct.core.entity.adgroup.service.validation.AddAdGroupValidationService;
import ru.yandex.direct.core.entity.campaign.model.CampaignTypeSource;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.hypergeo.service.HyperGeoService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.copyentity.CopyValidationResultUtils.filterDefects;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefectIds.Number.MAX_ADGROUPS_IN_CAMPAIGN;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@ParametersAreNonnullByDefault
@Component
public class AdGroupCopyPrevalidator implements EntityPrevalidator {
    private final AddAdGroupValidationService adGroupValidationService;
    private final AdGroupBsTagsSettingsProvider adGroupBsTagsSettingsProvider;
    private final AdGroupGeoTreeProviderFactory adGroupGeoTreeProviderFactory;
    private final CampaignRepository campaignRepository;
    private final ClientGeoService clientGeoService;
    private final HyperGeoService hyperGeoService;

    public AdGroupCopyPrevalidator(AddAdGroupValidationService adGroupValidationService,
                                   AdGroupBsTagsSettingsProvider adGroupBsTagsSettingsProvider,
                                   AdGroupGeoTreeProviderFactory adGroupGeoTreeProviderFactory,
                                   CampaignRepository campaignRepository,
                                   ClientGeoService clientGeoService,
                                   HyperGeoService hyperGeoService) {
        this.adGroupValidationService = adGroupValidationService;
        this.adGroupBsTagsSettingsProvider = adGroupBsTagsSettingsProvider;
        this.adGroupGeoTreeProviderFactory = adGroupGeoTreeProviderFactory;
        this.campaignRepository = campaignRepository;
        this.clientGeoService = clientGeoService;
        this.hyperGeoService = hyperGeoService;
    }

    @SuppressWarnings("rawtypes")
    public ValidationResult prevalidate(EntityContext context, CopyOperationContainer copyContainer) {
        var clientId = copyContainer.getClientIdTo();

        List<AdGroup> clonedAdGroups = StreamEx.of(context.getObjects(AdGroup.class)).map(AdGroup::copy).toList();
        context.setParentIds(clonedAdGroups, new CampaignContainsAdGroups());

        var campaignIds = listToSet(clonedAdGroups, AdGroup::getCampaignId);
        var campaignTypeSourceById = campaignRepository
                .getCampaignsTypeSourceMap(copyContainer.getShardTo(), campaignIds, false);

        var bsTagSettings = adGroupBsTagsSettingsProvider.getAdGroupBsTagsSettings(clonedAdGroups, clientId);
        var campaignTypes = EntryStream.of(campaignTypeSourceById)
                .mapValues(CampaignTypeSource::getCampaignType)
                .toMap();
        var clientGeoTree = clientGeoService.getClientTranslocalGeoTree(clientId);
        var geoTreeProvider = adGroupGeoTreeProviderFactory.create(clientGeoTree, campaignTypes);

        var hyperGeoIds = StreamEx.of(clonedAdGroups)
                .map(AdGroup::getHyperGeoId)
                .nonNull()
                .toSet();
        Map<Long, HyperGeo> hyperGeoByIds = !hyperGeoIds.isEmpty()
                ? hyperGeoService.getHyperGeoById(clientId, hyperGeoIds)
                : emptyMap();

        ValidationResult<List<AdGroup>, Defect> result = adGroupValidationService.validate(
                clonedAdGroups,
                hyperGeoByIds,
                campaignTypeSourceById,
                bsTagSettings,
                geoTreeProvider,
                clientId,
                copyContainer.getOperatorUid(),
                copyContainer.getShardTo());

        return filterDefects(result, Set.of(MAX_ADGROUPS_IN_CAMPAIGN));
    }

}
