package ru.yandex.direct.core.copyentity.prevalidators;

import java.util.HashMap;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityContext;
import ru.yandex.direct.model.Entity;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.ValidationResult;

/**
 * Осуществляет общую превалидацию по всем классам сущностей (превалидаторы настраиваются)
 */
@ParametersAreNonnullByDefault
public class Prevalidator {
    private final HashMap<Class<? extends Entity<?>>, EntityPrevalidator> prevalidators;

    public Prevalidator() {
        this.prevalidators = new HashMap<>();
    }

    /**
     * Добавляет новый типизированный превалидатор в пул валидации
     *
     * @param entityClass  какой тип сущности валидирует новый типизированный превалидатор
     * @param prevalidator добавляемый типизированный превалидатор
     */
    public <T extends Entity<?>> Prevalidator addPrevalidator(Class<T> entityClass, EntityPrevalidator prevalidator) {
        prevalidators.put(entityClass, prevalidator);
        return this;
    }

    /**
     * Осуществляет валидацию переданных сущностей всеми имеющимися превалидаторами
     */
    @SuppressWarnings("unchecked")
    public Map<Class, MassResult> prevalidate(EntityContext context, CopyOperationContainer copyContainer) {
        Map<Class, MassResult> results = new HashMap<>();
        for (Class<? extends Entity<?>> entityClass : context.getEntityClasses()) {
            if (!prevalidators.containsKey(entityClass)) {
                continue;
            }
            ValidationResult vr = prevalidators.get(entityClass).prevalidate(context, copyContainer);
            if (vr.hasAnyErrors()) {
                results.put(entityClass, new MassResult(null, vr, ResultState.BROKEN));
            }
        }

        return results;
    }
}
