package ru.yandex.direct.core.copyentity.translations;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupValidationService;
import ru.yandex.direct.core.entity.campaign.model.BaseCampaign;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants;
import ru.yandex.direct.core.entity.retargeting.Constants;
import ru.yandex.direct.core.entity.retargeting.model.RetargetingCondition;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.model.Entity;
import ru.yandex.direct.utils.StringUtils;

import static ru.yandex.direct.core.entity.retargeting.Constants.MAX_NAME_LENGTH;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class RenameProcessor {

    private static final int AD_GROUP_NAME_MAX_LENGTH = AdGroupValidationService.MAX_NAME_LENGTH;
    private static final int CAMPAIGN_NAME_MAX_LENGTH = CampaignConstants.MAX_CAMPAIGN_NAME_LENGTH;

    private final TranslationService translationService;

    public RenameProcessor(TranslationService translationService) {
        this.translationService = translationService;
    }

    /**
     * Генерирует имя для копирования с учетом локали и перевода. Имя генерируется методом
     * {@link StringUtils#generateCopyName(String, String)}.
     * @param name оригинальное имя
     * @param maxLength максимальная длина имени копии
     * @param copyNameTranslatablePrefix префикс для копии и ее номера
     * @param copyNameTranslatableReplacer заменитель оригинального имени, если имя копии выйдет длиннее maxLength
     * @param copyIdReplacer идентификатор копируемого объекта, в виде строки. Нужен, когда имя копии
     *                       вышло длиннее maxLength
     * @param locale локаль, в которой нужно сделать копию
     * @return имя для копии
     */
    public String generateTranslatableCopyName(
            String name, int maxLength, Translatable copyNameTranslatablePrefix,
            Translatable copyNameTranslatableReplacer, String copyIdReplacer, Locale locale) {
        String copyNamePrefix = translationService.translate(copyNameTranslatablePrefix, locale);
        String newName = StringUtils.generateCopyName(name, copyNamePrefix);
        if (newName.length() > maxLength) {
            String copyNameReplacer = translationService.translate(copyNameTranslatableReplacer, locale);
            newName = copyNameReplacer + copyIdReplacer;
            if (newName.length() > maxLength) {
                newName = copyIdReplacer;
            }
            if (newName.length() > maxLength) {
                newName = newName.substring(0, maxLength);
            }
        }
        return newName;
    }

    /**
     * Генерирует имена для копирования с учетом локали и перевода. Имена генерируются методом
     * {@link StringUtils#generateCopyNames(Collection, List, String)}. Если длины получившихся имен будут больше чем
     * maxNameLength, то слишком длинные имена имена будут заменены сначала на copyNameTranslatableReplacer + id,
     * а если это не поможет, то просто на id. После чего будет запущена вторая итерация уникализации имен и их обрезки.
     * Если после второй итерации имена окажутся не уникальными - значит не повезло, и метод вернет не уникальные имена.
     * Повторять уникализацию дальше нет смысла по следующим соображениям: Можно придумать такой набор данных,
     * при которых даже бесконечное повторение обрезки и уникализации не приведет к полной уникализации имен.
     * Однако на практике получить неуникальные данные после второго прохода - практически нереально. Это возможно
     * только в случае, когда длина copyNameTranslatableReplacer + id будет больше maxNameLength, а для реальных данных
     * она всегда меньше maxNameLength.
     *
     *
     * @param originalNames список уже существующих имен
     * @param entities список объектов, для которых нужно постараться получить уникальные имена
     * @param nameGetter функциональный метод получения имени объекта
     * @param maxNameLength максимальная длина имени объекта
     * @param copyNameTranslatablePrefix префикс для копии и ее номера
     * @param copyNameTranslatableReplacer заменитель оригинального имени, если имя копии выйдет длиннее maxNameLength
     * @param locale локаль, в которой нужно сделать копию
     * @param <T> тип сущности
     * @return список имен для копий.
     */
    public <T extends Entity<Long>> List<String> generateTranslatableCopyNames(
            Collection<String> originalNames, List<T> entities, Function<T, @Nullable String> nameGetter,
            int maxNameLength, Translatable copyNameTranslatablePrefix, Translatable copyNameTranslatableReplacer,
            Locale locale) {
        String copyNamePrefix = translationService.translate(copyNameTranslatablePrefix, locale);
        List<String> newNames = mapList(entities, nameGetter);
        List<String> uniqueNames = StringUtils.generateCopyNames(originalNames, newNames, copyNamePrefix);
        if (allNameLengthsNotGreaterThanMax(uniqueNames, maxNameLength)) {
            return uniqueNames;
        }
        // Если часть получившихся имен вышла длиннее чем maxNameLength, попробуем их сократить
        List<String> correctedNames = correctLongNames(
                entities, uniqueNames, maxNameLength, copyNameTranslatableReplacer, locale);
        // И прогоним уникализацию еще раз
        List<String> secondTurnUniqueNames =
                StringUtils.generateCopyNames(originalNames, correctedNames, copyNamePrefix);

        return allNameLengthsNotGreaterThanMax(secondTurnUniqueNames, maxNameLength) ? secondTurnUniqueNames :
                correctLongNames(entities, secondTurnUniqueNames, maxNameLength, copyNameTranslatableReplacer, locale);
    }

    private boolean allNameLengthsNotGreaterThanMax(List<String> names, int maxNameLength) {
        return names.stream().allMatch(name -> name == null || name.length() <= maxNameLength);
    }

    @NotNull
    private <T extends Entity<Long>> List<String> correctLongNames(
            List<T> entities,
            List<String> newNames,
            int maxNameLength,
            Translatable copyNameTranslatableReplacer,
            Locale locale) {
        List<String> correctedUniqueNames = new ArrayList<>(newNames.size());
        for (int i = 0; i < newNames.size(); i++) {
            String newName = newNames.get(i);
            Entity<Long> entity = entities.get(i);
            if (newName.length() > maxNameLength) {
                String copyNameReplacer = translationService.translate(copyNameTranslatableReplacer, locale);
                newName = copyNameReplacer + entity.getId();
                if (newName.length() > maxNameLength) {
                    newName = entity.getId().toString();
                }
                if (newName.length() > maxNameLength) {
                    newName = newName.substring(0, maxNameLength);
                }
            }
            correctedUniqueNames.add(newName);
        }
        return correctedUniqueNames;
    }

    /**
     * Возвращает новое имя копируемой группы с учетом локали и перевода, если группа копируется внутри
     * своей же кампании. Если группа переносится в другую кампанию, или копируется вместе с кампанией,
     * то вернет оригинальное имя группы, при необходимости обрезанное до максимальной допустимой длинны
     *
     * @param adGroup группа, которая будет копироваться
     * @param copyContainer конфигурация копирования
     * @return имя группы для копирования
     */
    public String generateAdGroupCopyNameIfNeeded(AdGroup adGroup, CopyOperationContainer copyContainer) {
        if (copyContainer.isCopyingToSameEntity(BaseCampaign.class, adGroup.getCampaignId())) {
            return generateAdGroupCopyName(adGroup.getName(), adGroup.getId(), copyContainer.getLocale());
        } else if (adGroup.getName().length() > AD_GROUP_NAME_MAX_LENGTH) {
            return adGroup.getName().substring(0, AD_GROUP_NAME_MAX_LENGTH);
        }
        return adGroup.getName();
    }

    /**
     * Генерирует новое имя группы методом {@link StringUtils#generateCopyName(String, String)} с префиксом копии
     * {@link CopyNamesTranslations#adGroupCopyPrefix()}. Если не укладываемся в лимит на имя группы
     * {@link AdGroupValidationService#MAX_NAME_LENGTH}, то заменяем имя группы на
     * {@link CopyNamesTranslations#adGroupLongNameCopyReplacer()} + {@code adGroup.getId()}
     */
    public String generateAdGroupCopyName(String adGroupName, Long adGroupId, Locale locale) {
        return generateTranslatableCopyName(
                adGroupName, AD_GROUP_NAME_MAX_LENGTH,
                CopyNamesTranslations.INSTANCE.adGroupCopyPrefix(),
                CopyNamesTranslations.INSTANCE.adGroupLongNameCopyReplacer(),
                String.valueOf(adGroupId), locale);
    }

    public String generateCampaignCopyName(CommonCampaign campaign, CopyOperationContainer copyContainer) {
        return generateCampaignCopyName(campaign.getName(), campaign.getId(), copyContainer.getLocale());
    }

    /**
     * Генерирует новое имя кампании методом {@link StringUtils#generateCopyName(String, String)} с префиксом копии
     * {@link CopyNamesTranslations#campaignNameCopyPrefix()}. Если не укладываемся в лимит на имя кампании
     * {@link CampaignConstants#MAX_CAMPAIGN_NAME_LENGTH},
     * то заменяем имя кампании на
     * {@link CopyNamesTranslations#campaignLongNameCopyReplacer()} + {@code campaign.getId()}
     */
    public String generateCampaignCopyName(String campaignName, Long campaignId, Locale locale) {
        return generateTranslatableCopyName(
                campaignName, CAMPAIGN_NAME_MAX_LENGTH,
                CopyNamesTranslations.INSTANCE.campaignNameCopyPrefix(),
                CopyNamesTranslations.INSTANCE.campaignLongNameCopyReplacer(),
                String.valueOf(campaignId), locale);
    }


    /**
     * Пытается сгенерировать уникальные имена условий ретаргетинга методом
     * {@link StringUtils#generateCopyNames(Collection, List, String)} с префиксом копии
     * {@link CopyNamesTranslations#retargetingConditionCopyPrefix()}. Если не укладываемся в лимит на имя
     * условия ретаргетинга {@link Constants#MAX_NAME_LENGTH}, то заменяем имя на
     * {@link CopyNamesTranslations#retargetingConditionLongNameCopyReplacer()} + {@link RetargetingCondition#getId()}
     */
    public void tryToUnifyRetargetingCopyNames(
            Collection<String> originalNames, List<RetargetingCondition> conditionsToUpdate, Locale locale) {
        List<String> uniqueCopiedNames = generateTranslatableCopyNames(
                originalNames,
                conditionsToUpdate,
                RetargetingCondition::getName,
                MAX_NAME_LENGTH,
                CopyNamesTranslations.INSTANCE.retargetingConditionCopyPrefix(),
                CopyNamesTranslations.INSTANCE.retargetingConditionLongNameCopyReplacer(),
                locale);
        EntryStream.zip(conditionsToUpdate, uniqueCopiedNames)
                .forKeyValue(RetargetingCondition::setName);
    }

    public static Locale getCopyLocaleByChiefUser(User chiefUser) {
        return Locale.forLanguageTag(chiefUser.getLang().getLangString());
    }
}
