package ru.yandex.direct.core.entity.abt.repository;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.jooq.Field;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.abt.service.UaasConditionParams;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.stream.Collectors.toMap;
import static org.jooq.impl.DSL.iif;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENTS;

@Repository
public class UaasInfoRepository {

    private static final Field<Boolean> IS_AGENCY_CLIENT =
            iif(CLIENTS.AGENCY_CLIENT_ID.isNotNull(), true, false).as("isAgencyClient");

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    public UaasInfoRepository(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper
    ) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
    }

    public Map<Long, UaasConditionParams> getUaasConditionsParams(List<Long> clientIds) {
        return shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).stream()
                .mapKeyValue(this::clientsWithFeaturesByShard)
                .flatMapToEntry(Function.identity())
                .toMap();
    }

    private Map<Long, UaasConditionParams> clientsWithFeaturesByShard(int shard, List<Long> clientIds) {
        if (clientIds.isEmpty()) {
            return Map.of();
        }
        List<ClientAttributes> clientsAttributes;
        clientsAttributes = dslContextProvider.ppc(shard)
                .selectDistinct(CLIENTS.CLIENT_ID, CLIENTS.CREATE_DATE, CLIENTS.ROLE, IS_AGENCY_CLIENT)
                .from(CLIENTS)
                .where(CLIENTS.CLIENT_ID.in(clientIds))
                .fetch(
                        record -> new ClientAttributes()
                                .withClientId(record.get(CLIENTS.CLIENT_ID))
                                .withClientStartDate(record.get(CLIENTS.CREATE_DATE).toString())
                                .withClientRole(record.get(CLIENTS.ROLE).getLiteral())
                                .withIsAgencyClient(record.get(IS_AGENCY_CLIENT))
                );
        return clientsAttributes.stream()
                .collect(toMap(clientAttributes -> clientAttributes.clientId,
                        clientAttributes -> new UaasConditionParams()
                                .withClientRole(clientAttributes.clientRole)
                                .withClientCreateDate(clientAttributes.clientCreateDate)
                                .withIsAgencyClient(clientAttributes.isAgencyClient),
                        UaasInfoRepository::mergeParams));
    }

    private static UaasConditionParams mergeParams(UaasConditionParams params1, UaasConditionParams params2) {
        return params1.withIsAgencyClient(params1.getIsAgencyClient() || params2.getIsAgencyClient());
    }

    private class ClientAttributes {
        private Long clientId;
        private String clientCreateDate;
        private String clientRole;
        private boolean isAgencyClient;

        public String getClientCreateDate() {
            return clientCreateDate;
        }

        ClientAttributes withClientId(Long clientId) {
            this.clientId = clientId;
            return this;
        }

        ClientAttributes withClientStartDate(String clientStartDate) {
            this.clientCreateDate = clientStartDate;
            return this;
        }

        ClientAttributes withClientRole(String clientRole) {
            this.clientRole = clientRole;
            return this;
        }

        public ClientAttributes withIsAgencyClient(boolean isAgencyClient) {
            this.isAgencyClient = isAgencyClient;
            return this;
        }
    }
}
