package ru.yandex.direct.core.entity.abt.service;

import java.util.Set;
import java.util.concurrent.ExecutionException;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.expression.EvaluationContext;
import org.springframework.expression.Expression;
import org.springframework.expression.MethodResolver;
import org.springframework.expression.TypedValue;
import org.springframework.expression.spel.SpelParserConfiguration;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.DataBindingPropertyAccessor;
import org.springframework.expression.spel.support.SimpleEvaluationContext;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;

import ru.yandex.misc.thread.ExecutionRuntimeException;

@Component
public class UaasConditionEvaluator {
    private static final Logger logger = LoggerFactory.getLogger(UaasConditionEvaluator.class);

    private Cache<String, Expression> expressionCache;
    private SpelExpressionParser spelExpressionParser;

    public UaasConditionEvaluator() {
        this.expressionCache = CacheBuilder.newBuilder()
                .maximumSize(1_000)
                .build();
        SpelParserConfiguration config = new SpelParserConfiguration();
        this.spelExpressionParser = new SpelExpressionParser(config);
    }

    public Boolean evaluateUnsafe(String condition, UaasConditionParams uaasConditionParams) {
        Expression expression;
        try {
            expression = expressionCache.get(condition, () -> spelExpressionParser.parseExpression(condition));
        } catch (ExecutionException e) {
            throw new ExecutionRuntimeException(e);
        }
        EvaluationContext evaluationContext =
                getEvaluationContext(uaasConditionParams);
        return expression.getValue(evaluationContext, Boolean.class);
    }

    public Boolean evaluate(String condition, UaasConditionParams uaasConditionParams) {
        try {
            return evaluateUnsafe(condition, uaasConditionParams);
        } catch (RuntimeException e) {
            logger.error("Failed to parse uaas condition: " + condition, e);
            return false;
        }
    }

    /**
     * Важно использовать {@link SimpleEvaluationContext}, {@link StandardEvaluationContext}, так при вычислении
     * выражения первый может оперировать только переданным объектом, а второй способен вызывать любой код,
     * например "T(java.lang.Runtime).getRuntime().exit(1)";
     */
    private EvaluationContext getEvaluationContext(UaasConditionParams uaasConditionParams) {
        return SimpleEvaluationContext.forPropertyAccessors(DataBindingPropertyAccessor.forReadOnlyAccess())
                .withRootObject(uaasConditionParams)
                .withMethodResolvers(containsFeatureResolver())
                .build();
    }

    private static MethodResolver containsFeatureResolver() {
        return (context, targetObject, name, argumentTypes) -> {
            // Проверяем, что это действительно вызов Set.contains(x)
            if (!"contains".equals(name) || !(targetObject instanceof Set) || argumentTypes.size() != 1) {
                return null;
            }

            return (evaluationContext, target, arguments) -> {
                Set targetSet = (Set) target;
                return new TypedValue(targetSet.contains(arguments[0]));
            };
        };
    }
}
