package ru.yandex.direct.core.entity.abt.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.NullNode;
import com.google.common.collect.Streams;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogUaasData;
import ru.yandex.direct.common.log.service.LogUaasDataService;
import ru.yandex.direct.core.entity.abt.container.AllowedFeatures;
import ru.yandex.direct.core.entity.abt.container.TestInfo;
import ru.yandex.direct.core.entity.abt.container.UaasInfoRequest;
import ru.yandex.direct.core.entity.abt.container.UaasInfoResponse;
import ru.yandex.direct.core.entity.abt.repository.UaasInfoRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.uaas.UaasClient;
import ru.yandex.direct.uaas.UaasRequest;
import ru.yandex.direct.uaas.UaasResponse;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.stream.Collectors.toList;

/**
 * Сервис для получения и обработки ab-экспериментов из uaas
 */
@Service
public class UaasInfoService {

    private static final Logger logger = LoggerFactory.getLogger(UaasInfoService.class);
    private static final String YANDEX_UID_COOKIE_TEMPLATE = "yandexuid=%s";
    private static final String YEXP_COOKIE_TEMPLATE = "yexp=%s";
    private final UaasClient uaasClient;
    private final String service;
    private final String handlerName;
    private final LogUaasDataService logUaasDataService;
    private final UaasInfoRepository uaasInfoRepository;
    private final UaasConditionEvaluator uaasConditionEvaluator;
    private final EnvironmentNameGetter environmentNameGetter;

    public UaasInfoService(UaasClient uaasClient, @Value("${uaas.service}") String service, @Value(
            "${uaas.handler}") String handlerName, LogUaasDataService logUaasDataService,
                    UaasInfoRepository uaasInfoRepository,
                    UaasConditionEvaluator uaasConditionEvaluator, EnvironmentNameGetter environmentNameGetter) {
        this.uaasClient = uaasClient;
        this.service = service;
        this.handlerName = handlerName;
        this.logUaasDataService = logUaasDataService;
        this.uaasInfoRepository = uaasInfoRepository;
        this.uaasConditionEvaluator = uaasConditionEvaluator;
        this.environmentNameGetter = environmentNameGetter;
    }

    /**
     * Для списка клиентов получает информацию об экспериментах в uaas
     * Извлекает информацию о номере эксперимента, фключенных фичах и условиях эксперимента
     * Проверяет условия экспериментов
     * Выбирает только те, для которых условие выполнено
     * Логирует обработанную информацию
     */
    public List<UaasInfoResponse> getInfo(Collection<UaasInfoRequest> uaasInfoRequests,
                                          AllowedFeatures allowedFeatures) {
        var uaasInfoList = uaasInfoRequests.stream()
                .map(this::getFeaturesFromUaas)
                .filter(Objects::nonNull)
                .collect(toList());

        var clientIdsWithCondition = uaasInfoList.stream()
                .filter(UaasFullInfo::hasCondition)
                .map(UaasFullInfo::getClientId)
                .collect(toList());
        Map<Long, UaasConditionParams> clientIdUaasConditionParamsMap =
                uaasInfoRepository.getUaasConditionsParams(clientIdsWithCondition);

        return uaasInfoList.stream()
                .map(info -> {
                    var uaasConditionParams = clientIdUaasConditionParamsMap.get(info.getClientId());
                    enrichUaasConditionParams(uaasConditionParams, info);
                    return checkConditionAndGetResponse(info, uaasConditionParams, allowedFeatures);
                })
                .collect(Collectors.toList());

    }

    public List<UaasInfoResponse> getInfo(Collection<UaasInfoRequest> uaasInfoRequests) {
        return getInfo(uaasInfoRequests, AllowedFeatures.allAllowed());
    }

    public UaasInfoResponse getInfo(UaasInfoRequest uaasInfoRequest, AllowedFeatures allowedFeatures) {
        return getInfo(List.of(uaasInfoRequest), allowedFeatures).get(0);
    }

    public UaasInfoResponse getInfo(UaasInfoRequest uaasInfoRequest) {
        return getInfo(List.of(uaasInfoRequest), AllowedFeatures.allAllowed()).get(0);
    }

    private UaasFullInfo getFeaturesFromUaas(UaasInfoRequest uaasInfoRequest) {
        var uaasClientIdRequest = new UaasRequest(service)
                .withHost(uaasInfoRequest.getHost())
                .withUserAgent(uaasInfoRequest.getUserAgent())
                .withIp(uaasInfoRequest.getIp())
                .withText(uaasInfoRequest.getText())
                .withCookies(getCookies(uaasInfoRequest.getYandexUid(), uaasInfoRequest.getYexpCookie()))
                .withCuid(uaasInfoRequest.getClientId() != null ? String.valueOf(uaasInfoRequest.getClientId()) :
                        null);

        try {
            UaasResponse uaasResponse = uaasClient.split(uaasClientIdRequest);
            return processUaasResponse(uaasInfoRequest, uaasResponse);
        } catch (RuntimeException ex) {
            logger.error("Failed to get features from uaas for " + uaasClientIdRequest, ex);
            return null;
        }
    }

    private String getCookies(String yandexUid, String yexpCookie) {
        List<String> cookiesList = new ArrayList<>();
        if (yandexUid != null) {
            cookiesList.add(String.format(YANDEX_UID_COOKIE_TEMPLATE, yandexUid));
        }
        if (yexpCookie != null) {
            cookiesList.add(String.format(YEXP_COOKIE_TEMPLATE, yexpCookie));
        }
        return cookiesList.isEmpty() ? null : String.join(";", cookiesList);
    }

    private UaasFullInfo processUaasResponse(UaasInfoRequest uaasInfoRequest, UaasResponse uaasResponse) {
        List<String> flags = uaasResponse.getFlags();
        List<TestInfo> tests = new ArrayList<>();
        boolean hasCondition = false;
        for (var flag : flags) {
            JsonNode flagsJsonNode;
            try {
                flagsJsonNode = JsonUtils.getObjectMapper().readTree(flag.getBytes());
            } catch (IOException e) {
                logger.error("Failed to parse flag for uaas " + flag, e);
                continue;
            }
            for (var handler : flagsJsonNode) {
                if (handler.has("HANDLER") && handlerName.equals(handler.get("HANDLER").asText())) {
                    var testInfo = getTestInfo(handler);
                    hasCondition = hasCondition || testInfo.hasCondition();
                    tests.add(testInfo);
                }
            }
        }
        return new UaasFullInfo()
                .withBoxesCrypted(uaasResponse.getBoxedCrypted())
                .withBoxes(uaasResponse.getBoxes())
                .withTests(tests)
                .withClientId(uaasInfoRequest.getClientId())
                .withYandexUid(uaasInfoRequest.getYandexUid())
                .withHasCondition(hasCondition)
                .withConfigVersion(uaasResponse.getConfigVersion())
                .withInterfaceLang(uaasInfoRequest.getInterfaceLang())
                .withEnabledFeatures(uaasInfoRequest.getEnabledFeatures());
    }

    /**
     * Если у теста есть condition, вычислияет его
     * Если condition не выполняется, то информация о тесте не логируется и не возвращается в ответе
     * Если condition выполняется, но тест содерижт в себе фичу, которой нет в списке допустимых, то информация о
     * тесте не логируется и не возвращается в ответе
     * В остальных случаях лог пишется, в ответ информация возвращается
     */

    private UaasInfoResponse checkConditionAndGetResponse(UaasFullInfo uaasInfo,
                                                          UaasConditionParams uaasConditionParams,
                                                          AllowedFeatures allowedFeatures) {
        Set<String> disablesTestIds = new HashSet<>();
        List<TestInfo> suitableTests = new ArrayList<>();
        List<String> notAllowedTestIds = new ArrayList<>();
        for (var testInfo : uaasInfo.getTests()) {
            if (testInfo.hasCondition() &&
                    !uaasConditionEvaluator.evaluate(testInfo.getCondition(), uaasConditionParams)) {
                disablesTestIds.addAll(testInfo.getTestIds());
            } else {
                if (allowedFeatures.isAllFeaturesAllowed(testInfo.getFeatures())) {
                    suitableTests.add(testInfo);
                } else {
                    notAllowedTestIds.addAll(testInfo.getTestIds());
                }
            }
        }

        disablesTestIds.addAll(notAllowedTestIds);
        var uaasInfoResponse = new UaasInfoResponse()
                .withClientId(ClientId.fromLong(uaasInfo.getClientId()))
                .withBoxesCrypted(uaasInfo.getBoxesCrypted())
                .withBoxes(filterByDisabledTestIds(uaasInfo.getBoxes(), disablesTestIds))
                .withBoxesCrypted(uaasInfo.getBoxesCrypted())
                .withConfigVersion(uaasInfo.getConfigVersion())
                .withTests(suitableTests);


        logUaasInfo(uaasInfo.getClientId(), uaasInfo.getYandexUid(), uaasInfoResponse);
        logNotAllowedTestIds(notAllowedTestIds);
        return uaasInfoResponse;
    }

    private String filterByDisabledTestIds(String expBoxes, Set<String> disabledTestIds) {
        if (disabledTestIds.isEmpty()) {
            return expBoxes;
        }
        return Arrays.stream(expBoxes.split(";"))
                .filter(expBox -> !disabledTestIds.contains(expBox.split(",")[0]))
                .collect(Collectors.joining(";"));
    }


    @SuppressWarnings("UnstableApiUsage")
    private List<String> getTestIdNode(JsonNode handlerNode) {
        if (!handlerNode.has("TESTID")) {
            logger.error("Incorrect handler format, expected CONTEXT node {}", handlerNode);
            return List.of();
        }
        return Streams.stream(handlerNode.get("TESTID").iterator()).map(JsonNode::asText).collect(Collectors.toList());
    }

    private String getConditionNode(JsonNode handlerNode) {
        return handlerNode.has("CONDITION") ? handlerNode.get("CONDITION").asText() : "";
    }

    private TestInfo getTestInfo(JsonNode handlerNode) {
        var testIds = getTestIdNode(handlerNode);
        var condition = getConditionNode(handlerNode);
        List<String> testFeatures = getFeatures(handlerNode);
        return new TestInfo()
                .withTestIds(testIds)
                .withCondition(condition)
                .withFeatures(testFeatures);
    }

    private List<String> getFeatures(JsonNode handlerNode) {
        List<String> testFeatures = new ArrayList<>();
        JsonNode featuresIterator = getFeaturesNode(handlerNode);
        for (var feature : featuresIterator) {
            testFeatures.add(feature.asText());
        }
        return testFeatures;
    }

    private JsonNode getFeaturesNode(JsonNode handlerNode) {
        JsonNode result = NullNode.getInstance();
        if (!handlerNode.has("CONTEXT")) {
            logger.warn("Incorrect handler format, expected CONTEXT node {}", handlerNode);
            return result;
        }
        var contextNode = handlerNode.get("CONTEXT");
        if (!contextNode.has("MAIN")) {
            logger.warn("Incorrect handler format, expected MAIN node {}", contextNode);
            return result;
        }
        var mainNode = contextNode.get("MAIN");
        if (!mainNode.has(handlerName)) {
            logger.warn("Incorrect handler format, expected custom handler ({}) node {}", handlerName, mainNode);
            return result;
        }
        var customHandlerNode = mainNode.get(handlerName);
        if (!customHandlerNode.has("flags")) {
            return result;
        }
        result = customHandlerNode.get("flags");
        return result;
    }

    private void logUaasInfo(Long clientId, String yandexUid, UaasInfoResponse response) {
        var uaasDataLogRecord = new LogUaasData()
                .withClientId(clientId)
                .withYandexUid(yandexUid)
                .withExpBoxes(response.getBoxes())
                .withFeatures(String.join(", ", response.getFeatures()))
                .withConfigVersion(response.getConfigVersion());
        logUaasDataService.logUaasData(uaasDataLogRecord);
    }

    /**
     * Логирует те test-id, содержащие фичу, которой нет с списке доступных
     */
    private void logNotAllowedTestIds(List<String> notAllowedTestIds) {
        if (!notAllowedTestIds.isEmpty()) {
            logger.error("Test ids {} contains features that are not in database", notAllowedTestIds);
        }
    }

    private void enrichUaasConditionParams(UaasConditionParams params, UaasFullInfo uaasFullInfo) {
        if (Objects.isNull(params)) {
            return;
        }
        params.withInterfaceLang(uaasFullInfo.getInterfaceLang());
        params.withEnv(environmentNameGetter.get());
        params.withManuallyEnabledFeatures(uaasFullInfo.getEnabledFeatures());
    }

    private class UaasFullInfo {
        private String yandexUid;
        private Long clientId;
        private String boxes;
        private String boxesCrypted;
        private List<TestInfo> tests;
        private boolean hasCondition;
        private String configVersion;
        private String interfaceLang;
        private Set<String> enabledFeatures;

        private String getYandexUid() {
            return yandexUid;
        }

        private UaasFullInfo withYandexUid(String yandexUid) {
            this.yandexUid = yandexUid;
            return this;
        }

        private Long getClientId() {
            return clientId;
        }

        private UaasFullInfo withClientId(Long clientId) {
            this.clientId = clientId;
            return this;
        }

        private String getBoxes() {
            return boxes;
        }

        private UaasFullInfo withBoxes(String boxes) {
            this.boxes = boxes;
            return this;
        }

        private String getBoxesCrypted() {
            return boxesCrypted;
        }

        private UaasFullInfo withBoxesCrypted(String boxesCrypted) {
            this.boxesCrypted = boxesCrypted;
            return this;
        }

        private List<TestInfo> getTests() {
            return tests;
        }

        private UaasFullInfo withTests(List<TestInfo> tests) {
            this.tests = tests;
            return this;
        }

        private boolean hasCondition() {
            return hasCondition;
        }

        private UaasFullInfo withHasCondition(boolean hasCondition) {
            this.hasCondition = hasCondition;
            return this;
        }

        private String getConfigVersion() {
            return configVersion;
        }

        private UaasFullInfo withConfigVersion(String configVersion) {
            this.configVersion = configVersion;
            return this;
        }

        public String getInterfaceLang() {
            return interfaceLang;
        }

        public UaasFullInfo withInterfaceLang(String interfaceLang) {
            this.interfaceLang = interfaceLang;
            return this;
        }

        public Set<String> getEnabledFeatures() {
            return enabledFeatures;
        }

        public UaasFullInfo withEnabledFeatures(Set<String> enabledFeatures) {
            this.enabledFeatures = enabledFeatures;
            return this;
        }
    }
}
