package ru.yandex.direct.core.entity.account.score.repository;

import java.util.Collection;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.account.score.model.AccountScore;
import ru.yandex.direct.core.entity.client.repository.ClientMapping;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppc.tables.AccountScore.ACCOUNT_SCORE;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class AccountScoreRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<AccountScore> jooqMapper;

    @Autowired
    public AccountScoreRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        jooqMapper = JooqMapperWithSupplierBuilder.builder(AccountScore::new)
                .map(property(AccountScore.CLIENT_ID, ACCOUNT_SCORE.CLIENT_ID))
                .map(property(AccountScore.DATE, ACCOUNT_SCORE.DATE))
                .map(convertibleProperty(AccountScore.FACTORS, ACCOUNT_SCORE.FACTORS_JSON,
                        ClientMapping::accountScoreFromDb, ClientMapping::accountScoreToDb))
                .map(property(AccountScore.SCORE, ACCOUNT_SCORE.SCORE))
                .map(property(AccountScore.TYPE, ACCOUNT_SCORE.TYPE))
                .build();
    }

    /**
     * Извлечь из базы самые свежие данные про показатели качества аккаунта для заданных клиентов
     *
     * @see <a href="https://yandex.ru/support/direct/efficiency/account-quality-index.html">Показатель
     * качества аккаунта</a> в пользовательской документации
     */
    public Collection<AccountScore> fetchLatestByClientIds(int shard, Collection<ClientId> clientIds) {
        ru.yandex.direct.dbschema.ppc.tables.AccountScore acFresher = ACCOUNT_SCORE.as("acFresher");
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(ACCOUNT_SCORE)
                .leftOuterJoin(acFresher).on(
                        ACCOUNT_SCORE.CLIENT_ID.eq(acFresher.CLIENT_ID).and(ACCOUNT_SCORE.DATE.lt(acFresher.DATE)))
                .where(ACCOUNT_SCORE.CLIENT_ID.in(clientIds))
                .and(acFresher.CLIENT_ID.isNull()) // новее нет
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Извлечь из базы данные про показатели качества аккаунта для заданного клиента.
     * Ручка не массовая т.к. не ожидается, что понадобиться где-то еще.
     */
    public List<AccountScore> getLatestAndPreviewsAccountScore(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(ACCOUNT_SCORE)
                .where(ACCOUNT_SCORE.CLIENT_ID.eq(clientId.asLong()))
                .orderBy(ACCOUNT_SCORE.DATE.desc())
                .limit(2)
                .fetch(jooqMapper::fromDb);
    }

    public void addAccountScores(int shard, Collection<AccountScore> accountScores) {
        new InsertHelper<>(dslContextProvider.ppc(shard), ACCOUNT_SCORE)
                .addAll(jooqMapper, accountScores)
                .executeIfRecordsAdded();
    }
}
