package ru.yandex.direct.core.entity.addition.callout.repository;

import java.math.BigInteger;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.addition.callout.container.CalloutSelection;
import ru.yandex.direct.core.entity.addition.callout.model.Callout;
import ru.yandex.direct.core.entity.addition.callout.model.CalloutDeleted;
import ru.yandex.direct.core.entity.addition.callout.model.CalloutsStatusModerate;
import ru.yandex.direct.dbschema.ppc.enums.AdditionsItemCalloutsStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersAdditionsAdditionsType;
import ru.yandex.direct.dbschema.ppc.tables.records.AdditionsItemCalloutsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Arrays.asList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_ADDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8;

@Repository
public class CalloutRepository {
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final JooqMapperWithSupplier<Callout> calloutMapper;
    private Collection<Field<?>> fieldsToRead;

    @Autowired
    public CalloutRepository(ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;

        calloutMapper = createCalloutMapper();
        fieldsToRead = calloutMapper.getFieldsToRead();
    }

    /**
     * Вычисляет хеш по полю text
     *
     * @param callout уточнение
     */
    protected static BigInteger calcHash(Callout callout) {
        checkNotNull(callout.getText(), "callout text cannot be null");
        return getMd5HalfHashUtf8(callout.getText());
    }

    private static JooqMapperWithSupplier<Callout> createCalloutMapper() {
        return JooqMapperWithSupplierBuilder.builder(Callout::new)
                .map(property(Callout.ID, ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                .map(property(Callout.TEXT, ADDITIONS_ITEM_CALLOUTS.CALLOUT_TEXT))
                .map(property(Callout.CLIENT_ID, ADDITIONS_ITEM_CALLOUTS.CLIENT_ID))
                .map(convertibleProperty(Callout.HASH, ADDITIONS_ITEM_CALLOUTS.HASH,
                        RepositoryUtils::bigIntegerFromULong,
                        RepositoryUtils::bigIntegerToULong))
                .map(property(Callout.FLAGS, ADDITIONS_ITEM_CALLOUTS.FLAGS))
                .map(convertibleProperty(Callout.STATUS_MODERATE, ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE,
                        CalloutsStatusModerate::fromSource,
                        CalloutsStatusModerate::toSource))
                .map(property(Callout.CREATE_TIME, ADDITIONS_ITEM_CALLOUTS.CREATE_TIME))
                .map(property(Callout.LAST_CHANGE, ADDITIONS_ITEM_CALLOUTS.LAST_CHANGE))
                .map(booleanProperty(Callout.DELETED, ADDITIONS_ITEM_CALLOUTS.IS_DELETED))
                .build();
    }

    /**
     * Добавляет уточнения.
     * Если уточнение существует (по hash) выставляет в модели id из базы и сбрасывает флаг удаления.
     * Новым уточнениям выставляет ид в модели
     *
     * @param shard    шард для запроса
     * @param callouts список уточнений для добавления
     * @return список ид уточнений
     */
    public List<Long> add(int shard, List<Callout> callouts) {
        callouts.forEach(callout -> callout.withHash(calcHash(callout)));

        Map<Pair<Long, BigInteger>, CalloutDeleted> existCalloutsMap =
                getExistingCallouts(shard, callouts);
        callouts.forEach(callout -> {
            CalloutDeleted existCallout = existCalloutsMap.get(Pair.of(callout.getClientId(), callout.getHash()));
            callout.withId(existCallout != null ? existCallout.getId() : null);
        });

        List<Callout> newCallouts = filterList(callouts, sl -> sl.getId() == null);
        Set<Long> existCalloutIdsDeleted = callouts.stream()
                .filter(callout -> callout.getId() != null
                        && existCalloutsMap.get(Pair.of(callout.getClientId(), callout.getHash())).getDeleted())
                .map(Callout::getId)
                .collect(Collectors.toSet());

        int inserted = addToAdditionItemCalloutsTable(shard, newCallouts);
        setDeleted(shard, existCalloutIdsDeleted, Boolean.FALSE);

        // Если количество новых коллаутов и количество новых записей в таблице не совпадают - нужно еще раз сходить
        // в базу, чтобы получить нужные id
        if (inserted == newCallouts.size()) {
            return mapList(callouts, Callout::getId);
        } else {
            return mapList(getExistingCallouts(shard, callouts).values(), CalloutDeleted::getId);
        }
    }

    /**
     * Установка флага удаления, также обновляет lastChange
     *
     * @param shard      шард для запроса
     * @param calloutIds список ид уточнений
     * @param deleted    true - удален, false - не удален
     */
    public void setDeleted(int shard, Collection<Long> calloutIds, Boolean deleted) {
        setDeleted(dslContextProvider.ppc(shard), calloutIds, deleted);
    }

    /**
     * Установка флага удаления, также обновляет lastChange
     *
     * @param context    контекст
     * @param calloutIds список ид уточнений
     * @param deleted    true - удален, false - не удален
     */
    public void setDeleted(DSLContext context, Collection<Long> calloutIds, Boolean deleted) {
        context
                .update(ADDITIONS_ITEM_CALLOUTS)
                .set(ADDITIONS_ITEM_CALLOUTS.IS_DELETED, booleanToLong(deleted))
                .set(ADDITIONS_ITEM_CALLOUTS.LAST_CHANGE, LocalDateTime.now())
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(calloutIds))
                .execute();
    }

    /**
     * Получение списка существующих уточнений
     *
     * @param shard          шард для запроса
     * @param clientId       id клиента
     * @param calloutIds     список ид уточнений
     * @param includeDeleted искать ли удаленные уточнения
     * @return список существующих ид уточнений
     */
    public Set<Long> getExistingCalloutIds(int shard, ClientId clientId, Collection<Long> calloutIds,
                                           Boolean includeDeleted) {
        SelectConditionStep<Record1<Long>> selectConditionStep = dslContextProvider.ppc(shard)
                .select(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.CLIENT_ID.eq(clientId.asLong()))
                .and(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(calloutIds));
        if (!includeDeleted) {
            selectConditionStep =
                    selectConditionStep.and(ADDITIONS_ITEM_CALLOUTS.IS_DELETED.eq(booleanToLong(Boolean.FALSE)));
        }

        return selectConditionStep
                .fetchSet(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID);
    }

    public Map<Long, Long> getClientIdsByCalloutIds(int shard, Collection<Long> calloutIds) {
        return dslContextProvider.ppc(shard)
                .select(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID, ADDITIONS_ITEM_CALLOUTS.CLIENT_ID)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(calloutIds))
                .fetchMap(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID, ADDITIONS_ITEM_CALLOUTS.CLIENT_ID);
    }

    public Set<Long> getExistingCalloutIds(int shard, ClientId clientId, Collection<Long> calloutIds) {
        return getExistingCalloutIds(shard, clientId, calloutIds, false);
    }

    /**
     * Получить множество id групп, у которых есть объявления с уточнениями
     *
     * @param shard      шард для запроса
     * @param adGroupIds список id групп
     * @return {@link Set<Long>} id групп, у которых есть объявления с уточнениями
     */
    public Set<Long> getAdGroupIdsWithExistingCallouts(int shard, ClientId clientId, Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(BANNERS.PID)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .join(BANNERS_ADDITIONS)
                .on(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.eq(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                .join(BANNERS)
                .on(BANNERS.BID.eq(BANNERS_ADDITIONS.BID))
                .where(ADDITIONS_ITEM_CALLOUTS.CLIENT_ID.eq(clientId.asLong()))
                .and(ADDITIONS_ITEM_CALLOUTS.IS_DELETED.eq(booleanToLong(Boolean.FALSE)))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.callout))
                .and(BANNERS.PID.in(adGroupIds))
                .fetchSet(BANNERS.PID);
    }

    /**
     * Получение списка id существующих уточнений, не помеченных как удаленные, привязанных к баннерам с указанными
     * id групп
     *
     * @param shard      шард для запроса
     * @param adGroupIds список id групп
     * @return отображение id баннера => список id привязаных к нему уточнений
     */
    public Map<Long, List<Long>> getExistingCalloutIdsByAdGroupIds(int shard, ClientId clientId,
                                                                   Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.PID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .join(BANNERS_ADDITIONS)
                .on(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.eq(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID))
                .join(BANNERS)
                .on(BANNERS.BID.eq(BANNERS_ADDITIONS.BID))
                .where(ADDITIONS_ITEM_CALLOUTS.CLIENT_ID.eq(clientId.asLong()))
                .and(ADDITIONS_ITEM_CALLOUTS.IS_DELETED.eq(booleanToLong(Boolean.FALSE)))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.callout))
                .and(BANNERS.PID.in(adGroupIds))
                .fetchGroups(BANNERS.PID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID);
    }

    /**
     * Получение списка id существующих уточнений, не помеченных как удаленные, привязанных к баннерам с указанными id
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return отображение id баннера => список id привязаных к нему уточнений
     */
    public Map<Long, List<Long>> getExistingCalloutIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS_ADDITIONS.BID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID)
                .from(BANNERS_ADDITIONS)
                .join(ADDITIONS_ITEM_CALLOUTS)
                .on(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.eq(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID))
                .where(BANNERS_ADDITIONS.BID.in(bannerIds))
                .and(ADDITIONS_ITEM_CALLOUTS.IS_DELETED.eq(booleanToLong(Boolean.FALSE)))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.callout))
                .fetchGroups(BANNERS_ADDITIONS.BID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID);
    }

    /**
     * Получение списка уникальных id существующих уточнений,
     * не помеченных как удаленные, привязанных к баннерам с указанными id
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return уникальные id уточнений
     */
    public Set<Long> getUniqueExistingCalloutIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .selectDistinct(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID)
                .from(BANNERS_ADDITIONS)
                .join(ADDITIONS_ITEM_CALLOUTS)
                .on(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.eq(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID))
                .where(BANNERS_ADDITIONS.BID.in(bannerIds))
                .and(ADDITIONS_ITEM_CALLOUTS.IS_DELETED.eq(booleanToLong(Boolean.FALSE)))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.callout))
                .fetchSet(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID);
    }

    /**
     * Получение списка существующих уточнений, не помеченных как удаленные, привязанных к баннерам с указанными id
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return мапа bannerId -> список additionId
     */
    public Map<Long, List<Callout>> getExistingCalloutsByBannerIds(int shard, Collection<Long> bannerIds) {
        Map<Long, List<Long>> calloutsIdsByBannerIds =
                getExistingCalloutIdsByBannerIds(shard, bannerIds);

        Set<Long> calloutIds = StreamEx.of(calloutsIdsByBannerIds.values())
                .toFlatCollection(Function.identity(), HashSet::new);

        List<Callout> callouts = get(shard, calloutIds);
        Map<Long, Callout> calloutByCalloutId = listToMap(callouts, Callout::getId);

        return EntryStream.of(calloutsIdsByBannerIds)
                .mapValues(longs -> mapList(longs, calloutByCalloutId::get))
                .toMap();
    }

    /**
     * Получение списка существующих уточнений клиента
     *
     * @param shard    шард для запроса
     * @param clientId ид клиента
     * @return список существующих уточнений клиента
     */
    public Collection<Callout> getClientExistingCallouts(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.CLIENT_ID.eq(clientId.asLong()))
                .fetch()
                .map(calloutMapper::fromDb);
    }

    /**
     * Получает мапу (clientId, hash) -> существующие уточнения
     *
     * @param shard    шард для запроса
     * @param callouts список уточнений. Для поиска используются поля clientId, hash, text
     * @return мапа (clientId, hash) -> {@link CalloutDeleted}
     */
    public Map<Pair<Long, BigInteger>, CalloutDeleted> getExistingCallouts(int shard,
                                                                           Collection<Callout> callouts) {
        Condition condition = StreamEx.of(callouts)
                .map(callout -> ADDITIONS_ITEM_CALLOUTS.CLIENT_ID.eq(callout.getClientId())
                        .and(ADDITIONS_ITEM_CALLOUTS.HASH.eq(ULong.valueOf(callout.getHash())))
                        .and(ADDITIONS_ITEM_CALLOUTS.CALLOUT_TEXT.eq(callout.getText())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());

        Result<Record> result = dslContextProvider.ppc(shard)
                .select(asList(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID, ADDITIONS_ITEM_CALLOUTS.CLIENT_ID,
                        ADDITIONS_ITEM_CALLOUTS.HASH, ADDITIONS_ITEM_CALLOUTS.IS_DELETED))
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(condition)
                .fetch();
        return StreamEx.of(result)
                .toMap(rec -> Pair.of(rec.getValue(ADDITIONS_ITEM_CALLOUTS.CLIENT_ID),
                        rec.getValue(ADDITIONS_ITEM_CALLOUTS.HASH).toBigInteger()),
                        calloutMapper::fromDb,
                        (left, right) -> left.getId() <= right.getId() ? left : right);
    }

    /**
     * Получает уточнения по списку идентификаторов
     *
     * @param shard      шард для запроса
     * @param calloutIds список id уточнений (additions_item_id)
     * @return список уточнений
     */
    public List<Callout> get(int shard, Collection<Long> calloutIds) {
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(calloutIds))
                .orderBy(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID)
                .fetch(calloutMapper::fromDb);
    }

    /**
     * Получает уточнения по критериям {@link CalloutSelection}
     * <p>
     * Уточнения выдаются в порядке увеличения id
     *
     * @param shard       шард для запроса
     * @param clientId    идентификатор клиента, которому должны принадлежать получаемые уточнения
     * @param selection   описание критериев выборки
     * @param limitOffset смещение и кол-во элементов из последовательности подходящих уточнений
     * @return список уточнений
     */
    public List<Callout> get(int shard, ClientId clientId, @Nonnull CalloutSelection selection,
                             LimitOffset limitOffset) {
        SelectConditionStep<Record> step = dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.CLIENT_ID.eq(clientId.asLong()));

        if (selection.hasIds()) {
            step = step.and(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(selection.getIds()));
        }
        if (selection.hasStatuses()) {
            step = step.and(
                    ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE.in(
                            mapList(selection.getStatuses(), CalloutsStatusModerate::toSource)));
        }
        if (selection.hasLastChangeGreaterThan()) {
            step = step.and(
                    ADDITIONS_ITEM_CALLOUTS.LAST_CHANGE.greaterOrEqual(
                            selection.getLastChangeGreaterOrEqualThan()));
        }
        if (selection.hasDeleted()) {
            step = step.and(ADDITIONS_ITEM_CALLOUTS.IS_DELETED.eq(booleanToLong(selection.getDeleted())));
        }

        return step.orderBy(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID)
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .fetch(calloutMapper::fromDb);
    }

    /**
     * Удаляет уточнения по списку id
     *
     * @param shard шард для запроса
     * @param ids   список id уточнений (additions_item_id)
     */
    public void delete(int shard, Collection<Long> ids) {
        dslContextProvider.ppc(shard)
                .deleteFrom(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(ids))
                .execute();
    }

    /**
     * Устанавливает требуемый статус модерации уточнениям по списку id
     *
     * @param shard          шард для запроса
     * @param calloutIds     список id уточнений
     * @param statusModerate требуемый статус модерации
     */
    public void setStatusModerate(int shard, Collection<Long> calloutIds, CalloutsStatusModerate statusModerate) {
        dslContextProvider.ppc(shard)
                .update(ADDITIONS_ITEM_CALLOUTS)
                .set(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE, CalloutsStatusModerate.toSource(statusModerate))
                .set(ADDITIONS_ITEM_CALLOUTS.LAST_CHANGE, LocalDateTime.now())
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.in(calloutIds))
                .execute();
    }

    public AdditionsItemCalloutsStatusmoderate getCalloutsStatusModerate(int shard, Long calloutId) {
        return dslContextProvider.ppc(shard)
                .select(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE)
                .from(ADDITIONS_ITEM_CALLOUTS)
                .where(ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.eq(calloutId))
                .fetchOne(Record1::value1);
    }

    /**
     * Добавляет уточнение в таблицу additions_item_callouts. статус модерации выставляет в Ready
     *
     * @param shard    шард для запроса
     * @param callouts список уточнений
     * @return количество новых записей в таблице
     */
    protected int addToAdditionItemCalloutsTable(int shard, Collection<Callout> callouts) {
        if (callouts.isEmpty()) {
            return 0;
        }
        generateCalloutIds(callouts);
        InsertHelper<AdditionsItemCalloutsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), ADDITIONS_ITEM_CALLOUTS);
        callouts.forEach(callout -> insertHelper.add(calloutMapper, callout)
                .set(ADDITIONS_ITEM_CALLOUTS.STATUS_MODERATE, AdditionsItemCalloutsStatusmoderate.Ready)
                .set(ADDITIONS_ITEM_CALLOUTS.IS_DELETED, 0L)
                .newRecord());

        return insertHelper
                .onDuplicateKeyIgnore()
                .execute();
    }

    /**
     * Генерирует новые id уточнения (additions_item_id) и выставляет эти id каждому уточнению в переданном списке
     *
     * @param callouts список уточнений, которым нужно сгенерировать новые id (additions_item_id)
     */
    private void generateCalloutIds(Collection<Callout> callouts) {
        List<Long> ids = shardHelper.generateAdditionItemIds(callouts.size());
        StreamEx.of(callouts).zipWith(ids.stream())
                .forKeyValue(Callout::setId);
    }
}
